#######################################################################
#  This file is part of GNOWSYS: Gnowledge Networking and
#  Organizing System.
#
#  GNOWSYS is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 3 of
#  the License, or (at your option) any later version.
#
#  GNOWSYS is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public
#  License along with GNOWSYS (gpl.txt); if not, write to the
#  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
#  Boston, MA  02110-1301  USA59 Temple Place, Suite 330,
#
######################################################################
# Contributor: "Dinesh Joshi" <dinesh.joshi@yahoo.com>

class field:
	"""
	Handles the field definitions
	"""
	def __init__( self, dictField ):
		"""
		takes dictionary
		{ 'name', 'constraint', 'type' }
		"""
		
		# TODO: Throw error here for empty name!
		if not dictField.has_key( 'name' ):
			self.name = ""
		else:
			self.name = dictField[ 'name' ]
		
		if not dictField.has_key( 'type' ):
			self.ftype = ""
		else:
			self.ftype = dictField[ 'type' ]
	
	def getName( self ):
		"""
		returns the field's name
		"""
		return self.name
	
	def getType( self ):
		"""
		returns the field's type
		"""
		return self.ftype
	
	def genSQL( self ):
		"""
		returns the SQL string for the field definition
		"""
		strCnstr = ""
		strSQL = "%s %s" % ( self.name, self.ftype )
		
		return strSQL
	
	def show( self ):
		"""
		show the field details, for debugging purposes
		"""
		print "Field details ->"
		print "  -> name        = %s" % self.name
		print "  -> type        = %s" % self.ftype


class table:
	"""
	Handles the table definition
	"""
	def __init__( self, name ):
		"""
		initializes the table and its properties which are:
		1. fields
		2. candidate keys
		3. unique constraints
		4. references
		5. candidate key references
		"""
		self.name = name				# table name
		self.fields = []				# fields [ [ f1, c1], [ f2, c2], .. ] 
		self.ckeys = []					# candidate keys [ [ f1, f2 .. ], [ f1, f2.. ] ]
		
		self.uniquecnstr = []			# list of fields to include in UNIQUE constraint
		# 	eg. [ [ f1, f2 ], f3 ]
		#	will generate UNIQUE( f1, f2 )
		#			  and UNIQUE( f3 )
		#

		self.references = []			# references to other tables [ [ fk, pk, tbl ], [].. ]
		self.ckeyreferences = []		# candidate key references [ 
		#							[ tbl1, [ f1, f2 ... ] ],
		#							[ tbl2, [ f1, f2 ... ] ]
		#						   ]

	def getName( self ):
		"""
		returns the name of the table
		"""
		return self.name
	
	def getField( self, fldName ):
		"""
		returns the field definition of a particular field
		"""
		retval = -1
		
		for f in self.fields:
			if f[0].name == fldName:
				retval = f[0]
				break
		
		return retval
	
	
	def addField( self, fld, constraint ):
		"""
		adds a field to the table
		takes field object as input
		can take constraints as list
		"""
		self.fields.append( [ fld, constraint ] )
	
	def addUniqueConstraint( self, lstFlds ):
                """
		adds a uniqueness constraint on the list of fields
                """
		self.uniquecnstr.append( lstFlds )
		
	def addReference( self, fKey, pKey, refTable ):
		"""
		fKey = key in our table
		pKey = referenced key
		refTable = table being referenced
		"""
		self.references.append( [ fKey, pKey, refTable ] )
	
	def addFieldWithReference( self, refTable, refFld, thisTblCnstr ):
		"""
		In GNOWSYS, the referencing and referenced column
		both have the same name. So, no need to supply
		the field name separately
		
		thisTblCnstr: constraint on the field in THIS table
		"""
		
		#if thisTblFld == -1:
		self.addField( refFld, thisTblCnstr )
		#else:
		#	self.addField( thisTblFld, thisTblCnstr )

		self.addReference( refFld.getName(), refFld.getName(), refTable.getName() )
		
	def addCandidateKey( self, lstFlds ):
		"""
		Adds a candidate key. It produces a constraint of
		the type PRIMARY KEY ( a, b, c )
		"""
		self.ckeys.append( lstFlds )
		
	def addCkeyReference( self, refTable, lstFlds ):
		"""
		Adds a reference to a candidate key in other table.
		It produces a constraint of the type:
		FOREIGN KEY( a, b, c ) REFERENCES refTable ( a, b, c )
		"""
		self.ckeyreferences.append( [ refTable, lstFlds ] )

	def pg_sqlize_value( self, fieldDef, value ):
		# 		self.debug_print( "Fields Def: %s" % fieldDef )
		# 		self.debug_print( "Field val : %s" % value )

		if isinstance( value, list ):
			selValue = "ARRAY%s" % value
		else:
			selValue = "'%s'" % value
		return "%s::%s" % ( selValue, fieldDef[1] )


	def genInsertSQL( self, dictData ):
		"""
		Generates the SQL to insert into table
		Please remember, all foriegn key attributes, etc.. etc.. need to be dealt with
		separately. This will purely generate code to insert 1 row in the table.

		dictData must be in the following format:
		{ 'fldname' : 'fldvalue', .... }

		Logic:
		Excepting primary keys, put data key and values of all fields.
		For pk, insert "DEFAULT" in the SQL query for pk
		"""

		# First convert the field list into a dict { 'fldname':[ fld, cnstr ] }
		dictFlds = {}
		for f in self.fields:
			dictFlds[ f[0].getName() ] = f

		tblName = self.getName()
		fldNames = ", " . join( [ "%s" % f[0].getName() for f in self.fields ] )

		fldVals = ""
		lstFldVals = []
		for f in self.fields:
			if f[0].getType() == 'serial8' or f[0].getType() == 'bigserial':
				#				val = dictData[ f[0].getName() ]
				val = "DEFAULT"
			else:
				val = self.pg_sqlize_value( ['', f[0].getType() ], dictData[ f[0].getName() ] )
			lstFldVals.append( val )

		print lstFldVals
		fldVals = ", " . join( [ "%s" % v for v in lstFldVals ] )
		
		strSQL = "INSERT INTO %s ( %s ) VALUES ( %s );" % ( tblName, fldNames, fldVals )
		return strSQL
	
	
	def genSQL( self ):
		"""
		Logic:
		1. Generate SQL for the following:
		a. All fields
		b. Candidate keys
		c. Simple references
		d. Candidate key references
		2. Concatenate a, b, c, d
		3. Wrap inside the 'CREATE TABLE' SQL
		"""
		
		# Fields
		SQLflds = ",\n\t" . join( [ "%s %s" % ( f[0].genSQL(), f[1] ) for f in self.fields ] )
		
		
		# Candidate Keys
		strTmp = ""
		SQLCkeys = ""
		if len( self.ckeys ) > 0:
			for key in self.ckeys:
				k = ", " . join( [ "%s" % f.getName() for f in key ] )
				strTmp = strTmp + "PRIMARY KEY( %s ) " % k
			SQLCkeys = ",\n\t" + strTmp
		
		
		# Simple References
		strTmp = ""
		SQLSRefs = ""
		c=0
		
		if len( self.references ) > 0:
			SQLSRefs = ", "
			for ref in self.references:
				c = c + 1
				fk  = ref[0]
				pk  = ref[1]
				tbl = ref[2]
				strTmp = strTmp + "\n\tFOREIGN KEY( %s ) REFERENCES %s ( %s )" % ( fk, tbl, pk )
				if c < len( self.references ):
					strTmp = strTmp + ', '
					
		SQLSRefs = SQLSRefs + strTmp
		
		# Candidate key references
		SQLCKeyRefs = ""
		strTmp = ""
		c=0
		if len( self.ckeyreferences ) > 0:
			SQLCKeyRefs = ", "
			for myCkey in self.ckeyreferences:
				c = c + 1
				tblName = myCkey[0].getName()
				strFlds = ", " . join( [ "%s" % f.getName() for f in myCkey[1] ] )
				strTmp = strTmp + "\n\tFOREIGN KEY( %s ) REFRENCES %s ( %s )" % ( strFlds, tblName, strFlds )
				if c < len( self.ckeyreferences ):
					strTmp = strTmp + ', '
		
		SQLCKeyRefs = SQLCKeyRefs + strTmp

		# Unique constraints
		strTmp = ""
		SQLUCnstr = ""
		if len( self.uniquecnstr ) > 0:
			#print self.uniquecnstr
			for uc in self.uniquecnstr:
				k = ", " . join( [ "%s" % f.getName() for f in uc ] )
				strTmp = strTmp + "UNIQUE ( %s ) " % k
			
			SQLUCnstr = ",\n\t" + strTmp
		
		SQLComplete = "CREATE TABLE %s ( \n\t%s %s %s %s %s \n );" % ( self.getName(), SQLflds, SQLCkeys, SQLSRefs, SQLCKeyRefs, SQLUCnstr )
		return SQLComplete
	
	def findRef( self, fldName ):
                """
		given a field name, returns the references
                """
		for r in self.references:
			if r[0] == fldName:
				return r
			
		return 0
	
	def findUniqueKeys( self, fldName ):
		"""
		Locates if field has unique constraint on it
		if so, returns the field or list of fields with
		which it is associated to be unique.
		"""
		for lf in self.uniquecnstr:
			if isinstance( lf, list ):
				for f in lf:
					if f.getName() == fldName:
						return lf
			else:
				if lf.getName() == fldName:
					return f
			
		return 0
	
	def genHTML( self ):
		"""
		Depicts the table in HTML
		"""
		htmlTD = "\n\t<td>%s</td>"
		htmlTR = "\n\t<tr>%s</tr>"
		htmlTable = "<table border=1 style='margin-bottom:20px; width:800px;'><caption style='background-color:#AAAAAA;'><font size=5pt>" + self.getName() + "</font></caption><tr><th>Field name</th><th>Field Type</th><th>Constraint</th></tr>%s</table>"
		
		strRow = ""
		for f in self.fields:
			tdName = htmlTD % f[0].getName()
			tdType = htmlTD % f[0].getType()
			
			if f[1] != '':
				tdCnst = htmlTD % f[1]
			else:
				r = self.findRef( f[0].getName() )
				if r != 0:
					tdCnst = htmlTD % ( 'REFERENCES %s ( %s )' % ( r[2], r[1] ) )
				else:
					uc = self.findUniqueKeys( f[0].getName() )
					if uc != 0:
						k = ", " . join( [ "%s" % f.getName() for f in uc ] )
						tdCnst = htmlTD % ( "UNIQUE ( %s ) " % k )
					else:
						tdCnst = htmlTD % '&nbsp;'
			
			strRow = strRow + ( htmlTR % ( tdName + tdType + tdCnst ) )
		
		return( htmlTable % strRow )
	
	
	def show( self ):
                """
		show the definition of the entire table
                """
		print "Table name : %s" % self.name
		print "Fields :"
		for f in self.fields:
			print "Name       : " + f[0].getName()
			print "Type       : " + f[0].getType()
			print "Constraint : " + f[1]
			print
			
		print "References :"
		for r in self.references:
			print "Foreign key field    : " + r[0]
			print "Referenced key field : " + r[1]
			print "Referenced Table     : " + r[2]
			print
	
		print "Candidate keys :"
		for r in self.ckeys:
			for f in r:
				print f.getName() + "," ,
	
		print "\n"
		print "Candidate key references :"
		for r in self.ckeyreferences:
			print "referred table : " + r[0].getName()
			for f in r[1]:
				print f.getName() + "," ,
	
		print "\n"
		print "Unique keys :"
		for uc in self.uniquecnstr:
			print "Unique constraint on field: %s" % ( ", " . join( [ "%s" % f.getName() for f in uc ] ) )


# gnowsys wrapper over the vanilla table
class gnowsysTable:
        """
        """
	def __init__( self, tblname ):
                """
		initializes the gnowsysTable class it has the following properties:
		1. table name
		2. a vanilla table
		3. list of field tables
		4. list of reference fields
                """
		self.name = tblname
		self.vTable = table( tblname )		# vanilla table
		self.lstFieldTables = []
		self.lstRefFields = []
		self.fid_type = 'int8'
		
	def getField( self, fldName ):
                """
		given a field name, returns the definition
                """
		return self.vTable.getField( fldName )
	
	def addUniqueConstraint( self, _lstFlds ):
		"""
		Takes in the names, turns it into actual field objects
		using getField() method and creates the list of fields
		which is then passed to vTable's addUniqueConstraint() function
		"""
		tmpLst = []
		lstFlds =[]
		for f in _lstFlds:
			fld = self.vTable.getField( f[0] )
			#print fld.getName()
			if( fld != -1 ):
				tmpLst.append( fld )

		self.vTable.addUniqueConstraint( tmpLst )

	def addFields( self, lstFlds ):
		"""
		- lstFlds is a list of fields
		[ [ 'name', 'type', 'constraint' ], [], ... ]
		"""

		# define the fields
		for f in lstFlds:
			fld = field( { 'name':f[0], 'type':f[1] } )
			self.vTable.addField( fld, f[2] )
	
	def addFieldTables( self, lstFlds ):
		"""
		Adds fields ( ones with field tables )
		- Names field tables as tablename_fieldname
		- will create field tables with structure
		CREATE tablename_fieldname (
		fieldname_fid int8,
		fieldname_value type
		);
		- lstFlds is a list of fields
		[ [ 'name', 'type' ], [], ... ]
		"""
		
		# bad hack!
		fid_type = 'serial8'
		
		cnt=0
		for t in lstFlds:
			fname = t[0]	# field name
			ftype = t[1]	# field type
			
			tmpTblName = "%s_%s" % ( self.name, fname )		# construct table name
			tmpTbl = table( tmpTblName )					# init the table object
			
			# configs for field 1
			tmpFldName1 = "%s_fid" % ( fname )
			tmpFldType1 = fid_type
			
			# configs for field 2
			tmpFldName2 = "%s_value" % ( fname )
			tmpFldType2 = ftype
			
			# define the fields
			fld1 = field( {'name':tmpFldName1, 'type':tmpFldType1 } )
			fld2 = field( {'name':tmpFldName2, 'type':tmpFldType2 } )
			
			# add fields to the field table
			tmpTbl.addField( fld1, 'PRIMARY KEY' )
			tmpTbl.addField( fld2, 'UNIQUE' ) # The value field in the field table must be unique
			
			# insert the field table in our list
			self.lstFieldTables.append( tmpTbl )
			
			# hack!
			fld1 = field( {'name':tmpFldName1, 'type':self.fid_type } )
			
			# add reference to the field table in the main table
			self.vTable.addFieldWithReference( tmpTbl, fld1, '' )
	
	def addRefFields( self, refs ):
		"""
		Basically adds a foreign key field
		refs format:
		[ [ t1, f1 ], [ t2, f2]... ]
		
		t1: table
		f1: field name
		
		the field added to this table will have the same name
		as that of the field being referred.
		"""
		for r in refs:
			refTbl = r[0]						# referred table
			refFld = refTbl.getField( r[1][0] )	# referred field
			
			tmpFld = field( {'name':refFld.getName(), 'type':self.fid_type} )
			self.vTable.addFieldWithReference( refTbl, tmpFld, '' )

	def addFieldWithReference( self, ref ):
		"""
		Adds ONE foreign key field
		ref format:
		[ tbl, fld, constraint ]
		
		tbl: gnowsys table object
		fld: field object
		constraint: constraint on the field in ITS table
		
		the field added to this table will have the same name
		as that of the field being referred.
		"""
		refTbl = ref[0]				# referred table
		refFld = ref[1]				# referred field
		thisTblCnstr = ref[2]
		
		tmpFld = field( {'name':refFld.getName(), 'type':self.fid_type} )

		self.vTable.addFieldWithReference( refTbl, tmpFld, thisTblCnstr )

	def genHTML( self ):
                """
		depicts the table in plain HTML
                """
		strHTML = ""
		
		for ft in self.lstFieldTables:
			strHTML = strHTML + ft.genHTML()
		
		return( strHTML + self.vTable.genHTML() )
	
	def genSQL( self ):
		"""
		Generates SQL code of the current table configuration
		"""
		tblSQL = self.vTable.genSQL()
		
		fldTblSQL = ""
		if len( self.lstFieldTables ) > 0:
			fldTblSQL = "\n\n" . join( [ "%s" % t.genSQL() for t in self.lstFieldTables ] )
		
		return fldTblSQL + "\n\n" + tblSQL
	
	def getName( self ):
                """
		returns the table's name
                """
		return self.name
	
	def show( self ):
                """
		shows the definition of GNOWSYS table
                """
		print "GNOWSYS Table Name : %s" % self.name
		
		print "Main table's SQL is :"
		print self.vTable.genSQL()
		
		print
		print "I have the following properties"
		self.vTable.show()

		print
		print "I have the following field tables: "
		print
		for t in self.lstFieldTables:
			t.show()



