// -*- c++ -*-

//  Gnomoradio - rainbow/hub-client.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __RAINBOW_HUB_CLIENT_H
#define __RAINBOW_HUB_CLIENT_H

#include "rainbow/alarm.h"
#include "rainbow/http-client.h"
#include "rainbow/http-server.h"
#include "rainbow/license.h"
#include "rainbow/rdf-resource.h"
#include <libxml++/libxml++.h>
#include <glibmm.h>
#include <map>
#include <memory>

namespace Rainbow
{
	class Resource : public SigC::Object
	{
	public:
		Resource (const Glib::ustring &url_,
			  const Glib::ustring &filename_ = Glib::ustring())
			: url(url_),
			  filename(filename_),
			  download_percent(filename.size() ? 100 : 0),
			  obtained_info(filename.size() != 0),
			  prepared(filename.size() != 0),
			  prepare_once_info_is_obtained(false),
			  available_locations_cursor(0),
			  downloading(false),
			  sharable(false),
			  allocated_size(0),
			  refcnt(1)
			{ rdf_description_exists = (url.find('#') != Glib::ustring::npos); }
		~Resource () { /* if (fetching) dl->cancel(); */ }

		void ref () { ++refcnt; }
		void unref () { if (--refcnt == 0) delete this; }

		Glib::ustring get_filename () const { return filename; }
		Glib::ustring get_url () const { return url; }
		size_t get_allocated_size () const { return allocated_size; }

		int download_status () const { return download_percent; }
		bool downloaded () const { return prepared; }

		bool has_rdf () { return rdf_description_exists; }

		friend class HubClient;
		friend class Checksum;

		SigC::Signal1<void,bool> signal_found_info;
		SigC::Signal1<void,unsigned int> signal_downloading;
		SigC::Signal1<void,bool> signal_download_done;
		SigC::Signal0<void> signal_file_deleted;

	private:
		int refcnt;

		Glib::ustring filename, url;
		bool sharable;
		std::vector<guint8> checksum;
		int download_percent;
		bool prepare_once_info_is_obtained, prepared, obtained_info;
		std::vector<Glib::ustring> available_locations, mirrors;
		int available_locations_cursor;
		size_t allocated_size;

		std::auto_ptr<HttpClient> dl;
		bool downloading;

		bool rdf_description_exists;
	};

	class HubClient : public SigC::Object
	{
	public:
		HubClient (const Glib::ustring &hub);
		~HubClient ();

		ref_ptr<Resource> find (const Glib::ustring &url);
		ref_ptr<Resource> create (const Glib::ustring &url);
		bool get_filename_threadsafe (const Glib::ustring &url,
					      Glib::ustring &filename,
					      bool only_if_sharable=false);
		size_t get_size_threadsafe (const Glib::ustring &url);

		void prepare_resource (const ref_ptr<Resource> &r)
			{ prepare_resource(&*r); }
		void uncache_resource (const ref_ptr<Resource> &r)
			{ uncache_resource(&*r); }

		void set_hub (const Glib::ustring &hub);
		Glib::ustring get_hub () const { return hub_host; }

		void set_cache_size (unsigned int megabytes);
		unsigned int get_cache_size () const { return cache_size; }

		SigC::Signal1<void,size_t> signal_must_uncache; // in kilobytes

	private:
		void prepare_resource (Resource *r);
		void query_hub (Resource *r);
		void start_download (Resource *r);

		void load_database ();
		void save_database ();

		void connect ();
		void disconnect ();
		void on_connect (int socket);

		void on_save_alarm ();

		void rdf_file_ready_callback (bool success, Resource *r);
		void license_obtained_callback (Rainbow::ref_ptr<Rainbow::License> license, Resource *r);
		void file_download_done_callback (bool success, Resource *r);
		void file_download_percent_callback (unsigned int percent, Resource *r);
		void parse_resource (xmlpp::Element *xml, ref_ptr<RdfResource> rdf, Resource *r);
		void verify_checksum_callback (bool success, Resource *r);
		void download_success (Resource *r);

		void check_allocated_size (Resource *r);
		void set_allocated_size (Resource *r, size_t s);
		void uncache_resource (Resource *r);
		void review_cache_size ();

		static Glib::ustring random_filename ();

		HttpServer server;
		Alarm alarm; // to check the connection to the hub
		Glib::ustring hub_host;
		int hubsock;
		std::map<Glib::ustring,ref_ptr<Resource> > db;
		Glib::Mutex db_mutex;

		size_t total_allocated_size; // in kilobytes
		unsigned int cache_size; // in megabytes

		Alarm save_alarm;
		static const unsigned int save_alarm_interval = 600;

		static const unsigned short http_server_port = 4617;
		static const unsigned int reconnect_interval = 180;
	};
}

#endif
