// -*- c++ -*-

//  Gnomoradio - rainbow/http-client.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __RAINBOW_HTTP_CLIENT_H
#define __RAINBOW_HTTP_CLIENT_H

#include <string>
#include <map>

#include <glibmm.h>
#include <sigc++/sigc++.h>
#include <unistd.h>

#include "rainbow/util.h"

namespace Rainbow
{
	class HttpClient : public SigC::Object
	{
	public:
		HttpClient (const Glib::ustring &hostname,
			    unsigned short port = 80,
			    bool keep_alive = false);
		~HttpClient ();

		bool connected () const { return net_sock > 0; }
		void head (const Glib::ustring &remote_file);
		void get (const Glib::ustring &remote_file,
			  const Glib::ustring &local_file = Glib::ustring(),
			  unsigned int offset = 0,
			  unsigned int length = 0);
		void post (const Glib::ustring &remote_file,
			   const Glib::ustring &post_data,
			   const Glib::ustring &local_file = Glib::ustring(),
			   unsigned int offset = 0,
			   unsigned int length = 0);
		void cancel ();

		const std::string &get_buffer () const { return buffer; }
		const std::string &get_first_response_line () const { return first_line; }
		const std::string get_header (const std::string &key)
			{
				std::map<std::string,std::string>::iterator p;
				p = header.find(key);
				return (p == header.end()) ? std::string() : p->second;
			}

		SigC::Signal1<void,bool> signal_request_done;
		SigC::Signal1<void,unsigned int> signal_download_percent;

		static bool parse_url(const Glib::ustring &url,
				      Glib::ustring &host,
				      unsigned short &port,
				      Glib::ustring &file);
		static std::string url_encode (const char *original);

	private:
		void request_thread ();
		void send_signal_done ();
		void send_signal_percent ();

		// variables from last request
		std::string first_line, buffer;
		std::map<std::string,std::string> header;

		int net_sock, local_sock;
		Glib::ustring requested_host;
		bool keep_alive;
		struct sockaddr_in server;
		Glib::Dispatcher my_signal_done, my_signal_percent;
		Glib::Thread *thread;
		Glib::Mutex percent_mutex;

		// private variables for get
		bool success, head_only;
		Glib::ustring local_file, remote_file;
		Glib::ustring post_data;
		unsigned int offset, length;
		unsigned int percent;
	};
}

#endif
