//  Gnomoradio - gnomoradio/main-window.cc
//  Copyright (C) 2003-2004  Jim Garrison, Matt Gerginski
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "main-window.h"
#include "song-info.h"
#include "prefs-box.h"
#include "playlist-settings.h"
#include "playlist-druid.h"
#include "browser-link.h"
#include "song-info.h"
#include "conf.h"
#include <iostream>
#include <sstream>
#include "config.h"

using namespace std;
using namespace SigC;
using namespace Glib;
using namespace Gtk;
using namespace Wraporadio;
using namespace Gnomoradio;

TargetEntry Gnomoradio::MainWindow::song_target("GnomoradioSong");

Gnomoradio::MainWindow::MainWindow (Init *init, PlaylistChooserStore &chooser_store)
	: chooser(chooser_store.store),
	  store(&chooser_store),
	  list(init, chooser_store.store->children().begin()->get_value(chooser_store.song_list_col)),
	  controls(init, &list, this),
	  wr(init),
	  file_findcurrentsong(_("Find Current Song")),
	  edit_songinfo(_("Song Info...")),
	  playlist_settings(_("Settings...")),
	  playlist_remove(_("Remove...")),
	  playlist_removesong(_("Remove Song")),
	  shuffle_button(_("Shuffle")),
	  repeat_button(_("Repeat")),
	  search_online(_("Online Search")),
	  add_uri_box(0),
	  about_box(0)
{
	int width, height;
	width = Conf::get_int("main_window_width");
	height = Conf::get_int("main_window_height");
	if (width == 0)
		width = 600;
	if (height == 0)
		height = 400;
	set_default_size(width, height);

	try {
		set_icon_from_file(PIXMAPSDIR "/gnomoradio.png");
	} catch (...) {
	}

	// playlist chooser width
	int playlist_chooser_width = Conf::get_int("playlist_chooser_width");
	if (playlist_chooser_width)
		hpaned.set_position(playlist_chooser_width);

	// column widths
	std::stringstream ss;
	std::list<TreeViewColumn*> cols = list.get_tree_view()->get_columns();
	int col_num = 0;

	for (std::list<TreeViewColumn*>::iterator i = cols.begin(); i != cols.end(); ++i) {
		ss << "col_width_" << col_num;
		int col_width = Conf::get_int(ss.str());
	
		if (col_width <= 0)
			col_width = 200;
		
		(*i)->set_fixed_width(col_width);
		
		col_num++;
		ss.str("");
	}

	set_title("Gnomoradio");

	add(vbox);

	/* Menus */
	MenuBar *menu_bar = manage(new MenuBar);

	/* File  */
	MenuItem *file          = manage(new MenuItem(_("File")));
	Menu *file_menu         = manage(new Menu);
	MenuItem *file_add_uri  = manage(new MenuItem(_("Add URI...")));
	MenuItem *file_exit     = manage(new MenuItem(_("Exit")));
	
	file_menu->append(*file_add_uri);
	file_menu->append(*manage(new SeparatorMenuItem));
	file_menu->append(file_findcurrentsong);
	file_menu->append(*manage(new SeparatorMenuItem));
	file_menu->append(*file_exit);
	file->set_submenu(*file_menu);

	file_add_uri->signal_activate().connect(slot(*this, &MainWindow::on_file_add_uri_activate));
	file_findcurrentsong.signal_activate().connect(slot(*this, &MainWindow::find_current_song));
	file_findcurrentsong.set_sensitive(false);
	file_exit->signal_activate().connect(slot(*this, &MainWindow::on_file_exit_activate));

	/* Playlist */
	MenuItem *playlist        = manage(new MenuItem(_("Playlist")));
	Menu *playlist_menu       = manage(new Menu);

	Gtk::MenuItem *new_list = manage(new MenuItem(_("New...")));
	new_list->signal_activate().connect(slot(*this, &MainWindow::create_playlist));

	playlist_menu->append(*new_list);
	playlist_menu->append(playlist_settings);
	playlist_menu->append(playlist_removesong);
	playlist_menu->append(playlist_remove);
	playlist->set_submenu(*playlist_menu);

	playlist_settings.signal_activate().connect(slot(*this, &MainWindow::on_playlist_settings_activate));
	playlist_settings.set_sensitive(false);

	playlist_remove.signal_activate().connect(slot(*this, &MainWindow::on_playlist_remove_activate));
	playlist_remove.set_sensitive(false);

	playlist_removesong.signal_activate().connect(slot(*this, &MainWindow::on_playlist_removesong_activate));
	playlist_removesong.set_sensitive(false);

	/* Edit  */
	MenuItem *edit             = manage(new MenuItem(_("Edit")));
	Menu *edit_menu            = manage(new Menu);
	MenuItem *edit_preferences = manage(new MenuItem(_("Preferences...")));
	SeparatorMenuItem *sep2    = manage(new SeparatorMenuItem());

	edit_menu->append(edit_songinfo);
	edit_menu->append(*sep2);
	edit_menu->append(*edit_preferences);
	edit->set_submenu(*edit_menu);

	edit_songinfo.set_sensitive(false);
	edit_songinfo.signal_activate().connect(slot(*this, &MainWindow::on_edit_song_info_activate));

	edit_preferences->signal_activate().connect(slot(*this, &MainWindow::on_edit_preferences_activate));

	/* Help  */
	MenuItem *help       = manage(new MenuItem(_("Help")));
	Menu *help_menu      = manage(new Menu);
	MenuItem *help_about = manage(new MenuItem(_("About...")));

	help_menu->append(*help_about);
	help->set_submenu(*help_menu);

	help_about->signal_activate().connect(slot(*this, &MainWindow::on_help_about_activate));

	/* pack menus */
	menu_bar->append(*file);
	menu_bar->append(*edit);
	menu_bar->append(*playlist);
	menu_bar->append(*help);

	vbox.pack_start(*menu_bar, PACK_SHRINK);
	vbox.pack_start(top, PACK_SHRINK, 8);
	vbox.pack_start(hpaned, PACK_EXPAND_WIDGET);
	
	scrolled_window.add(chooser);
	scrolled_window.set_policy(POLICY_AUTOMATIC, POLICY_AUTOMATIC);
	scrolled_window.set_shadow_type(SHADOW_ETCHED_IN);
	left.pack_start(scrolled_window);
	left.pack_start(list_controls, PACK_SHRINK);

	shuffle_button.signal_clicked().connect(slot(*this, &MainWindow::on_shuffle_clicked));
	repeat_button.signal_clicked().connect(slot(*this, &MainWindow::on_repeat_clicked));
	list_controls.set_homogeneous(true);
	list_controls.pack_start(shuffle_button, PACK_EXPAND_WIDGET);
	list_controls.pack_start(repeat_button, PACK_EXPAND_WIDGET);

	list.get_tree_view()->get_selection()->signal_changed().connect(slot(*this, &MainWindow::set_edit_song_info_sensitivity));

	list.signal_different_list.connect(slot(*this, &MainWindow::different_list_viewed));
	different_list_viewed(list.current_list());

	Label *search_label = manage(new Label(_("Search: ")));

	search.signal_changed().connect(slot(*this, &MainWindow::search_changed));
	search.signal_activate().connect(slot(*this, &MainWindow::search_activate));
	search_online.signal_clicked().connect(slot(*this, &MainWindow::search_activate));
	search_online.set_sensitive(false);

	Gtk::HBox *search_hbox = manage(new HBox(false, 2));
	search_hbox->pack_start(*search_label, PACK_SHRINK);
	search_hbox->pack_start(search, PACK_SHRINK);
	search_hbox->pack_start(search_online, PACK_SHRINK);

	top.pack_start(controls);
	top.pack_start(*search_hbox, PACK_SHRINK, 8);

	hpaned.pack1(left, FILL);
	hpaned.pack2(list, EXPAND);
	hpaned.set_border_width(8);

	chooser.set_headers_visible(false);
	chooser.append_column(_("Icon"), chooser_store.icon_col);
	chooser.append_column_editable(_("List"), chooser_store.name_col);

	chooser.signal_button_press_event().connect(slot(*this, &MainWindow::on_chooser_clicked), false);

	Gtk::CellRendererText *renderer = static_cast<Gtk::CellRendererText *>(chooser.get_column_cell_renderer(1));

	renderer->signal_edited().connect(slot(*this, &MainWindow::on_playlist_rename));

	chooser.get_selection()->signal_changed().
		connect(slot(*this, &MainWindow::on_chooser_change));
	std::vector<TargetEntry> t;
	t.push_back(Gnomoradio::MainWindow::song_target);
	chooser.enable_model_drag_dest(t, Gdk::ACTION_COPY);
	chooser.signal_drag_data_received().connect(slot(*this, &MainWindow::on_chooser_drag_data_received));

	wr->player()->signal_state_changed.connect(slot(*this, &MainWindow::on_state_changed));
}

Gnomoradio::MainWindow::~MainWindow ()
{
	int w, h;
	get_size(w, h);

	Conf::set("main_window_width", w);
	Conf::set("main_window_height", h);

	std::stringstream ss;
	std::list<TreeViewColumn*> cols = list.get_tree_view()->get_columns();
	int col_num = 0;

	for (std::list<TreeViewColumn*>::iterator i = cols.begin(); i != cols.end(); ++i) {
		ss << "col_width_" << col_num;
		Conf::set(ss.str(), (*i)->get_width());
		col_num++;
		ss.str("");
	}

	Conf::set("playlist_chooser_width",
		  hpaned.get_position());
}

void Gnomoradio::MainWindow::different_list_viewed (SongList sl)
{
	shuffle_button.set_sensitive(list.base_list().get_shufflable());
	shuffle_button.set_active(list.base_list().get_shuffle());

	repeat_button.set_sensitive(sl.get_repeatable());
	repeat_button.set_active(sl.get_repeat());
}

bool Gnomoradio::MainWindow::on_chooser_clicked (GdkEventButton *event)
{
	if (event->type == GDK_BUTTON_PRESS
	    && event->button == 3) { // right mouse button
		int cell_x, cell_y;
		Gtk::TreeModel::Path path;
		Gtk::TreeViewColumn *column;
		chooser.get_path_at_pos(int(event->x), int(event->y), path, column, cell_x, cell_y);
		if (!path.gobj() || path.empty())
			return true;

		static auto_ptr<Gtk::Menu> menu;
		menu.reset(new Gtk::Menu);

		chooser.get_selection()->unselect_all();
		chooser.get_selection()->select(path, path);
		Wraporadio::SongList songlist = store->store->get_iter(path)->get_value(store->song_list_col);

		Gtk::MenuItem *settings = manage(new MenuItem(_("Settings...")));
		Gtk::MenuItem *remove = manage(new MenuItem(_("Remove...")));

		settings->signal_activate().connect(bind(slot(*this, &MainWindow::on_right_click_songlist_settings_activate), songlist));
		menu->append(*settings);

		remove->signal_activate().connect(bind(slot(*this, &MainWindow::remove_playlist), songlist));
		menu->append(*remove);

		remove->set_sensitive(songlist.destroyable());

		menu->show_all();
		menu->popup(event->button, event->time);

		return true;
	}

	return false; // I have no idea what this does either
}

void Gnomoradio::MainWindow::search_changed ()
{
	if (search.get_text().size() == 0) {
		if (chooser.get_selection()->get_selected())
			list.set(chooser.get_selection()->get_selected()->get_value(store->song_list_col));
		else
			list.set(store->store->children().begin()->get_value(store->song_list_col));

		playlist_settings.set_sensitive(true);
		playlist_remove.set_sensitive(list.current_list().destroyable());
		search_online.set_sensitive(false);
		return;
	}

	// separate out spaces
	ustring search_str(search.get_text());
	vector<ustring::size_type> spaces;
	// fixme: this may be suboptimal
	for (ustring::size_type i = 0; i < search_str.size(); ++i)
		if (search_str[i] == ' ')
			spaces.push_back(i);

	xmlpp::Document query;
	if (spaces.size() == 0)
		query.create_root_node("info")->set_child_text(search_str);
	else {
		xmlpp::Element *root = query.create_root_node("boolean");
		root->set_attribute("type", "and");
		ustring::size_type b = 0;
		for (vector<ustring::size_type>::iterator i = spaces.begin(); i != spaces.end(); ++i) {
			if (b != *i)
				root->add_child("info")->set_child_text(search_str.substr(b, *i - b));
			b = *i + 1;
		}
		if (b != search_str.size())
			root->add_child("info")->set_child_text(search_str.substr(b));
	}

	if (&*search_list)
		search_list->set_criteria(query.write_to_string());
	else
		search_list.reset(new SongListSearch(query.write_to_string()));

	list.set(*search_list);
	playlist_settings.set_sensitive(false);
	playlist_remove.set_sensitive(false);
	search_online.set_sensitive(true);
}

void Gnomoradio::MainWindow::search_activate ()
{
	if (&*search_list)
		search_list->search_online(search.get_text());
	search_online.set_sensitive(false);
}

void Gnomoradio::MainWindow::create_playlist ()
{
	(new PlaylistDruid())->show_all();
}

void Gnomoradio::MainWindow::remove_playlist (Wraporadio::SongList songlist)
{
	Glib::ustring playlist_title = songlist.get_name();
	
	Dialog dialog(_("Remove Playlist?"), true, true);
	
	dialog.set_resizable(false);
	dialog.set_border_width(5);
	dialog.add_button(_("Remove"), 1);
	dialog.add_button(_("Cancel"), 0);
	
	Label *question = manage(new Label(_("Really remove playlist: ") + playlist_title + _("?")));
	dialog.get_vbox()->pack_start(*question, PACK_SHRINK, 8);
	question->show();
	
	int response_yes = dialog.run();
	
	if (response_yes) {
		list.set(store->store->children().begin()->get_value(store->song_list_col));
		playlist_settings.set_sensitive(true);
		playlist_remove.set_sensitive(list.current_list().destroyable());
		songlist.destroy();
	}
}

void Gnomoradio::MainWindow::on_shuffle_clicked ()
{
	list.base_list().set_shuffle(shuffle_button.get_active());
	list.set(list.base_list());
	playlist_settings.set_sensitive(true);
	playlist_remove.set_sensitive(list.current_list().destroyable());
}

void Gnomoradio::MainWindow::on_repeat_clicked ()
{
	list.current_list().set_repeat(repeat_button.get_active());
}

void Gnomoradio::MainWindow::on_right_click_songlist_settings_activate (Wraporadio::SongList songlist)
{
	PlaylistSettingsBox *box = new PlaylistSettingsBox(songlist);

	box->show_all();
}

void Gnomoradio::MainWindow::on_playlist_rename (const Glib::ustring& path, const Glib::ustring& new_text)
{
	if (chooser.get_selection()->get_selected())
		chooser.get_selection()->get_selected()->get_value(store->song_list_col).set_name(new_text);
}

void Gnomoradio::MainWindow::on_chooser_change ()
{
	edit_songinfo.set_sensitive(false);
	playlist_removesong.set_sensitive(false);
	if (chooser.get_selection()->get_selected()) {
		list.set(chooser.get_selection()->get_selected()->get_value(store->song_list_col));
		playlist_remove.set_sensitive(list.current_list().destroyable());
		playlist_settings.set_sensitive(true);
	} else {
		playlist_remove.set_sensitive(false);
	}

	if (search.get_text().size() != 0)
		search.set_text("");
}

void Gnomoradio::MainWindow::set_edit_song_info_sensitivity ()
{
	int num_selected_rows = list.get_tree_view()->get_selection()->count_selected_rows();

	if (num_selected_rows) {
		edit_songinfo.set_sensitive();
		if (list.current_list().get_mutable())
			playlist_removesong.set_sensitive();
	} else {
		edit_songinfo.set_sensitive(false);
		playlist_removesong.set_sensitive(false);
	}
}

void Gnomoradio::MainWindow::on_file_add_uri_activate ()
{
	if (add_uri_box)
		add_uri_box->present();
	else {
		add_uri_box = new Window();
		add_uri_box->set_title(_("Add URI"));
		add_uri_box->set_border_width(8);
		add_uri_box->set_resizable(false);

		Label *URI = manage(new Label(_("URI: ")));
		Button *activate_button = manage(new Button(_("Add")));
		Button *close_button = manage(new Button(_("Close")));
		HBox *hbox = manage(new HBox(false, 8));
		HBox *hbox2 = manage(new HBox(false, 8));
		VBox *vbox = manage(new VBox(false, 8));

		uri_entry.signal_activate().connect(slot(*this, &MainWindow::on_uri_entry_activate));
		activate_button->signal_clicked().connect(slot(*this, &MainWindow::on_uri_entry_activate));
		close_button->signal_clicked().connect(slot(*this, &MainWindow::on_uri_close_activate));

		hbox->pack_start(*URI, PACK_SHRINK);
		hbox->pack_start(uri_entry, PACK_SHRINK);

		hbox2->pack_start(*activate_button);
		hbox2->pack_start(*close_button);

		vbox->pack_start(*hbox);
		vbox->pack_start(*hbox2);
		add_uri_box->add(*vbox);

		add_uri_box->show_all();
	}
}

void Gnomoradio::MainWindow::on_file_exit_activate ()
{
	hide();
}

void Gnomoradio::MainWindow::on_edit_song_info_activate ()
{
	Wraporadio::Song song = *list.get_selected_song();
	(new SongInfoBox(song))->show_all();
}

void Gnomoradio::MainWindow::on_edit_preferences_activate ()
{
	static Window *prefs;
	static bool prefs_open = false;

	if (prefs_open) {
		prefs->present();
		prefs_open = true;
	} else {
		prefs = new PrefsBox;
		prefs_open = false;
        }
}

void Gnomoradio::MainWindow::on_playlist_settings_activate ()
{
	Wraporadio::SongList selected = chooser.get_selection()->get_selected()->get_value(store->song_list_col);

	PlaylistSettingsBox *box = new PlaylistSettingsBox(selected);

	box->show_all();
}

void Gnomoradio::MainWindow::on_playlist_removesong_activate ()
{
	list.current_list().remove(list.get_selected_song());
}

void Gnomoradio::MainWindow::on_playlist_remove_activate ()
{
	Wraporadio::SongList r_list(chooser.get_selection()->get_selected()->get_value(store->song_list_col));

	remove_playlist(r_list);
}

void Gnomoradio::MainWindow::on_help_about_activate ()
{
	if (about_box)
		about_box->present();
	else {
		about_box = new Window();
		about_box->set_title(_("Gnomoradio " VERSION ": About"));
		about_box->set_border_width(8);
		about_box->set_resizable(false);

		RefPtr<Gdk::Pixbuf> logo;

		try {
			logo = Gdk::Pixbuf::create_from_file(PIXMAPSDIR "/gnomoradio/gnomoradio-heart.png");
		} catch (...) {
		}

		Image *logo_image = manage(new Image(logo));

		Table *table = manage(new Table(7, 2, false));
		about_box->add(*table);

		Label *title = manage(new Label("Gnomoradio " VERSION));
		Label *jim  = manage(new Label("Jim Garrison <garrison@gnomoradio.org>"));
		Label *matt  = manage(new Label("Matt Gerginski <mattgerg@gnomoradio.org>"));

		Label *description = manage(new Label(_("Gnomoradio is a music playing robot with a mind of its own")));

		Button *web_link   = manage(new Button("http://gnomoradio.org/"));
		web_link->signal_clicked().connect(bind(slot(&BrowserLink::open_url), "http://gnomoradio.org/"));

		Label *copyright   = manage(new Label(_("Copyright (c) 2003-2004 under the terms of the")));

		Label *gpl         = manage(new Label(_("GNU General Public License")));

		Button *close = manage(new Button(_("Close")));

		close->signal_clicked().connect(slot(*this, &MainWindow::on_about_box_close));

		VBox *license_vbox = manage(new VBox());

		license_vbox->pack_start(*copyright, PACK_SHRINK);
		license_vbox->pack_start(*gpl, PACK_SHRINK);

		/* pack that bitch */
		table->attach(*logo_image, 0, 1, 0, 6);
		table->attach(*title, 1, 2, 0, 1);
		table->attach(*jim, 1, 2, 1, 2);
		table->attach(*matt, 1, 2, 2, 3);
		table->attach(*description, 1, 2, 3, 4);
		table->attach(*web_link, 1, 2, 4, 5);
		table->attach(*license_vbox, 1, 2, 5, 6);
		table->attach(*close, 0, 2, 6, 7);

		table->set_row_spacings(8);
		table->set_col_spacings(8);
		table->set_row_spacing(1, 0);

		about_box->show_all();
	}
}

void Gnomoradio::MainWindow::on_uri_entry_activate ()
{
	Song(uri_entry.get_text())->import();

	if (add_uri_box) {
		delete add_uri_box;
		add_uri_box = 0;
	}
}

void Gnomoradio::MainWindow::on_uri_close_activate ()
{
	add_uri_box->hide();
}

void Gnomoradio::MainWindow::on_about_box_close ()
{
	if (about_box) {
		delete about_box;
		about_box = 0;
	}
}

void Gnomoradio::MainWindow::on_chooser_drag_data_received (const Glib::RefPtr<Gdk::DragContext>& context, int x, int y, GtkSelectionData* selection_data, guint info, guint time)
{
	if (selection_data->format != 8)
		return;

	char *url = g_strndup((char*) selection_data->data, selection_data->length);

	int cell_x, cell_y;
	TreeModel::Path path;
	TreeViewColumn *column;
	chooser.get_path_at_pos(x, y, path, column, cell_x, cell_y);

	store->store->get_iter(path)->get_value(store->song_list_col).push_back(Song(url));

	g_free(url);
}

void Gnomoradio::MainWindow::on_state_changed ()
{
	file_findcurrentsong.set_sensitive(&*wr->player()->get_song());
}

void Gnomoradio::MainWindow::find_current_song ()
{
	list.set(wr->player()->get_song_list());
	list.scroll_to_song(wr->player()->get_song_list()->get_current_song());
	playlist_settings.set_sensitive(true);
	playlist_remove.set_sensitive(list.current_list().destroyable());
}
