// -*- c++ -*-

//  Gnomoradio - roboradio/state.h
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __ROBORADIO_STATE_H
#define __ROBORADIO_STATE_H

#include "roboradio/song-list-radio.h"
#include "roboradio/song-list-library.h"
#include "rainbow/alarm.h"

#include <libxml++/libxml++.h>
#include <glibmm.h>

#include <memory>
#include <string>
#include <set>

namespace Roboradio
{
	class Recommendation;

	class State : public SigC::Object
	{
	public:
		State ();
		~State ();

		void load_preferences ();
		void load_songs ();
		void load_song_paths ();
		void load_song_lists ();
		Recommendation *load_recommendation ();
		void done_loading ();

		void save ();

		std::set<Glib::ustring> get_song_path () const
			{ return song_path; }
		void add_song_path (const Glib::ustring &value);
		void remove_song_path (const Glib::ustring &value);

		Time get_plan_ahead_time () const
			{ return plan_ahead_time; }
		void set_plan_ahead_time (Time t)
			{ plan_ahead_time = t; }

		bool get_rainbow_enabled () const
			{ return rainbow_enabled; }
		void set_rainbow_enabled (bool enabled=true);
		void set_rainbow_disabled ()
			{ set_rainbow_enabled(false); }

		Glib::ustring get_rainbow_hub () const
			{ return rainbow_hub; }
		void set_rainbow_hub (const Glib::ustring &host);

		unsigned int get_rainbow_cache_size () const
			{ return cache_size; }
		void set_rainbow_cache_size (unsigned int megabytes);

		bool get_conserve_privacy () const
			{ return conserve_privacy; }
		void set_conserve_privacy (bool conserve=true)
			{ conserve_privacy = conserve; }

		bool state_file_exists () const
			{ return state_file_already_exists; }

		SigC::Signal1<void,bool> signal_rainbow_enabled_changed;
		SigC::Signal1<void,Glib::ustring> signal_rainbow_hub_changed;
		SigC::Signal1<void,unsigned int> signal_rainbow_cache_size_changed;

		ref_ptr<SongList> library () const { return ref_ptr<SongList>(lib); }
		
	private:
		void find_songs (const Glib::ustring &directory, unsigned short recursion_level = 0);
		void scan_for_new_songs ();
		bool rescan_timeout_handler ();
		void on_save_alarm ();

		std::auto_ptr<xmlpp::DomParser> tree;

		std::set<Glib::ustring> song_path;
		Time plan_ahead_time;
		Glib::ustring rainbow_hub;
		unsigned int cache_size;
		bool rainbow_enabled, conserve_privacy;
		bool loaded_song_paths;
		bool state_file_already_exists;

		Rainbow::Alarm save_alarm;
		static const unsigned int save_interval = 600;

		SongListLibrary *lib;
	};
}

#endif
