//  Gnomoradio - roboradio/song-list.cc
//  Copyright (C) 2003-2004  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "song-list.h"
#include "song-list-radio.h"

using namespace std;
using namespace SigC;
using namespace Glib;
using namespace Roboradio;

static vector<ref_ptr<SongList> > lists;

SigC::Signal1<void,ref_ptr<SongList> > Roboradio::SongList::signal_new_named_song_list;
SigC::Signal1<void,ref_ptr<SongList> > Roboradio::SongList::signal_named_song_list_destroyed;
SigC::Signal1<void,ref_ptr<SongList> > Roboradio::SongList::signal_global_name_changed;

Roboradio::SongList::SongList (const ustring &n)
	: refcnt(0),
	  list_begin(0),
	  list_last(0),
	  count(0),
	  name(n),
	  current_song(0),
	  repeatable(true),
	  shufflable(true),
	  repeat(false),
	  do_upcoming_ref(true)
{
	if (n != "") {
		lists.push_back(ref_ptr<SongList>(this));
		signal_new_named_song_list.emit(ref_ptr<SongList>(this));
	}

	Song::signal_global_song_info_changed.connect(slot(*this, &SongList::on_song_info_changed));
	Song::signal_global_song_rating_changed.connect(slot(*this, &SongList::on_song_info_changed));
	Song::signal_global_song_length_changed.connect(slot(*this, &SongList::on_song_info_changed));
	Song::signal_global_song_status_changed.connect(slot(*this, &SongList::on_song_info_changed));
	Song::signal_global_song_import_progress.connect(slot(*this, &SongList::on_song_import_progress));
}

Roboradio::SongList::~SongList ()
{
	// free all elements
	if (!list_begin)
		return;
	Element *p = list_begin->next;
	while (p) {
		delete p->prev;
		p = p->next;
	}
	delete list_last;
}

bool Roboradio::SongList::destroyable ()
{
	return true;
}

void Roboradio::SongList::destroy ()
{
	if (!destroyable())
		return; // error

	ref_ptr<SongList> list(this);

	//remove from lists
	vector<ref_ptr<SongList> >::iterator i;
	for (i = lists.begin(); i != lists.end(); ++i)
		if (&(**i) == this) {
			lists.erase(i);
			break;
		}

	signal_destroyed();

	if (name != "")
		signal_named_song_list_destroyed(list);
}

void Roboradio::SongList::set_name (const ustring &n)
{
	if (name != n) {
		name = n;
		signal_name_changed(name);
		signal_global_name_changed(ref_ptr<SongList>(this));
	}
}

void Roboradio::SongList::set_repeat (bool r)
{
	if (repeatable && (r != repeat)) {
		repeat = r;
		signal_repeat_changed(repeat);
	}
}

void Roboradio::SongList::set_shuffle (bool s)
{
	if (get_shufflable() && (s != get_shuffle())) {
		if (s) {
			shuffle_list = ref_ptr<SongList>(new SongListRadio(ref_ptr<SongList>(this)));
			if (current_song != end()) {
				// transfer playing status
				shuffle_list->push_front(*current_song);
				shuffle_list->current_song = shuffle_list->begin();
				transfer_play(shuffle_list, shuffle_list->begin(), false);
			}
		} else {
			if (shuffle_list->current_song != shuffle_list->end()) {
				// transfer playing status back
				iterator song = begin();
				while (song != end() && &**song != &**shuffle_list->current_song)
				     ++song;
				if (song != end())
					shuffle_list->transfer_play(ref_ptr<SongList>(this), song, false);
				else
					shuffle_list->stop(); // there was some serious error here, and it may not be reported correctly
			}
			shuffle_list = ref_ptr<SongList>();
		}
		signal_shuffle_changed(s);
	}
}

Time Roboradio::SongList::calculate_total_time () const
{
	Time retval = 0;
	iterator i;

	for (i = begin(); i != end(); ++i)
		retval += (*i)->get_length();

	return retval;
}

void Roboradio::SongList::play (iterator start)
{
	set_shuffle(false);

	while (start != end() && !(*start)->get_status().ready)
		++start;
	if (start == end())
		return;

	(*start)->set_playback(PLAY);
	song_done_connection = (*start)->signal_done.connect(slot(*this, &SongList::next));
	current_song = start;
	signal_current_song_changed();
}

void Roboradio::SongList::stop ()
{
	(*current_song)->set_playback(STOP);
	song_done_connection.disconnect();
	current_song = end();
	signal_current_song_changed();
}

void Roboradio::SongList::transfer_play (ref_ptr<SongList> new_list,
					 iterator new_song,
					 bool restart_paused_song)
{
	if (current_song == end())
		new_list->play(new_song);
	else if (&**current_song != &**new_song
		 || (restart_paused_song
		     && (*current_song)->get_playback() == PAUSE)
		 || &*new_list == this) {
		stop();
		new_list->play(new_song);
	} else {
		song_done_connection.disconnect();
		current_song = end();
		new_list->song_done_connection = (*new_song)->signal_done.connect(slot(*new_list, &SongList::next));
		signal_current_song_changed();
		new_list->current_song = new_song;
		new_list->signal_current_song_changed();
	}
}

void Roboradio::SongList::prev ()
{
	(*current_song)->set_playback(STOP);
	song_done_connection.disconnect();
	if (current_song != begin())
		--current_song;
	while (current_song != begin() && !(*current_song)->get_status().ready)
		--current_song;
	(*current_song)->set_playback(PLAY);
	song_done_connection = (*current_song)->signal_done.connect(slot(*this, &SongList::next));
	signal_current_song_changed();
}

void Roboradio::SongList::next ()
{
	(*current_song)->set_playback(STOP);

	iterator song_we_started_with = current_song;
	for (;;) {
		++current_song;
		if (current_song == end()) {
			if (repeat)
				current_song = begin();
			else
				break;
		}
		if ((*current_song)->get_status().ready)
			break;
		if (current_song == song_we_started_with) {
			// we have cycled around the loop and no songs are available
			current_song = end();
			break;
		}
	}
	
	if (current_song == end() || &**current_song != &**song_we_started_with) {
		song_done_connection.disconnect();
		if (current_song != end())
			song_done_connection = (*current_song)->signal_done.connect(slot(*this, &SongList::next));
	}

	if (current_song != end())
		(*current_song)->set_playback(PLAY);
	else
		signal_done();
	signal_current_song_changed();
}

Roboradio::SongList::iterator Roboradio::SongList::insert (Roboradio::SongList::iterator pos, const SongRef &value)
{
	if (pos == end()) {
		push_back(value);
		// signal_inserted is called by push_back
		return iterator(list_last);
	} else {
		Element *el = new Element(value, do_upcoming_ref);
		el->prev = pos.node->prev;
		el->next = pos.node;
		if (pos.node->prev)
			pos.node->prev->next = el;
		else
			list_begin = el;
		pos.node->prev = el;
		++count;
		signal_inserted(iterator(el));
		return iterator(el);
	}
}

void Roboradio::SongList::move (iterator old_pos, iterator new_pos)
{
	if (old_pos == new_pos)
		return;

	// remove from list
	if (old_pos.node->prev)
		old_pos.node->prev->next = old_pos.node->next;
	else
		list_begin = old_pos.node->next;
	if (old_pos.node->next)
		old_pos.node->next->prev = old_pos.node->prev;
	else
		list_last = old_pos.node->prev;

	// insert in new position
	if (new_pos != end()) {
		old_pos.node->prev = new_pos.node->prev;
		old_pos.node->next = new_pos.node;
		if (new_pos.node->prev)
			new_pos.node->prev->next = old_pos.node;
		else
			list_begin = old_pos.node;
		new_pos.node->prev = old_pos.node;
	} else {
		old_pos.node->next = 0;
		old_pos.node->prev = list_last;
		if (list_last)
			list_last->next = old_pos.node;
		else
			list_begin = old_pos.node;
		list_last = old_pos.node;
	}

	signal_moved(old_pos);
}

Roboradio::SongList::iterator Roboradio::SongList::remove (iterator pos)
{
	// take care of case where this song is currently playing
	if (current_song == pos) {
		stop();
		signal_done();
	}

	iterator retval = iterator(pos.node->next);
	if (pos.node->prev)
		pos.node->prev->next = pos.node->next;
	else
		list_begin = pos.node->next;
	if (pos.node->next)
		pos.node->next->prev = pos.node->prev;
	else
		list_last = pos.node->prev;
	--count;
	signal_removed(pos);
	delete pos.node;
	return retval;
}

void Roboradio::SongList::clear ()
{
	while (list_last)
		pop_back();
}

void Roboradio::SongList::clear_except_playing ()
{
	while (list_last && current_song != iterator(list_last))
		pop_back();
	while (list_begin && current_song != iterator(list_begin))
		pop_front();
}

void Roboradio::SongList::pop_front ()
{
	if (list_begin) {
		// take care of case where this song is currently playing
		if (current_song == iterator(list_begin)) {
			stop();
			signal_done();
		}
	
		Element *old = list_begin;
		list_begin = list_begin->next;
		if (list_begin)
			list_begin->prev = 0;
		else
			list_last = 0;
		--count;
		signal_removed(iterator(old));
		delete old;
	}
}

void Roboradio::SongList::pop_back ()
{
	if (list_last) {
		// take care of case where this song is currently playing
		if (current_song == iterator(list_last)) {
			stop();
			signal_done();
		}

		Element *old = list_last;
		list_last = list_last->prev;
		if (list_last)
			list_last->next = 0;
		else
			list_begin = 0;
		--count;
		signal_removed(iterator(old));
		delete old;
	}
}

void Roboradio::SongList::push_front (const SongRef &value)
{
	Element *n = new Element(value, do_upcoming_ref);
	n->prev = 0;
	n->next = list_begin;
	if (list_begin)
		list_begin->prev = n;
	else
		list_last = n;
	list_begin = n;
	++count;
	signal_inserted(list_begin);
}

void Roboradio::SongList::push_back (const SongRef &value)
{
	Element *n = new Element(value, do_upcoming_ref);
	n->next = 0;
	n->prev = list_last;
	if (list_last)
		list_last->next = n;
	else
		list_begin = n;
	list_last = n;
	++count;
	signal_inserted(list_last);
}

vector<ref_ptr<SongList> > Roboradio::SongList::get_named_song_lists ()
{
	return lists;
}

void Roboradio::SongList::on_song_info_changed (SongRef s)
{
	for (iterator i = begin(); i != end(); ++i)
		if (&**i == &*s)
			signal_song_info_changed(i);
}

void Roboradio::SongList::on_song_import_progress (SongRef s, unsigned int p)
{
	for (iterator i = begin(); i != end(); ++i)
		if (&**i == &*s)
			signal_song_import_progress(i, p);
}
