//  Gnomoradio - roboradio/audio/ogg.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "roboradio/audio/ogg.h"
#include <esd.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/select.h>
#include <unistd.h>

#include <string>
#include <algorithm>

#if USE_VORBIS

#if (G_BYTE_ORDER == G_BIG_ENDIAN)
#define READ_ENDIAN 1
#else
#define READ_ENDIAN 0
#endif

using namespace std;
using namespace Glib;

Roboradio::Audio::Ogg::Ogg (const ustring &file)
	: filename(file),
	  thread(0)
{
	done_dispatch.connect(slot(*this, &Ogg::send_done));
	position_dispatch.connect(slot(*this, &Ogg::send_position_changed));
}

Roboradio::Audio::Ogg::~Ogg ()
{
	if (thread)
		stop();
}

void Roboradio::Audio::Ogg::play ()
{
	if (thread)
		return;

	_pause = false;
	_stop = false;
	_position = 0;
	_seek = -1;

	if (!thread_supported())
		thread_init();
	thread = Thread::create(SigC::slot(*this, &Ogg::thread_function), true);
}

void Roboradio::Audio::Ogg::stop ()
{
	if (!thread)
		return;

	mutex.lock();
	_stop = true;
	mutex.unlock();
	thread->join();
	thread = 0;
}

void Roboradio::Audio::Ogg::pause ()
{
	Mutex::Lock lock(mutex);
	_pause = true;
}

void Roboradio::Audio::Ogg::unpause ()
{
	Mutex::Lock lock(mutex);
	_pause = false;
}

void Roboradio::Audio::Ogg::seek (unsigned int pos)
{
	Mutex::Lock lock(mutex);
	_seek = pos;
}

int Roboradio::Audio::Ogg::get_position ()
{
	Mutex::Lock lock(mutex);
	return _position;
}

map<ustring,ustring> Roboradio::Audio::Ogg::get_info (unsigned int &length)
{
	length = 0;
	map<ustring,ustring> info;

	FILE *file = fopen(filename.c_str(), "rb");
	if (file) {
		OggVorbis_File vf;
		if (ov_open(file, &vf, 0, 0) != 0) {
			fclose(file);
		} else {
			length = (unsigned int) ov_time_total(&vf, -1);
			vorbis_comment *c = ov_comment(&vf, -1);
			if (c) {
				for (int i = 0; i < c->comments; ++i) {
					try {
						ustring comment(locale_to_utf8(string(c->user_comments[i])));
						ustring::size_type eq = comment.find('=');
						if (eq != ustring::npos) {
							string key(comment.substr(0, eq));
							transform(key.begin(), key.end(), key.begin(), ::tolower);
							info.insert(make_pair(key, comment.substr(eq + 1)));
						}
					} catch (...) {
					}
				}
			}
			ov_clear(&vf);
		}
	}

	map<ustring,ustring>::iterator p = info.find("title");
	if (p == info.end()
	    || p->second == "") {
		info.erase("title");
		ustring title(filename);
		ustring::size_type s = title.rfind('/');
		if (s != ustring::npos)
			title = title.substr(s + 1);
		info.insert(make_pair(ustring("title"), title));
	}

	return info;
}

void Roboradio::Audio::Ogg::thread_function ()
{
	FILE *file = fopen(filename.c_str(), "rb");

	if (!file) {
		mutex.lock();
		_position = -1;
		mutex.unlock();
		done_dispatch();
		return;
	}

	OggVorbis_File vf;
	if (ov_open(file, &vf, 0, 0) != 0) {
		fclose(file);
		mutex.lock();
		_position = -1;
		mutex.unlock();
		done_dispatch();
		return;
	}

	const unsigned int buf_size = ESD_BUF_SIZE;
	char buffer[buf_size];

	int esd_socket = esd_play_stream_fallback(ESD_BITS16 | ESD_STREAM | ESD_PLAY | ESD_STEREO, 44100, 0, 0);

	if (esd_socket > 0) {
		int bitstream = 0;
		for (;;) {
			bool currently_paused;
			bool stopped = false;
			{
				Mutex::Lock lock(mutex);
				if (_seek >= 0) {
					ov_time_seek(&vf, double(_seek));
					_seek = -1;
				}
				currently_paused = _pause;
			}
			
			if (currently_paused) {
				for (;;) {
					struct timeval tv;
					tv.tv_sec = 1;
					tv.tv_usec = 0;
					select(0, 0, 0, 0, &tv);
					Mutex::Lock lock(mutex);
					stopped = _stop;
					if (!_pause || _stop)
						break;
				}
				if (stopped)
					break;
			}

			long r = ov_read(&vf, buffer, buf_size, READ_ENDIAN, 2, 1, &bitstream);
			if (r <= 0)
				break;
			else {
				size_t bytes_written = 0;
				while (bytes_written < r) {
					ssize_t w = write(esd_socket, buffer + bytes_written, r - bytes_written);
					if (w < 0) {
						r = 0;
						break;
					}
					bytes_written += w;
				}

				unsigned int current_pos = (unsigned int) ov_time_tell(&vf);

				Mutex::Lock lock(mutex);
				if (r <= 0 || _stop)
					break;
				if (current_pos != _position) {
					_position = current_pos;
					position_dispatch();
				}
			}
		}
		close(esd_socket);
	}

	mutex.lock();
	bool s = _stop;
	mutex.unlock();

	if (!s)
		done_dispatch();

	ov_clear(&vf);
}

void Roboradio::Audio::Ogg::send_position_changed ()
{
	signal_position_changed(get_position());
}

void Roboradio::Audio::Ogg::send_done ()
{
	thread->join();
	thread = 0;
	signal_done();
}

#endif
