/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

const Lang = imports.lang;
const Cairo = imports.cairo;
const GObject = imports.gi.GObject;
const Gdk = imports.gi.Gdk;
const Gtk = imports.gi.Gtk;

PixelGrid = new GType(_PixelGrid = {
    parent: Gtk.DrawingArea.type,
    name: "PixelGrid",
    properties: [
        {
            // FIXME: This may not be accessible in the C interface
            name: "puzzlePixels",
            type: GObject.TYPE_ARRAY,
            default_value: null,
            flags: GObject.ParamFlags.READABLE
                 | GObject.ParamFlags.WRITABLE
                 | GObject.ParamFlags.CONSTRUCT
        },
        {
            name: "drawingColor",
            type: GObject.TYPE_UINT,
            default_value: null,
            flags: GObject.ParamFlags.READABLE
                 | GObject.ParamFlags.WRITABLE
                 | GObject.ParamFlags.CONSTRUCT
        }
    ],

    signals: [
        // Maintain this order to work around argument count bug in seed
        {
            name: "puzzle_full"
        },
        {
            name: "puzzle_changed",
            parameters: [GObject.TYPE_OBJECT]
        }
    ],

    _UNKNOWN_PIXEL_COLOR: 0x7f7f7fff,
    _BACKGROUND_COLOR: 0xffffffff,
    _NO_COLOR: 0,

    __puzzlePixels: null,
    __drawingColor: null,
    _unknownPixelsCount: 0,

    _lastPressedButton: null,
    _lastButtonPressPosition: null,

    class_init: function(klass, prototype) {
        prototype._BACKGROUND_COLOR = _PixelGrid._BACKGROUND_COLOR;
        prototype._UNKNOWN_PIXEL_COLOR = _PixelGrid._UNKNOWN_PIXEL_COLOR;
        prototype._NO_COLOR = _PixelGrid._NO_COLOR;

        prototype.__puzzlePixels = _PixelGrid.__puzzlePixels;
        prototype.__drawingColor = _PixelGrid.__drawingColor;
        prototype._unknownPixelsCount = _PixelGrid._unknownPixelsCount;

        prototype._lastPressedButton = _PixelGrid._lastPressedButton;
        prototype._lastButtonPressPosition =
            _PixelGrid._lastButtonPressPosition;

        prototype.__defineGetter__("_puzzlePixels",
                                   _PixelGrid._getPuzzlePixels);
        prototype.__defineSetter__("_puzzlePixels",
                                   _PixelGrid._setPuzzlePixels);
        prototype.__defineGetter__("puzzlePixels", _PixelGrid._getPuzzlePixels);
        prototype.__defineSetter__("puzzlePixels", _PixelGrid._setPuzzlePixels);

        prototype.__defineGetter__("_drawingColor",
                                   _PixelGrid._getDrawingColor);
        prototype.__defineSetter__("_drawingColor",
                                   _PixelGrid._setDrawingColor);
        prototype.__defineGetter__("drawingColor", _PixelGrid._getDrawingColor);
        prototype.__defineSetter__("drawingColor", _PixelGrid._setDrawingColor);

        prototype.createBlank = _PixelGrid.createBlank;
        prototype.reset = _PixelGrid.reset;
        prototype._countUnknownPixels = _PixelGrid._countUnknownPixels;
        prototype._onDraw = _PixelGrid._onDraw;
        prototype._getPixelFromWidgetCoordinates =
            _PixelGrid._getPixelFromWidgetCoordinates;
        prototype._onButtonPress = _PixelGrid._onButtonPress;
        prototype._onMouseMove = _PixelGrid._onMouseMove;
        prototype._setPixelsColor = _PixelGrid._setPixelsColor;
        prototype.revealRemaining = _PixelGrid.revealRemaining;
    },

    init: function() {
        this.add_events(Gdk.EventMask.BUTTON_PRESS_MASK);
        this.add_events(Gdk.EventMask.BUTTON1_MOTION_MASK);
        this.add_events(Gdk.EventMask.BUTTON2_MOTION_MASK);
        this.add_events(Gdk.EventMask.BUTTON3_MOTION_MASK);

        this.signal.expose_event.connect(Lang.bind(this, this._onDraw));
        this.signal.button_press_event.connect(Lang.bind(this,
                                                 this._onButtonPress));
        this.signal.motion_notify_event.connect(Lang.bind(this,
                                                 this._onMouseMove));
    },

    createBlank: function(columns, rows) {
        this._numRows = rows;
        this._numColumns = columns;
        this.reset();
    },

    reset: function() {
        this.__puzzlePixels = [];
        for (var y = 0; y < this._numRows; ++y) {
            this.__puzzlePixels[y] = [];
            for (var x = 0; x < this._numColumns; ++x) {
                this.__puzzlePixels[y][x] = this._NO_COLOR;
            }
        }

        this.queue_draw();
        this._unknownPixelsCount = this._numRows * this._numColumns;
    },

    _getPuzzlePixels: function() {
        return this.__puzzlePixels;
    },

    _setPuzzlePixels: function(puzzlePixels) {
        this.__puzzlePixels = puzzlePixels;

        this._numRows = this._puzzlePixels.length;
        this._numColumns = this._puzzlePixels[0].length;

        this._countUnknownPixels();

        this.queue_draw();
    },

    _getDrawingColor: function() {
        return this.__drawingColor;
    },

    _setDrawingColor: function(value) {
        this.__drawingColor = value;
    },

    _countUnknownPixels: function() {
        this._unknownPixelsCount = 0;

        for (var y = 0; y < this._numRows; ++y) {
            for (var x = 0; x < this._numColumns; ++x) {
                if (this._puzzlePixels[y][x] == this._NO_COLOR)
                    ++this._unknownPixelsCount;
            }
        }
    },

    _onDraw: function(drawingArea) {
        var cairoContext = new Cairo.Context.from_drawable(drawingArea.window);

        var size = drawingArea.window.get_size();
        cairoContext.scale(size.width / this._numColumns,
                           size.height / this._numRows);

        for (var i = 0; i < this._numRows; ++i) {
            for (var j = 0; j < this._numColumns; ++j) {
                var color = this._puzzlePixels[i][j];
                if (color == this._NO_COLOR)
                    color = this._UNKNOWN_PIXEL_COLOR;

                var r = ((color >> 24) & 0xff) / 255.0;
                var g = ((color >> 16) & 0xff) / 255.0;
                var b = ((color >> 8) & 0xff) / 255.0;

                cairoContext.set_source_rgb(r, g, b);
                cairoContext.rectangle(j, i, 1, 1);
                cairoContext.fill();
            }
        }

        cairoContext.set_source_rgb(0.0, 0.0, 0.0);

        // TODO: change the scale to use integer boundaries for line
        // drawing
        for (i = 0; i < this._numRows + 1; ++i) {
            cairoContext.line_width = 1.0;
            if (i % 5 == 0)
                cairoContext.line_width = 3.0;
            if (i == 0 || i == this._numRows)
                cairoContext.line_width = 6.0;

            cairoContext.move_to(0, i);
            cairoContext.line_to(this._numColumns, i);
            cairoContext.save();
            cairoContext.scale(this._numColumns / size.width,
                               this._numRows / size.height);
            cairoContext.stroke();
            cairoContext.restore();
        }

        for (i = 0; i < this._numColumns + 1; ++i) {
            cairoContext.line_width = 1.0;
            if (i % 5 == 0)
                cairoContext.line_width = 3.0;
            if (i == 0 || i == this._numColumns)
                cairoContext.line_width = 6.0;

            cairoContext.move_to(i, 0);
            cairoContext.line_to(i, this._numRows);
            cairoContext.save();
            cairoContext.scale(this._numColumns / size.width,
                               this._numRows / size.height);
            cairoContext.stroke();
            cairoContext.restore();
        }

        return false;
    },

    _getPixelFromWidgetCoordinates: function(x, y) {
        var widgetSize = this.window.get_size();

        var row = Math.floor(y * this._numRows / widgetSize.height);
        var column = Math.floor(x * this._numColumns / widgetSize.width);

        if ((row < 0) || (row >= this._numRows) ||
            (column < 0) || (column >= this._numColumns)) {
            print("Unexpected row, column from click. Coding error.");
            return null;
        }

        return {y: row, x: column};
    },

    _onButtonPress: function(widget, event) {
        event = event.button;

        var pixel = this._getPixelFromWidgetCoordinates(event.x, event.y);
        if (pixel == null)
            return false;

        this._lastPressedButton = event.button;
        this._lastButtonPressPosition = pixel;

        var newColor = this._drawingColor;
        if (event.button == 1)
            newColor = this._drawingColor;
        else if (event.button == 3)
            newColor = this._BACKGROUND_COLOR;
        else if (event.button == 2)
            newColor = this._NO_COLOR;
        else
            return false;

        var pixelChanges = new PixelChanges();
        pixelChanges.changeArray = [{ x: pixel.x,
                                      y: pixel.y,
                                      newColor: newColor}];

        this._setPixelsColor(pixelChanges);

        return false;
    },

    _onMouseMove: function(widget, event) {
        event = event.motion;

        var newColor = this._drawingColor;
        if (this._lastPressedButton == 1)
            newColor = this._drawingColor;
        else if (this._lastPressedButton == 3)
            newColor = this._BACKGROUND_COLOR;
        else if (this._lastPressedButton == 2)
            newColor = this._NO_COLOR;
        else
            return false;

        var widgetSize = this.window.get_size();
        if (event.x < 0)
            event.x = 0;

        if (event.x >= widgetSize.width)
            event.x = widgetSize.width - 1;

        if (event.y < 0)
            event.y = 0;

        if (event.y >= widgetSize.height)
            event.y = widgetSize.height - 1;

        var pixel = this._getPixelFromWidgetCoordinates(event.x, event.y);
        if (pixel == null)
            return false;

        var xDelta = pixel.x - this._lastButtonPressPosition.x;
        var yDelta = pixel.y - this._lastButtonPressPosition.y;

        var pixelChanges = new PixelChanges();
        pixelChanges.changeArray = [];

        if (Math.abs(yDelta) > Math.abs(xDelta)) {
            if (yDelta == 0)
                return false;

            var increment = Math.round(yDelta / Math.abs(yDelta));
            for (var i = this._lastButtonPressPosition.y;
                 i != (pixel.y + increment);
                 i += increment) {
                pixelChanges.changeArray.push({
                    x: this._lastButtonPressPosition.x,
                    y: i,
                    newColor: newColor
                });
            }
        } else {
            if (xDelta == 0)
                return false;

            var increment = Math.round(xDelta / Math.abs(xDelta));
            for (var i = this._lastButtonPressPosition.x;
                 i != (pixel.x + increment);
                 i += increment) {
                pixelChanges.changeArray.push({
                    x: i,
                    y: this._lastButtonPressPosition.y,
                    newColor: newColor
                });
            }
        }

        this._setPixelsColor(pixelChanges);

        return false;
    },

    _setPixelsColor: function(pixelChanges) {
        var emitPixelChanges = new PixelChanges();
        emitPixelChanges.puzzlePixels = this._puzzlePixels;
        emitPixelChanges.changeArray = [];

        for (var i = 0; i < pixelChanges.changeArray.length; ++i) {
            var change = pixelChanges.changeArray[i];
            change.prevColor = this._puzzlePixels[change.y][change.x];

            if (change.prevColor == change.newColor)
                continue;

            emitPixelChanges.changeArray.push(change);

            this._puzzlePixels[change.y][change.x] = change.newColor;

            if (change.prevColor == this._NO_COLOR)
                --this._unknownPixelsCount;
            else if (change.newColor == this._NO_COLOR)
                ++this._unknownPixelsCount;
        }

        if(emitPixelChanges.changeArray.length == 0)
            return;

        this.queue_draw();

        this.signal.puzzle_changed.emit(emitPixelChanges);

        if (this._unknownPixelsCount == 0)
            this.signal.puzzle_full.emit();
    },

    revealRemaining: function() {
        for (var y = 0; y < this._numRows; ++y) {
            for (var x = 0; x < this._numColumns; ++x) {
                if (this._puzzlePixels[y][x] == this._NO_COLOR)
                    this._puzzlePixels[y][x] = this._BACKGROUND_COLOR;
            }
        }

        this.queue_draw();
    }
});

PixelChanges = new GType(_PixelChanges = {
    parent: GObject.TYPE_OBJECT,
    name: "PixelChanges"
});
