/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* parted_functions.c contains GNU Parted functions */

#include "parted_functions.h"

//This code was extracted of the "GNU Parted" program, thanks for all.
int
ped_assert (int cond, const char *cond_text,
			const char *file, int line, const char *function)
{
	PedExceptionOption opt;

	if (cond)
		return 1;

	opt = ped_exception_throw (PED_EXCEPTION_BUG,
							   PED_EXCEPTION_IGNORE_CANCEL,
							   "Assertion (%s) at %s:%d in function %s() failed.",
							   cond_text, file, line, function);

	return opt == PED_EXCEPTION_IGNORE;
}

#ifdef __GNUC__

#define PED_ASSERT(cond, action)				\
	do {							\
	if (!ped_assert (cond,					\
			 #cond,					\
			 __FILE__,				\
			 __LINE__,				\
			 __PRETTY_FUNCTION__))			\
	{							\
		action;						\
	}							\
	} while (0)

#else /* !__GNUC__ */

#define PED_ASSERT(cond, action)				\
	do {							\
	if (!ped_assert (cond,					\
			 #cond,					\
			 _("unknown"),				\
			 0,					\
			 _("unknown")))				\
	{							\
		action;						\
	}							\
	} while (0)
#endif /* __GNUC__ */

int
snap (PedSector * sector, PedSector new_sector, PedGeometry * range)
{
	PED_ASSERT (ped_geometry_test_sector_inside (range, *sector), return 0);
	if (!ped_geometry_test_sector_inside (range, new_sector))
		return 0;
	*sector = new_sector;
	return 1;
}

typedef enum
{
	MOVE_NO = 0,
	MOVE_STILL = 1,
	MOVE_UP = 2,
	MOVE_DOWN = 4
} EMoves;

enum
{								/* Don't change these values
								 * - They would never do it */
	SECT_START = 0,
	SECT_END = -1
};

EMoves
prefer_snap (PedSector s, int what, PedGeometry * range, EMoves * allow,
			 PedPartition * part, PedSector * dist)
{
	PedSector up_dist = -1, down_dist = -1;
	PedSector new_sect;
	EMoves move;

	PED_ASSERT (what == SECT_START || what == SECT_END, return 0);

	if (!(*allow & (MOVE_UP | MOVE_DOWN)))
	{
		*dist = 0;
		return MOVE_STILL;
	}

	if (*allow & MOVE_UP)
	{
		new_sect = part->geom.end + 1 + what;
		if (ped_geometry_test_sector_inside (range, new_sect))
			up_dist = new_sect - s;
		else
			*allow &= ~MOVE_UP;
	}

	if (*allow & MOVE_DOWN)
	{
		new_sect = part->geom.start + what;
		if (ped_geometry_test_sector_inside (range, new_sect))
			down_dist = s - new_sect;
		else
			*allow &= ~MOVE_DOWN;
	}

	move = MOVE_STILL;
	if ((*allow & MOVE_UP) && (*allow & MOVE_DOWN))
	{
		if (down_dist < up_dist || (down_dist == up_dist && what == SECT_START))
			move = MOVE_DOWN;
		else if (up_dist < down_dist || (down_dist == up_dist
										 && what == SECT_END))
			move = MOVE_UP;
		else
			PED_ASSERT (0, return 0);
	}
	else if (*allow & MOVE_UP)
		move = MOVE_UP;
	else if (*allow & MOVE_DOWN)
		move = MOVE_DOWN;

	*dist = (move == MOVE_DOWN ? down_dist : (move == MOVE_UP ? up_dist : 0));
	return move;
}

/* Snaps a partition to nearby partition boundaries.  This is useful for
 * gobbling up small amounts of free space, and also for reinterpreting small
 * changes to a partition as non-changes (eg: perhaps the user only wanted to
 * resize the end of a partition).
 * 	Note that this isn't the end of the story... this function is
 * always called before the constraint solver kicks in.  So you don't need to
 * worry too much about inadvertantly creating overlapping partitions, etc.
 */
void
snap_to_boundaries (PedGeometry * new_geom, PedGeometry * old_geom,
					PedDisk * disk,
					PedGeometry * start_range, PedGeometry * end_range)
{
	PedPartition *start_part;
	PedPartition *end_part;
	PedSector start = new_geom->start;
	PedSector end = new_geom->end;
	PedSector start_dist = -1, end_dist = -1;
	EMoves start_allow, end_allow, start_want, end_want;
	int adjacent;

	start_want = end_want = MOVE_NO;
	start_allow = end_allow = MOVE_STILL | MOVE_UP | MOVE_DOWN;

	start_part = ped_disk_get_partition_by_sector (disk, start);
	end_part = ped_disk_get_partition_by_sector (disk, end);
	adjacent = (start_part->geom.end + 1 == end_part->geom.start);

	/* If we can snap to old_geom, then we will... */
	/* and this will enforce the snaped positions  */
	if (old_geom)
	{
		if (snap (&start, old_geom->start, start_range))
			start_allow = MOVE_STILL;
		if (snap (&end, old_geom->end, end_range))
			end_allow = MOVE_STILL;
	}

	/* If start and end are on the same partition, we */
	/* don't allow them to cross. */
	if (start_part == end_part)
	{
		start_allow &= ~MOVE_UP;
		end_allow &= ~MOVE_DOWN;
	}

	/* Let's find our way */
	start_want =
		prefer_snap (start, SECT_START, start_range, &start_allow,
					 start_part, &start_dist);
	end_want =
		prefer_snap (end, SECT_END, end_range, &end_allow, end_part, &end_dist);

	PED_ASSERT (start_dist >= 0 && end_dist >= 0, return);

	/* If start and end are on adjacent partitions,    */
	/* and if they would prefer crossing, then refrain */
	/* the farest to do so. */
	if (adjacent && start_want == MOVE_UP && end_want == MOVE_DOWN)
	{
		if (end_dist < start_dist)
		{
			start_allow &= ~MOVE_UP;
			start_want = prefer_snap (start, SECT_START,
									  start_range, &start_allow,
									  start_part, &start_dist);
			PED_ASSERT (start_dist >= 0, return);
		}
		else
		{
			end_allow &= ~MOVE_DOWN;
			end_want = prefer_snap (end, SECT_END,
									end_range, &end_allow, end_part, &end_dist);
			PED_ASSERT (end_dist >= 0, return);
		}
	}

	/* New positions */
	start = (start_want == MOVE_DOWN ? start_part->geom.start :
			 (start_want == MOVE_UP ? start_part->geom.end + 1 : start));
	end = (end_want == MOVE_DOWN ? end_part->geom.start - 1 :
		   (end_want == MOVE_UP ? end_part->geom.end : end));
	PED_ASSERT (ped_geometry_test_sector_inside (start_range, start), return);
	PED_ASSERT (ped_geometry_test_sector_inside (end_range, end), return);
	PED_ASSERT (start <= end, return);
	ped_geometry_set (new_geom, start, end - start + 1);
}

PedConstraint *
constraint_intersect_and_destroy (PedConstraint * a, PedConstraint * b)
{
	PedConstraint *result = ped_constraint_intersect (a, b);
	ped_constraint_destroy (a);
	ped_constraint_destroy (b);
	return result;
}

PedConstraint *
constraint_from_start_end (PedDevice * dev,
						   PedGeometry * range_start, PedGeometry * range_end)
{
	return ped_constraint_new (ped_alignment_any, ped_alignment_any,
							   range_start, range_end, 1, dev->length);
}

//End of "GNU Parted" code
