/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* net.c implements functions to work on line.
 * Include:
 * - The server and clients of a link mode.
 * - The server and clients of a p2p mode.
 * - Order the net from slowest to fastest computer*/

#include <stdio.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <libintl.h>
#include <locale.h>

#include "util.h"
#include "net.h"

#define _(x) gettext(x)
#define N_(x) (x)

#define port_ping 7772
#define port_data 7773

int
get_wait_seconds ()
{
	return wait_seconds;
}

/* This function is executed in a link, wait for a signal to continue, *
 * call to the speed test and return the ip address of the next link. *
 * This function communicates with the function "net_scan" of the server */
int
answer ()
{
	int sock_udp;
	int next_link;
	struct sockaddr_in udp;
	socklen_t size = sizeof (struct sockaddr);

	udp.sin_family = AF_INET;
	udp.sin_port = htons (port_ping);
	udp.sin_addr.s_addr = INADDR_ANY;

	if ((sock_udp = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	{
		msgerror (_("Unable to open UDP socket"));
		return -1;
	}

	if ((bind (sock_udp, (struct sockaddr *) &udp, size)) < 0)
	{
		msgerror (_("Unable to open UDP port"));
		return -1;
	}

	printf (_("Waiting for carrier's signal...\n"));
	fflush (stdout);
	if ((recvfrom (sock_udp, 0, 0, 0, (struct sockaddr *) &udp, &size) < 0))
	{
		msgerror (_("It was not possible to receive the signal"));
		return -1;
	}

	if ((sendto (sock_udp, 0, 0, 0, (struct sockaddr *) &udp, size)) < 0)
	{
		msgerror (_("It was not was possible to send the signal"));
		return -1;
	}

	printf (_("Waiting to next link ip address... "));
	fflush (stdout);
	if ((recvfrom
		 (sock_udp, &next_link, sizeof (int), 0, (struct sockaddr *) &udp,
		  &size)) < 0)
	{
		msgerror (_("It was not possible to receive the signal"));
		return -1;
	}
	else
	{
		printf ("OK\n");
		fflush (stdout);
	}

	close (sock_udp);

	return next_link;
}

/* Send a signal to the net and wait the answers of the links. * 
 * This function communicates with the function "answer" of the links */
int
net_scan (int hosts)
{
	int j;
	int i = 0;
	int sock_udp;
	fd_set sockets_list;
	int links[hosts];
	struct sockaddr_in udp;
	struct timeval timeout = { wait_seconds, 0 };
	socklen_t size = sizeof (struct sockaddr);

	udp.sin_family = AF_INET;
	udp.sin_port = htons (port_ping);
	if (broadcast_addr[0] != '\0')
	{
		udp.sin_addr.s_addr = inet_addr (broadcast_addr);
	}
	else
	{
		udp.sin_addr.s_addr = inet_addr ("224.0.0.1");
	}

	if ((sock_udp = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
	{
		msgerror (_("Unable to open UDP socket"));
		return -1;
	}

	int opt = 1;
	setsockopt (sock_udp, SOL_SOCKET, SO_BROADCAST, &opt, sizeof (opt));

	if ((sendto (sock_udp, 0, 0, 0, (struct sockaddr *) &udp, size)) < 0)
	{
		msgerror (_("It was not possible to send the signal"));
		return -1;
	}

	FD_ZERO (&sockets_list);
	FD_SET (sock_udp, &sockets_list);

	printf (_("Gnetic now will create a matrix of nodes in this order:\n"));
	fflush (stdout);
	while (select (sock_udp + 1, &sockets_list, NULL, NULL, &timeout) > 0)
	{
		if (i <= hosts - 1)
		{
			recvfrom (sock_udp, 0, 0, 0, (struct sockaddr *) &udp, &size);
			printf ("%i: %s\n", i + 1, inet_ntoa (udp.sin_addr));
			fflush (stdout);
			links[i] = udp.sin_addr.s_addr;
		}
		else
		{
			printf (_("WARNING: Only #%i links, use -m\n"), hosts);
			fflush (stdout);
		}
		links[i + 1] = 0;
		i++;
	}

	if (!i)
	{
		msgerror (_("There are no links in this network"));
		return -1;
	}


	for (j = 0; j < i; j++)
	{
		udp.sin_addr.s_addr = links[j];
		if ((sendto
			 (sock_udp, &links[j + 1], sizeof (int), 0,
			  (struct sockaddr *) &udp, size)) < 0)
		{
			msgerror (_
					  ("Isn't was possible to contact with someone of receivers"));
			return -1;
		}
	}
	close (sock_udp);

	return links[0];
}

/* This function wait for the previuos link, establishes connection with *
 * the following link and return a structure that contains two sockets *
 * connected with the previous link and the following link *
 * This function communicates with the function "sender" of the server */
struct socks_link
receiver ()
{
	int ip_next_link;
	int sock_sender;
	struct socks_link sockets;
	struct sockaddr_in host_sender;
	struct sockaddr_in host_receiver;
	socklen_t size = sizeof (struct sockaddr);

	ip_next_link = answer ();

	host_sender.sin_family = AF_INET;
	host_sender.sin_port = htons (port_data);
	host_sender.sin_addr.s_addr = INADDR_ANY;

	if ((sock_sender = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
		msgerror (_("Unable to open TCP socket"));
	}

	if ((bind (sock_sender, (struct sockaddr *) &host_sender, size)) < 0)
	{
		msgerror (_("Unable to open TCP port"));
	}

	if ((listen (sock_sender, 1)) < 0)
	{
		msgerror (_("Listen was fail"));
	}

	printf (_("Waiting to carrier...\n"));
	fflush (stdout);
	if ((sockets.fdo =
		 accept (sock_sender, (struct sockaddr *) &host_sender, &size)) < 0)
	{
		msgerror (_("Unable to open connection..."));
	}

	printf (_("Signal was detected from %s\n"),
			inet_ntoa (host_sender.sin_addr));
	fflush (stdout);

	host_receiver.sin_family = AF_INET;
	host_receiver.sin_port = htons (port_data);
	host_receiver.sin_addr.s_addr = ip_next_link;

	if (host_receiver.sin_addr.s_addr)
	{
		if ((sockets.fdd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
		{
			msgerror (_("Unable to open TCP socket"));
		}

		printf ( _("Contacting to %s... "),
				 inet_ntoa (host_receiver.sin_addr));
		fflush (stdout);
		sleep (1);

		if ((connect (sockets.fdd, (struct sockaddr *) &host_receiver, size)) <
			0)
		{
			msgerror (_("Unable to contact the destination"));
		}
		else
		{
			printf (_("OK!\n"));
			fflush (stdout);
		}
	}
	else
	{
		printf (_("This is the last link\n"));
		fflush (stdout);
		sockets.fdd = 0;
	}

	return sockets;
}

/* Contact with the first link. *
 * This function communicates with the function "receiver" of the links */
int
sender (int hosts)
{
	int sock_tcp;
	struct sockaddr_in host_receiver;
	socklen_t size = sizeof (struct sockaddr);

	host_receiver.sin_family = AF_INET;
	host_receiver.sin_port = htons (port_data);
	host_receiver.sin_addr.s_addr = net_scan (hosts);

	if (host_receiver.sin_addr.s_addr == -1)
	{
		msgerror (_("Unable to determine the direction of broadcast, \n \
Please, insert manually with -b option . \n \
In order to know the broadcast direction of this network, execute ifconfig"));
		return -1;
	}

	if ((sock_tcp = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
		msgerror (_("Unable to open TCP socket"));
		return -1;
	}
	printf (_("Contacting to %s... "),
			 inet_ntoa (host_receiver.sin_addr));
	fflush (stdout);
	sleep (1);

	if ((connect (sock_tcp, (struct sockaddr *) &host_receiver, size)) < 0)
	{
		msgerror (_("Unable to contact the destination"));
	}
	else
	{
		printf (_("OK!\n"));
		fflush (stdout);
	}

	return sock_tcp;
}

//This functions are for a tcp direct connection between two machines

/* This function is called for the sender and establishes a connection with *
 * the receiver that should be listening *
 * This function communicates with the function "tcp_server" of the receiver */
int
tcp_client (char *client_addr)
{
	int sock_tcp;
	struct sockaddr_in host_client;
	socklen_t size = sizeof (struct sockaddr);
	struct hostent *dns_name;

	host_client.sin_family = AF_INET;
	host_client.sin_port = htons (port_data);

	dns_name = gethostbyname (client_addr);
	if (dns_name)
	{
		host_client.sin_addr.s_addr =
			inet_addr (inet_ntoa (*(struct in_addr *) dns_name->h_addr));
	}
	else
	{
		host_client.sin_addr.s_addr = inet_addr (client_addr);
	}

	if ((sock_tcp = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
		msgerror (_("Unable to open TCP socket"));
	}
	printf (_("Contacting to %s... "),
			 inet_ntoa (host_client.sin_addr));
	fflush (stdout);

	if ((connect (sock_tcp, (struct sockaddr *) &host_client, size)) < 0)
	{
		msgerror (_("Unable to contact the destination"));
	}
	else
	{
		printf ("OK!\n");
		fflush (stdout);
	}

	return sock_tcp;
}

/* This function is called for the receiver and wait for a connection from the *
 * server. This function communicates with the function "tcp_client" of the 
 * sender*/
int
tcp_server ()
{
	int sock_tcp;
	int sock_return;
	struct sockaddr_in server_host;
	socklen_t size = sizeof (struct sockaddr);

	server_host.sin_family = AF_INET;
	server_host.sin_port = htons (port_data);
	server_host.sin_addr.s_addr = INADDR_ANY;

	if ((sock_tcp = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
		msgerror (_("Unable to open TCP socket"));
	}

	if ((bind (sock_tcp, (struct sockaddr *) &server_host, size)) < 0)
	{
		msgerror (_("Unable to open TCP port"));
	}

	if ((listen (sock_tcp, 1)) < 0)
	{
		msgerror (_("Listen was fail"));
	}

	printf (_("Waiting to carrier...\n"));
	fflush (stdout);
	if ((sock_return =
		 accept (sock_tcp, (struct sockaddr *) &server_host, &size)) < 0)
	{
		msgerror (_("Unable to open conection..."));
	}

	printf (_("Signal was detected from %s\n"),
			inet_ntoa (server_host.sin_addr));
	fflush (stdout);

	return sock_return;
}
