/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* main.c implements the control of program.
 * Include:
 * - Manage the parameters of program.
 * - Show the version of gnetic.
 * - Show help.*/

#include "image.h"
#include "util.h"
#include "slash.h"
#include "net.h"
#include "disk.h"
#include "grub.h"

#define _FILE_OFFSET_BITS 64
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <locale.h>
#include "gettext.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define _VERSION_ "0.4.1"

#define _(x) gettext(x)
#define N_(x) (x)

#define ENCODING "UTF-8"

const char *cmd;

/* This function shows the help */
void
usage (FILE * stream, int code)
{
	fprintf (stream, _("Usage: %s FUNCTION "
					   "[ -d, --device DEVICE ] [ -f, --file FILE ] \n"
					   "\t[ -a, --address CLIENT-IP-ADDRESS ]"
					   "  [ -m, --max-hosts NUMBER ] \n"
					   "\t[ -w, --wait SECONDS ] " "  [ -q, --quick ]\n"
					   "\t[ -b, --broadcast BROADCAST ] "
					   " [ -n, --no-data ]\n "
					   "\t[ -g, --grub_install /BOOT-PARTITION ]\n"), cmd);
	fprintf (stream,
			 _("FUNCTION is made up of one of these specifications:\n\n"
			   "  For local archives: (All these options imply -d and -f)\n"
			   "  -c, --create\t\t	Creates a DNA image.\n"
			   "  -r, --restore\t\t	Restores a DNA image.\n"
			   "\n  For images over the network: "
			   "(All these options imply -d or -f)\n" "  Link mode:\n"
			   "  -s, --sender-mode\t	Sends data to the network. \n"
			   "  -l, --link-mode\t	Receives data from the network.\n"
			   "\n  Point-2-Point:\n"
			   "  -S, --p2p-server\t	Sends server's data "
			   "(This option implies -a).\n"
			   "  -C, --p2p-client\t	Receives data from the server.\n"));
	fprintf (stream,
			 _("\n  Others:\n" "  -h, --help\t		Show this help.\n"
			   "  -v, --version\t		Show gNETic version.\n"));
	exit (code);
}

/* Prints the gnetic version and credits */
void
version ()
{
	printf (_("gNetic %s\n"
			  "Written by Jesús Burgos and Joan Lledó\n\n"
			  "Copyright (C) 2005 Jesús Burgos\n"
			  "gnetic comes with NO WARRANTY,\n"
			  "to the extent permitted by law. \n"
			  "You may redistribute copies of gnetic\n"
			  "under the terms of the GNU General Public License.\n"
			  "For more information about these matters,\n"
			  "see the file named COPYING.\n"), _VERSION_);
	fflush (stdout);
	exit (0);
}

/* This function creates a DNA image */
int
create (char *image, char *device)
{
	struct stat filestat;
	struct params_dev_fd params;

	if (!image || !device)
		usage (stderr, 1);
	if (lstat (device, &filestat) || !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The entered device is not a valid block device"));
		return -1;
	}

	if ((params.sockets.fdd =
		 open (image, O_WRONLY | O_TRUNC | O_CREAT, 0644)) < 0)
	{
		msgerror (_("Error creating the DNA image file"));
		return -1;
	}
	params.sockets.fdo = 0;
	params.devicepath = device;
	if (dev2fd (&params) < 0)
	{

		msgerror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function restores a DNA image */
int
restore (char *image, char *device)
{
	struct stat filestat;
	struct params_dev_fd params;

	if (!image || !device)
		usage (stderr, 1);

	if (dna_check (image))
	{
		msgerror (_("%s is not a valid dna image\n"), image);
		return -1;
	}

	if ((lstat (device, &filestat) != 0 || stat (device, &filestat) != 0)
		&& !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The entered device is not a valid block device"));
		return -1;
	}
	if ((params.sockets.fdo = open (image, O_RDONLY)) < 0)
	{
		msgerror (_("Error creating the DNA image file"));
		return -1;
	}

	params.sockets.fdd = 0;
	params.devicepath = device;
	if (fd2dev (&params) < 0)
	{
		msgerror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function sends an image in link-mode */
int
string_server_image (char *image, int fdd)
{
	int fdo;
	struct stat filestat;

	if (dna_check (image))
	{
		msgerror (_("%s is not a valid dna image\n"));
		return -1;
	}

	if (lstat (image, &filestat) < 0)
	{
		msgerror (_("Unable to get file information"));
		return -1;
	}

	set_image_size (filestat.st_size / 1024);

	if ((fdo = open (image, O_RDONLY)) < 0)
	{
		msgerror (_("Error opening the DNA image file"));
		return -1;
	}
	if ((lsendfile (fdd, fdo, filestat.st_size, 0)) < 0)
	{
		msgerror (_("Error in lsendfile"));
		return -1;
	}

	return 0;
}

/* This function sends a device in link-mode */
int
string_server_device (char *device, int fdd)
{
	struct stat filestat;
	struct params_dev_fd params;

	if ((lstat (device, &filestat) != 0 || stat (device, &filestat) != 0)
		&& !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The device introduced is not a valid block device"));
		return -1;
	}
	params.sockets.fdo = 0;
	params.sockets.fdd = fdd;
	params.devicepath = device;
	if ((dev2fd (&params)) < 0)
	{
		msgerror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function launches the link-mode server */
int
string_server (char *image, char *device, int max_hosts)
{
	int fdd;

	if (!get_wait_seconds ())
		set_wait_seconds (2);

	if ((!image && !device) || (image && device))
		usage (stderr, 1);

	fdd = sender (max_hosts ? max_hosts : 64);

	if (fdd < 0)
	{
		msgerror (_("Error looking for links."));

		return -1;
	}

	if (image)
	{
		if (string_server_image (image, fdd) < 0)
		{
			msgerror (_("Error while sending image."));

			return -1;
		}
	}
	else if (device)
	{
		if (string_server_device (device, fdd) < 0)
		{
			msgerror (_("Error while sending image."));

			return -1;
		}
	}

	return 0;
}

/* This function receives a image in link-mode */
int
string_link_image (struct params_dev_fd *params)
{
	int fdfile;

	/* params->devicepath contains the image path :-P */
	if ((fdfile =
		 open (params->devicepath, O_WRONLY | O_TRUNC | O_CREAT, 0644)) < 0)
	{
		msgerror (_("Error opening the DNA image file"));
		return -1;
	}

	if ((lsendfile (fdfile, params->sockets.fdo, 0, params->sockets.fdd)) < 0)
	{
		msgerror (_("Error in linker"));
		return -1;
	}


	return 0;
}

/* This function receives a device in link-mode */
int
string_link_device (struct params_dev_fd *params)
{
	struct stat filestat;
	if ((lstat (params->devicepath, &filestat) == 0
		 || stat (params->devicepath, &filestat) == 0)
		&& !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The device introduced is not a valid block device"));
		return -1;
	}

	if ((fd2dev (params)) < 0)
	{
		perror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function launches a link-mode client */
int
string_link (char *image, char *device)
{
	struct params_dev_fd params;

	if ((!image && !device) || (image && device))
		usage (stderr, 1);

	set_fd_is_socket ();

	params.sockets = receiver ();

	if (image)
	{
		params.devicepath = image;

		if (string_link_image (&params) < 0)
		{
			msgerror (_("Error while sending image"));

			return -1;
		}
	}
	if (device)
	{
		params.devicepath = device;

		if (string_link_device (&params) < 0)
		{
			msgerror (_("Error while sending device image"));

			return -1;
		}
	}

	return 0;
}

/* This function sends an image in point-to-point-mode */
int
p2p_server_image (char *image, int fdd)
{
	int fdo;
	struct stat filestat;

	if (dna_check (image))
	{
		msgerror (_("%s is not a valid dna image\n"), image);
		return -1;
	}

	if (lstat (image, &filestat) < 0)
	{
		msgerror (_("Unable to get file information"));
		return -1;
	}

	set_image_size (filestat.st_size / 1024);

	if ((fdo = open (image, O_RDONLY)) < 0)
	{
		msgerror (_("Error opening the DNA image file"));
		return -1;
	}
	if ((lsendfile (fdd, fdo, filestat.st_size, 0)) < 0)
	{
		msgerror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function sends an device in point-to-point-mode */
int
p2p_server_device (char *device, int fdd)
{
	struct stat filestat;
	struct params_dev_fd params;

	if ((lstat (device, &filestat) != 0 || stat (device, &filestat) != 0)
		&& !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The device is not a valid block device"));
		return -1;
	}
	params.sockets.fdo = 0;
	params.sockets.fdd = fdd;
	params.devicepath = device;
	if ((dev2fd (&params)) < 0)
	{
		msgerror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function launches a point-to-point-mode server */
int
p2p_server (char *image, char *device, char *address)
{
	int fdd;

	if ((!image && !device) || (image && device) || (!address))
		usage (stderr, 1);

	fdd = tcp_client (address);

	if (image)
	{
		if (p2p_server_image (image, fdd) < 0)
		{
			msgerror (_("Error while sending image"));

			return -1;
		}
	}
	if (device)
	{
		if (p2p_server_device (device, fdd) < 0)
		{
			msgerror (_("Error while sending image"));

			return -1;
		}
	}

	return 0;
}

/* This function receives an image in point-to-point mode */
int
p2p_client_image (char *image, int fdo)
{
	int fdd;

	if ((fdd = open (image, O_WRONLY | O_TRUNC | O_CREAT, 0644)) < 0)
	{
		msgerror (_("Error opening the DNA image file"));
		return -1;
	}
	if ((lsendfile (fdd, fdo, 0, 0)) < 0)
	{
		msgerror (_("Error in linker"));
		return -1;
	}

	return 0;
}

/* This function receives a device in point-to-point mode */
int
p2p_client_device (char *device, int fdo)
{
	struct stat filestat;
	struct params_dev_fd params;

	if ((lstat (device, &filestat) != 0 || stat (device, &filestat) != 0)
		&& !S_ISBLK (filestat.st_mode))
	{
		msgerror (_("The device introduced is not a valid block device"));
		return -1;
	}
	params.sockets.fdo = fdo;
	params.sockets.fdd = 0;
	params.devicepath = device;
	if ((fd2dev (&params)) < 0)
	{
		perror (_("Error: Check file permissions"));
		return -1;
	}

	return 0;
}

/* This function launches a point-to-point-mode client */
int
p2p_client (char *image, char *device)
{
	int fdo;

	if ((!image && !device) || (image && device))
		usage (stderr, 1);

	fdo = tcp_server ();

	set_fd_is_socket ();

	if (image)
	{
		if (p2p_client_image (image, fdo) < 0)
		{
			msgerror (_("Error while sending image"));

			return -1;
		}
	}
	if (device)
	{
		if (p2p_client_device (device, fdo) < 0)
		{
			msgerror (_("Error while sending device image"));

			return -1;
		}
	}

	return 0;
}

/* It read and interpret the options of the command line and execute the *
 * pertinent functions. */
int
main (int argc, char **argv)
{
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	cmd = argv[0];

	enum
	{ none, _create, _restore, _string_server, _string_client, _p2p_server,
		_p2p_client, _grub_install
	} function = none;
	char option;
	int max_hosts = 0;
	char *image = NULL;
	char *device = NULL;
	char *address = NULL;
	char *grub_work_partition = NULL;
	set_quick_mode (0);

	const char options_c[] = "hvcrsld:f:CSa:m:qw:b:ng:";
	const struct option options_l[] = {
		{"help", 0, NULL, 'h'},
		{"version", 0, NULL, 'v'},
		{"create", 0, NULL, 'c'},
		{"restore", 0, NULL, 'r'},
		{"sender-mode", 0, NULL, 's'},
		{"link-mode", 0, NULL, 'l'},
		{"p2p-server", 0, NULL, 'S'},
		{"p2p-client", 0, NULL, 'C'},
		{"device", 1, NULL, 'd'},
		{"file", 1, NULL, 'f'},
		{"address", 1, NULL, 'a'},
		{"max-hosts", 1, NULL, 'm'},
		{"quick", 0, NULL, 'q'},
		{"wait", 1, NULL, 'w'},
		{"broadcast", 1, NULL, 'b'},
		{"no-data", 0, NULL, 'n'},
		{"grub-install", 1, NULL, 'g'},
		{NULL, 0, NULL, 0}
	};

	do
	{							// Process the options
		option = getopt_long (argc, argv, options_c, options_l, NULL);
		switch (option)
		{
		case 'h':
		{
			usage (stdout, 0);
			break;
		}
		case 'v':
		{
			version ();
			break;
		}
		case 'c':
		{
			if (function != none)
				usage (stderr, 1);
			function = _create;
			break;
		}
		case 'r':
		{
			if (function != none)
				usage (stderr, 1);
			function = _restore;
			break;
		}
		case 's':
		{
			if (function != none)
				usage (stderr, 1);
			function = _string_server;
			break;
		}
		case 'l':
		{
			if (function != none)
				usage (stderr, 1);
			function = _string_client;
			break;
		}
		case 'S':
		{
			if (function != none)
				usage (stderr, 1);
			function = _p2p_server;
			break;
		}
		case 'C':
		{
			if (function != none)
				usage (stderr, 1);
			function = _p2p_client;
			break;
		}
		case 'f':
		{
			if (!strrchr (optarg, '/'))	// If it is a relative path
			{
				char tmp[256];
				sprintf (tmp, "./%s", optarg);
				image = tmp;
			}
			else
			{
				image = optarg;
			}
			break;
		}
		case 'd':
		{
			set_disk_type (optarg);
			device = optarg;
			break;
		}
		case 'a':
		{
			address = optarg;
			break;
		}
		case 'm':
		{
			max_hosts = atoi (optarg);
			break;
		}
		case 'q':
		{
			set_quick_mode (1);
			break;
		}
		case 'w':
		{
			if (atoi (optarg))
				set_wait_seconds (atoi (optarg));
			else
				usage (stderr, 1);
			break;
		}
		case 'b':
		{
			set_broadcast_addr (optarg);
			break;
		}
		case 'n':
		{
			set_no_data (true);
			break;
		}
		case 'g':
		{
			grub_work_partition = optarg;
			function = _grub_install;
			break;
		}
		case -1:
			break;
		case '?':
		default:
		{
			usage (stderr, 1);
			break;
		}
		}
	}
	while (option != -1);

	switch (function)
	{
	case _create:
	{
		if (create (image, device) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
	case _restore:
	{
		if (restore (image, device) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
		/* network functions - string mode */
	case _string_server:
	{
		if (string_server (image, device, max_hosts) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
	case _string_client:
	{
		if (string_link (image, device) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
		/* network functions - p2p mode */
	case _p2p_server:
	{
		if (p2p_server (image, device, address) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
	case _p2p_client:
	{
		if (p2p_client (image, device) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
	case _grub_install:
	{
		if (device == NULL)
		{
			msgerror ("You must specify a device with the '-d' option");
			showbt ();
			return -1;
		}

		if (restore_grub (device, grub_work_partition) < 0)
		{
			showbt ();
			return -1;
		}
		break;
	}
	default:
	{
		usage (stderr, 1);
		break;
	}
	}

	printf (_("\nFINISH!\n"));
	fflush (stdout);

	return 0;
}
