/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* disk.c implements functions to interact with the disks, some of these
 * interactions include:
 * - Remove and create partitions
 * - Create file systems, using libparted or the convenient external command
 * - Mount and unmount partitions in /tmp/gnetic-XXXXXX
 * These functions are commonly used by filesystem.c */

#include "util.h"
#include "image.h"
#include "disk.h"
#include "parted_functions.h"
#include "bootsector.h"

#define _FILE_OFFSET_BITS 64
#include <sys/types.h>

#include <stdio.h>
#include <unistd.h>
#include <parted/parted.h>
#include <ext2fs/ext2_fs.h>
#include <ext2fs/ext2fs.h>
#include <uuid/uuid.h>
#include <sys/mount.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <regex.h>
#include <string.h>
#include <mntent.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include <libintl.h>
#include <locale.h>

#define _(x) gettext(x)
#define N_(x) (x)

#define tmp_prefix P_tmpdir "/gnetic-XXXXXX"

struct mount_info
{
	char dir[32];				// Path where is mounted. Empty if not mounted
	enum fs_codes fs_code;		// File System
} temp_mount;

struct
{
	char *ped_name;
	char *lin_name;
	enum fs_codes cod_fs;
	enum fs_types fs_type;
	char *external_cmd;
	char *parameters;
	int parted_csupport;		/* Specifies if libparted has create support
								 * for that file system */
} system_table[] =
{
	// ext2, ext3 and ext4
	{
	"ext2", "ext2", ext2, nodes, NULL, NULL, 1},
	{
	"ext3", "ext3", ext3, nodes, "/sbin/mke2fs", "-t ext3", 0},
	{
	"ext4", "ext4", ext4, nodes, "/sbin/mke2fs", "-t ext4", 0},
		// jfs
	{
	"jfs", "jfs", jfs, nodes, "/sbin/mkfs.jfs", "-q", 0},
		// xfs
	{
	"xfs", "xfs", xfs, nodes, "/sbin/mkfs.xfs", "-f -i size=512", 0},
		// reiserfs
	{
	"reiserfs", "reiserfs", reiserfs, nodes, "/sbin/mkreiserfs", "-f", 0},
		// swap
	{
	"linux-swap", NULL, swap, none, NULL, NULL, 1},
		// hfs, hfs+ and hfsx
	{
	"hfs", "hfs", hfs, nodes, "/usr/bin/hformat", "", 0},
	{
	"hfs+", "hfsplus", hfsp, nodes, "/usr/bin/hformat", "", 0},
//  {"hfsx", "hfsplus"},        ?
		// fat
	{
	"fat16", "vfat", fat16, fat, NULL, NULL, 1},
	{
	"fat32", "vfat", fat32, fat, NULL, NULL, 1},
		// ntfs
	{
	"ntfs", "ntfs", ntfs, fat, "/usr/sbin/mkntfs", "-Q", 0},
/*
  // ufs
  {"sun-ufs", "ufs"},
  {"hp-ufs", "ufs"},
*/
	{
	0, 0, 0, 0, 0, 0, 0}
};

/* returns 0 if libparted can create the file system specified in 'code'. */
int
parted_has_create_support (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (system_table[i].cod_fs == code)
		{
			return system_table[i].parted_csupport;
		}
	}
	return 0;					// is not in the table
}

/* Returns the device length in kilobytes */
off_t
dev_size (PedDevice * dev)
{
	return dev->length * (float) (dev->sector_size / 1024.0);
}

/* Returns the partition length in kilobytes */
off_t
part_size (PedPartition * part)
{
	return part->geom.length * (float) (part->disk->dev->sector_size / 1024.0);
}

int
is_blk (const char *path)
{
	struct stat filestat;
	if (stat (path, &filestat) < 0)
	{
		msgerror (_("Error getting the stat of '%s'"), path);
		return -1;
	}

	return (filestat.st_mode & S_IFMT) == S_IFBLK;
}

/* if the partition with major == major and minor == minor is mounted:
 *   Puts in *puntmontaparti the directory where that partition is mounted and
 *   returns 1.
 * else:
 *   Returns 0 and *puntmontaparti is not modified
 */
int
is_mounted (int major, int minor, char *puntmontaparti)
{
	struct mntent *filesys;
	struct stat dev;
	FILE *f;
	f = setmntent ("/etc/mtab", "r");
	if (!f)
	{
		msgerror (_("Error while opening /etc/mtab"));
		return -1;
	}

	while ((filesys = getmntent (f)))
	{
		if (lstat (filesys->mnt_fsname, &dev) != -1)
		{
			if ((major == (dev.st_rdev >> 8))
				&& (minor == (dev.st_rdev & 0xff)))
			{
				if (puntmontaparti)
					strcpy (puntmontaparti, filesys->mnt_dir);
				return 1;
			}
		}
	}

	endmntent (f);
	return 0;
}

/* Gets the kind of filesystem (enum fs_types) of code */
enum fs_types
get_fs_type (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].cod_fs; i++)
	{
		if (system_table[i].cod_fs == code)
			return system_table[i].fs_type;
	}
	return none;
}

/* Gets the linux equivalent file system name for the parted file system 
 * ped_name */
char *
get_fsname (const char *ped_name)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (strstr (system_table[i].ped_name, ped_name)
			|| strstr (ped_name, system_table[i].ped_name))		
		{
			return system_table[i].lin_name;
		}
	}
	return NULL;
}

/* Gets the linux equivalent file system name for the gnetic fs code */
char *
get_fsname_from_code (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (code == system_table[i].cod_fs)
			return system_table[i].lin_name;
	}
	return NULL;
}

/* Gets the gnetic equivalent file system enum code for the parted file system
 * ped_name */
enum fs_codes
get_fs_code (const char *name)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (strstr (system_table[i].ped_name, name)
				|| strstr (name, system_table[i].ped_name))
		{
			return system_table[i].cod_fs;
		}
	}
	return empty;
}

/* Gets the linux equivalent file system name for the gnetic fs code */
char *
get_pedname_from_code (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (system_table[i].cod_fs == code)
		{
			return system_table[i].ped_name;
		}
	}
	return NULL;
}

/* Gets the external command for format a file system */
char *
get_file_system_external_cmd (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (system_table[i].cod_fs == code)
		{
			return system_table[i].external_cmd;
		}
	}
	return NULL;
}

/* Gets the parameters for a external command for format a file system */
char *
get_file_system_external_cmd_parameter (enum fs_codes code)
{
	int i;
	for (i = 0; system_table[i].lin_name || system_table[i].ped_name; i++)
	{
		if (system_table[i].cod_fs == code)
		{
			return system_table[i].parameters;
		}
	}
	return NULL;
}

/* This function makes sure that all the necessary commands exist */
int
check_for_commands (struct part_info *partitions)
{
	int i;

	for (i = 0; partitions[i].used_part; i++)
	{
		if (!parted_has_create_support (partitions[i].fs_code)
			&& partitions[i].type != extended)
		{
			struct stat filestat;
			char *command_path;

			command_path = get_file_system_external_cmd (partitions[i].fs_code);

			if (command_path == NULL)
			{
				msgerror (_("Error, unknow or empty filesystem"));
				return -1;
			}

			if (stat (command_path, &filestat) < 0)
			{
				msgerror (_("Error, the needed command '%s' not found"),
						  command_path);
				return -1;
			}

			if (partitions[i].fs_code == ntfs)
			{
				strcpy (command_path, "/sbin/mount.ntfs-3g");

				if (stat (command_path, &filestat) < 0)
				{
					msgerror (_("Error, the needed command '%s' not found"),
							  command_path);
					return -1;
				}
			}
		}
		else
		{
			continue;
		}
	}

	return 0;
}

/* Writes in *name the path of the device with major == major and
 * minor == minor. act means if the internal 'cache' will be refreshed */
int
procpt (unsigned char major, unsigned char minor, char *name, int refresh)
{
	struct procpart
	{
		unsigned char major;
		unsigned char minor;
		char name[100];
	};
	static struct procpart parts[64];
	static int i = 0;
	// if (the array is empty || the caller wanted to refresh the part list)
	if (!i || refresh)
	{
		FILE *f;
		char line[100], nompt[100];
		int maj, min, tam;
		if (!(f = fopen ("/proc/partitions", "r")))
		{
			msgerror (_("Error while opening /proc/partitions"));
			return -1;
		}

		while (fgets (line, sizeof (line), f))
		{
			if (sscanf (line, " %d %d %d %[^\n ]", &maj, &min, &tam, nompt) !=
				4)
				continue;
			parts[i].major = maj;
			parts[i].minor = min;
			strcpy (parts[i].name, nompt);
			i++;
		}
		fclose (f);
	}
	int j;
	for (j = 0; j < i; j++)
	{
		if ((parts[j].major == major) && (parts[j].minor == minor))
		{
			strcpy (name, parts[j].name);
			break;
		}
	}
	if (j == i)
		return -1;
	else
		return 0;
}

/* Gets the major of a device */
unsigned char
get_major (const char *dev)
{
	struct stat info_d;
	if (stat (dev, &info_d) < 0)
	{
		msgerror (_("Error while opening the device"));
		return -1;
	}
	return info_d.st_rdev >> 8;
}

/* Gets the minor of a device */
unsigned char
get_minor (const char *dev)
{
	struct stat info_d;
	if (stat (dev, &info_d) < 0)
	{
		msgerror (_("Error while opening the device"));
		return -1;
	}
	return info_d.st_rdev & 0xff;
}

/* Gets a PedPartition type variable from the partition path */
PedPartition *
get_part_from_path (const char *dev_path, int part_num)
{
	PedDevice *dev;
	PedDisk *disk;
	PedPartition *part;

	if (!(dev = ped_device_get (dev_path)))
	{
		msgerror (_("Unable to open device '%s'"), dev_path);
		return NULL;
	}

	if (!(disk = ped_disk_new (dev)))
	{
		msgerror (_("Error while reading partition table on '%s'"), dev_path);
		return NULL;
	}

	if (!(part = ped_disk_get_partition (disk, part_num)))
	{
		msgerror (_
				  ("Error while reading the partition no. %i on '%s'"),
				  part_num, dev_path);
		return NULL;
	}

	return part;
}

/* FIXME: get_part_offset() y get_part_size() make practically the same
 *        work */

/* Returns the offset of the begin of a partition */
unsigned long long
get_part_offset (const char *dev_path, int part_num)
{
	PedPartition *part = get_part_from_path (dev_path, part_num);
	if (!part)
	{
		msgerror (_
				  ("Error getting offset of the partition no. %i at '%s'"),
				  part_num, dev_path);
		return -1;
	}

	return part->geom.start * part->disk->dev->sector_size;
}

/* Returns the size of a partition */
unsigned long long
get_part_size (const char *dev_path, int part_num)
{
	PedPartition *part = get_part_from_path (dev_path, part_num);
	if (!part)
	{
		msgerror (_
				  ("Error getting size of the partition no. %i at '%s'"),
				  part_num, dev_path);
		return -1;
	}

	return part->geom.length * part->disk->dev->sector_size;
}

/* Puts in *disk the path of the disk where 'part' is allowed */
int
get_disk_path_from_part (const char *part, char *disk)
{
	strcpy(disk, "/dev/");
	char *disk_path = (char *) disk + strlen (disk);
	int disk_major, disk_minor;
	if (((disk_major = get_major (part)) < 0)
		|| ((disk_minor = get_minor (part)) < 0))
	{
		msgerror (_("Unable to get partition major/minor"));
		return -1;
	}
	disk_minor -= disk_minor % (disk_max_partitions+1);
	procpt (disk_major, disk_minor, disk_path, 0);
	
	return 0;
}

/* Puts in *part the path of the specified partition in 'disk' */
int
get_part_path_from_disk(const char* disk, int part_num, char *part)
{
	strcpy(part, "/dev/");
	char *part_path = (char *) part + strlen (part);
	if (procpt (get_major (disk), get_minor (disk) + part_num,
				part_path, 0) < 0)
	{
		msgerror (_("Error while getting the partition path"));
		return -1;
	}
	
	return 0;
}

/* Returns the used kilobytes of a partition */
off_t
fs_used_kbytes (const char *dir)
{
	struct statfs info;
	int used_blocks;
	if (statfs (dir, &info) < 0)
	{
		msgerror (_("Error in stat()"));
		return -1;
	}

	used_blocks = info.f_blocks - info.f_bfree;
	return used_blocks * (info.f_bsize / (float) 1024);
}

/* Returns the free kilobytes of a partition */
off_t
fs_free_kbytes (const char *dir)
{
	struct statfs info;
	if (statfs (dir, &info) < 0)
	{
		msgerror (_("Error in stat()"));
		return -1;
	}

	return info.f_bfree * (info.f_bsize / (float) 1024);
}

/* Returns the number of kilobytes used in *partition */
off_t
used_space (PedPartition * partition)
{
	off_t used_kbytes;
	char puntmontaparti[64];
	int part_num;

	part_num = partition->num;

	mount_part_n (partition->disk->dev->path, part_num,
				  get_fs_code (partition->fs_type->name), 0, puntmontaparti);

	used_kbytes = fs_used_kbytes (puntmontaparti);

	if (used_kbytes < 0)
	{
		msgerror (_
				  ("Error while getting the amount of kilobytes used on '%s'"),
				  ped_partition_get_path (partition));
		goto err_close_all;
	}

	umount_part ();
	return used_kbytes;

  err_close_all:
	umount_part ();
	return -1;
}

//These functions read and write the volume label of a partition

/* Read the label of the file system of a partition */
int
read_label (const char *part, enum fs_codes fs_code, unsigned char *label)
{
	int fdr;
	int offset = 0;
	int size = 0;

	memset (label, 0, 28);

	switch (fs_code)
	{
	case ext2:
	case ext3:
	case ext4:
	{
		offset = 0x478;
		size = 16;
		break;
	}
	case jfs:
	{
		offset = 0x8065;
		size = 11;
		break;
	}
		/*case xfs : //Not supported yet
		 * {
		 * offset=0x6c;
		 * size=12;
		 * break;
		 * } */
	case reiserfs:
	{
		offset = 0x10064;
		size = 16;
		break;
	}
	case swap:
	{
		offset = 0x41c;
		size = 15;
		break;
	}
	case hfs:
	case hfsp:
	{
		offset = 0x425;
		size = 27;
		break;
	}
	case fat16:
	{
		offset = 0x2b;
		size = 11;
		break;
	}
	case fat32:
	{
		offset = 0x47;
		size = 11;
		break;
	}
	default:
	{
		msgerror (_("WARNING: Could not determine the filesystem %s"), part);
		return -1;
	}
	}

	if ((fdr = open (part, O_RDONLY)) < 0)
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	if ((lseek (fdr, offset, SEEK_SET) < 0))
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	if ((read (fdr, label, size)) < 0)
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	close (fdr);

	return 0;
}

/* Write the label of the file system of a partition */
int
write_label (const char *part, enum fs_codes fs_code, unsigned char *label)
{
	int fdw;
	int offset = 0;
	int size = 0;

	switch (fs_code)
	{
	case ext2:
	case ext3:
	case ext4:
	{
		offset = 0x478;
		size = 16;
		break;
	}
	case jfs:
	{
		offset = 0x8065;
		size = 11;
		break;
	}
		/*case xfs : //Not supported yet
		 * {
		 * offset=0x6c;
		 * size=12;
		 * break;
		 * } */
	case reiserfs:
	{
		offset = 0x10064;
		size = 16;
		break;
	}
	case swap:
	{
		offset = 0x41c;
		size = 15;
		break;
	}
	case hfs:
	case hfsp:
	{
		offset = 0x425;
		size = 27;
		break;
	}
	case fat16:
	{
		offset = 0x2b;
		size = 11;
		break;
	}
	case fat32:
	{
		offset = 0x47;
		size = 11;
		break;
	}
	default:
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
		return -1;
	}
	}

	if ((fdw = open (part, O_WRONLY)) < 0)
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	if ((lseek (fdw, offset, SEEK_SET) < 0))
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	if ((write (fdw, label, size)) < 0)
	{
		msgerror (_("WARNING: Error writing volume label of %s"), part);
	}

	close (fdw);

	return 0;
}

/* Returns the type of partition (normal, extended, logical) */
PedPartitionType
get_part_type (enum partition_types type)
{
	PedPartitionType part_type;
	switch (type)
	{
	case primary:
		part_type = PED_PARTITION_NORMAL;
		break;
	case extended:
		part_type = PED_PARTITION_EXTENDED;
		break;
	case logical:
		part_type = PED_PARTITION_LOGICAL;
		break;
	}
	
	return part_type;
}

/* creates the specified file system in partition (The file system is specified
 * int the second argument: fs_code) */
int
create_file_system (PedPartition * partition, enum fs_codes fs_code)
{
	// Saber si se puede usar libparted, o es necesario un comando externo
	if (parted_has_create_support (fs_code))
	{
		// Create the partition using libparted
		PedFileSystem *fs;
		if (!(fs = ped_file_system_create (&partition->geom,
										   ped_file_system_type_get
										   (get_pedname_from_code (fs_code)),
										   NULL)))
		{
			msgerror (_("Error creating the file system by using libparted"));
			return -1;
		}
		ped_file_system_close (fs);
	}
	else
	{
		// Create the partition calling a external command
		char cmdline[64];
		char *dev_path = ped_partition_get_path (partition);
		sprintf (cmdline, "%s %s %s >/dev/null 2>&1",
				 get_file_system_external_cmd (fs_code),
				 get_file_system_external_cmd_parameter (fs_code), dev_path);
		if (system (cmdline))
		{
			msgerror (_
					  ("Error creating the file system by using an external "
					   "command: %s, or device is mounted (see /tmp)"),
					  get_file_system_external_cmd (fs_code));
			return -1;
		}
		free (dev_path);
	}
	return 0;
}

/* Write the uuid of a ext2,3 and 4 filesystem */
int
write_ext2_uuid (const char *device, char *uuid)
{
	ext2_filsys fs;
	struct ext2_super_block *sb;
	dgrp_t i;
	int set_csum = 0;

	ext2fs_open2 (device, NULL, EXT2_FLAG_RW | EXT2_FLAG_JOURNAL_DEV_OK,
				  0, 0, unix_io_manager, &fs);

	if (!fs)
	{
		msgerror (_("WARNING: uuid of %s could not be writed"), device);
		ext2fs_close (fs);
		return -1;
	}
	
	sb = fs->super;
	fs->flags &= ~EXT2_FLAG_MASTER_SB_ONLY;
	fs->flags |= EXT2_FLAG_SUPER_ONLY;
	
	if (sb->s_feature_ro_compat &
		EXT4_FEATURE_RO_COMPAT_GDT_CSUM) {
		/*
		 * Determine if the block group checksums are
		 * correct so we know whether or not to set
		 * them later on.
		 */
		for (i = 0; i < fs->group_desc_count; i++)
			if (!ext2fs_group_desc_csum_verify(fs, i))
				break;
		if (i >= fs->group_desc_count)
			set_csum = 1;
	}
	
	if (uuid_parse (uuid, sb->s_uuid)) ; //Write the uuid
	
	if (set_csum)
	{
		for (i = 0; i < fs->group_desc_count; i++)
			ext2fs_group_desc_csum_set(fs, i);
		fs->flags &= ~EXT2_FLAG_SUPER_ONLY;
	}
	
	ext2fs_mark_super_dirty (fs);
	ext2fs_close (fs);

	return 0;
}

/* Write the uuid of a linux-swap partition */
int
write_swap_uuid (const char *device, char *uuid)
{
	int fdw;

	fdw = open (device, O_WRONLY);

	lseek (fdw, 0x40c, SEEK_SET);

	write (fdw, uuid, 16);

	close (fdw);

	return 0;
}

/* Write changes to disk */
int
disk_commit(PedDisk *disk)
{
	while(!ped_disk_commit (disk))
	{
		if (errno=EBUSY) // Device busy, wait a second
		{
			sleep(1);
			continue;
		}
		else
		{
			return 0;
		}
	}
	
	return 0;
}

/* Determine the filesystem and write its uuid */
int
write_uuid (const char *device, char *uuid, enum fs_codes fs_code)
{
	//At the moment, gnetic only knows how to read/write a ext2/3  an swap uuid's.
	switch (fs_code)
	{
	case ext2:
	case ext3:
	case ext4:
	{
		return write_ext2_uuid (device, uuid);
	}
	case swap:
	{
		return write_swap_uuid (device, uuid);
	}
	default:
	{
		msgerror (_("WARNING: Error writing the uuid of %s"), device);
		return -1;
	}
	}
}

/* Creates the specified file system in partition (The file system is specified
 * int the second argument: fs_code) */
int
write_partition (const char *dev_path, struct part_info *part)
{
	char disk_ab_path[128];
	PedDevice *dev;
	PedDisk *disk;
	PedPartition *partition;
	int part_num;
	
	if (get_disk_path_from_part (dev_path, disk_ab_path) < 0)
	{
		msgerror (_
				  ("Error while trying to ascertain the disk where the current"
				   " partition resides"));
		return -1;
	}

	part_num=get_minor (dev_path) % (disk_max_partitions+1);
	
	dev = ped_device_get (disk_ab_path);
	if (!dev)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	disk = ped_disk_new (dev);
	if (!disk)
	{
		msgerror (_("Unable to open disk"));
		return -1;
	}
	
	partition = ped_disk_get_partition (disk, part_num);
	if (!partition)
	{
		msgerror (_("Unable to open partition"));
		return -1;
	}
	
	if (part->fs_code != empty)
	{
		printf (_
			("Formatting %s as %s...\n"), dev_path,
			get_pedname_from_code (part->fs_code));
		fflush(stdout);

		/* Umount partition before format*/
		char cmd[256];
		sprintf(cmd,"umount %s >/dev/null 2>&1", dev_path);
		system(cmd);
		
		/* Formatting */
		if (create_file_system (partition, part->fs_code) < 0)
			return -1;
	}

	// - bootable flag
	if (part->bootable)
	{
		ped_partition_set_flag (partition, PED_PARTITION_BOOT, 1);
	}

	// - lba flag
	if (part->lba)
	{
		ped_partition_set_flag (partition, PED_PARTITION_LBA, 1);
	}

	// - label of filesystem
	write_label (dev_path, part->fs_code, part->label);

	// uuid of filesystem
	write_uuid (dev_path, part->uuid, part->fs_code);

	// Now it is when all the made work is written to the disc
	if (disk_commit(disk)<0)
	{
		msgerror (_("Error commiting to disk"));
		return -1;
	}

	sync ();
	sleep (1);

	return 0;
}

/* Write a partition table on 'dev'.
 * Based on mkpartfs function in "GNU Parted" code. Thanks for all. */
int
write_part_info_on_dev (PedDevice * dev, struct part_info *partitions, const char* disk_path)
{
	PedDisk *disk;
	PedPartition *part;
	PedPartitionType part_type;
	PedFileSystemType *fs_type = ped_file_system_type_get ("ext2");
	PedGeometry *range_start = NULL, *range_end = NULL;
	PedSector start, end, temp = 0;
	PedConstraint *constraint;
	PedSector logical_part_start = 0;
	off_t disk_size_kb = dev_size (dev);
	int i;
	if (!(disk = ped_disk_new (dev)))
		return -1;
	
	system("swapoff -a"); //Disable the swap partitions

	printf (_("Deleting all partitions!...\n"));
	fflush(stdout);
	if ((ped_disk_delete_all (disk)) < 0)
	{
		msgerror (_("Error while deleting all partitions"));
		return -1;
	}

	if (disk_commit (disk) <0)
	{
		msgerror (_("Error commiting to disk"));
		return -1;
	}
	sync ();
	sleep (2);

	for (i = 0; partitions[i].used_part || i < 4; i++)
	{
		if (!partitions[i].used_part)
			continue;
		off_t part_size_kb = partitions[i].used_part * disk_size_kb / 100;
		// set partition and file system type:
		// PED_PARTITION_(NORMAL||EXTENDED||LOGICAL)

		part_type = get_part_type (partitions[i].type);
		if (part_type == PED_PARTITION_LOGICAL)
		{
			temp = logical_part_start;
			logical_part_start =
				temp + ((part_size_kb * 1024) / dev->sector_size);
		}

		if (part_type != PED_PARTITION_EXTENDED)
		{
			if (partitions[i].fs_code == empty)
			{
				msgerror (_("Error, some partition has no file system"));
				return -1;
			}
			else
			{
				fs_type =
				ped_file_system_type_get (get_pedname_from_code
										  (partitions[i].fs_code));
			}
		}

		// set partition bounds
		start = temp;
		end = temp + ((part_size_kb * 1024) / dev->sector_size);
		temp = end;
		/* The limit of the partition not always will be the exact sector that 
		 * we wish, both ranks of down define the margin on which it will be able 
		 * to vary that limit. They are two PedGeometry objects */
		range_start = ped_geometry_new (dev, start, 500);
		if (end + 500 >= dev->length)
			range_end =
				ped_geometry_new (dev, end - 500, dev->length - (end - 500));
		else
			range_end = ped_geometry_new (dev, end, 500);

		// create the PedPartition which will be written to disk
		if (!(part = ped_partition_new (disk, part_type, fs_type, start, end)))
		{
			msgerror (_("Error creating the partition"));
			return -1;
		}

		/* snap_to_boundaries seems verifies that no partition is sly by the new 
		 * one, if outside thus would trim the limit */
		snap_to_boundaries (&part->geom, NULL, disk, range_start, range_end);
		/* FIXME: At the moment ped_file_system_type_get can not fail, but if
		 * system_table[] is modified, get_pedname_from_code() could return NULL
		 */

		// Create constraints
		constraint =
			constraint_intersect_and_destroy
			(ped_file_system_get_create_constraint
			 (fs_type, dev), constraint_from_start_end (dev,
														range_start,
														range_end));
		constraint = constraint_from_start_end (dev, range_start, range_end);
		constraint->min_size =
			(partitions[i].min_size * 1024) / dev->sector_size;
		// Add partition to disk
		printf (_("Creating partition %i...\n"),i+1);
		fflush(stdout);
		if (!ped_disk_add_partition (disk, part, constraint))
		{
			msgerror (_("Error while creating partition"));
			return -1;
		}

		if (part_type == PED_PARTITION_EXTENDED)
			logical_part_start = start;
		if (fs_type != NULL)
		{
			if (!ped_partition_set_system (part, fs_type))
				msgerror (_
						  ("Error while setting system type for this partition"));
		}
	}

	printf (_("Writing to disk...\n"));
	fflush(stdout);

	// Now it is when all the made work is written to the disc
	if (disk_commit (disk)<0)
	{
		msgerror (_("Error commiting to disk"));
		return -1;
	}

	sync ();
	sleep (2);
	/*
	 * if ((i = ioctl(fd, BLKRRPART)) != 0)
	 * {
	 * msgerror(_("Error calling to ioctl()"));
	 * return -1;
	 * }
	 */
	sync ();
	sleep (4);					/* for sync() */

	printf (_("All partitions writed to disk!\n"));
	fflush(stdout);

	for (i = 0; partitions[i].used_part || i < 4; i++)
	{
		if (!partitions[i].used_part)
			continue;
		if (partitions[i].type == extended)
			continue;

		char part_ab_path[128];
		if(get_part_path_from_disk(disk_path, i+1, part_ab_path) <0)
		{
			msgerror(_("Colud not get the partition path"));
			return -1;
		}

		if (write_partition (part_ab_path, &partitions[i]) < 0)
		{
			msgerror (_("Error while creating the file system"));
			return -1;
		}
	}
	
	system("swapon -a"); //Enable the swap partitions

	ped_constraint_destroy (constraint);
	ped_disk_destroy (disk);
	return 1;
}

/* Read the table of partitions saved in image DNA and create a similar one *
 * in the hard disk */
int
write_disk (const char *dev_path, struct part_info partitions[nlogicals + 4])
{
	PedDevice *device;
	off_t disk_size_kb;
	off_t total_parts_kb = 0;
	int i;
	device = ped_device_get (dev_path);
	disk_size_kb = dev_size (device);
	for (i = 0; partitions[i].used_part; i++)
	{
		if (partitions[i].min_size >
			(disk_size_kb * partitions[i].used_part) / 100)
		{
			msgerror (_("There is not enough space on some partition"));
			return -1;
		}
		if (partitions[i].type != extended)
			total_parts_kb += partitions[i].min_size;
	}

	if (total_parts_kb > disk_size_kb)
	{
		msgerror (_("There is not enough space on device"));
		return -1;
	}

	return write_part_info_on_dev (device, partitions, dev_path);
}

int
mount_ntfs (const char *device, char *mount_path)
{
	char command[256];

	sprintf (command, "mount -t ntfs-3g %s %s -o force >/dev/null 2>&1",
			 device, mount_path);

	if (system (command) < 0)
	{
		msgerror (_("Error mounting a ntfs filesystem for write in %s"),
				  device);
		return -1;
	}

	return 0;
}

/* Mount a partition */
int
mount_part (const char *device, enum fs_codes fs,
			unsigned long options, char *mount_path)
{
	if (is_mounted (get_major (device), get_minor (device), mount_path))
	{
		return 0;
	}

	if (temp_mount.dir[0] != '\0')
	{
		fprintf (stderr, _("mounted: %s\n"), temp_mount.dir);
		msgerror (_("There is already one partition mounted temporary"));
		return -1;
	}

	strcpy (mount_path, tmp_prefix);
	if (!mkdtemp (mount_path))
	{
		msgerror (_("Unable to create temporary directory"));
		return -1;
	}

	if (fs == ntfs)
	{
		mount_ntfs (device, mount_path);
	}
	else if (mount (device, mount_path, get_fsname_from_code (fs), options, 0)
			 < 0)
	{
		msgerror (_("Unable to mount partition %s"), device);
		rmdir (mount_path);
		return -1;
	}

	strcpy (temp_mount.dir, mount_path);
	temp_mount.fs_code = fs;
	return 0;
}

/* Mounts the part no. part_num of the disk 'disk' with the file system 'fs'.
 * The mount point will be written on mount_path. */
int
mount_part_n (const char *disk, int part_num, enum fs_codes fs,
			  unsigned long options, char *mount_path)
{
	//Get the partition path
	char part_ab_path[128];
	
	if(get_part_path_from_disk(disk, part_num, part_ab_path) <0)
	{
		msgerror(_("Colud not get the partition path"));
		return -1;
	}

	if (mount_part (part_ab_path, fs, 0, mount_path) < 0)
	{
		msgerror (_("Could not mount partition"));
		return -1;
	}

	return 0;
}

/* Unmounts the partition mounted temporarily by mount_part().
 * NOTE: umount_part will not return -1 when there is no partition mounted
	 temporarily */
int
umount_part ()
{
	if (temp_mount.dir[0] == '\0')
		return 0;				// No partition mounted

	if (temp_mount.fs_code == ntfs)
	{
		char command[256];

		sprintf (command, "umount %s", temp_mount.dir);

		system (command);
		sync ();
	}
	else if (umount (temp_mount.dir) < 0)
	{
		msgerror (_("Unable to unmount the partition"));
		return -1;
	}

	if (rmdir (temp_mount.dir) < 0)
	{
		msgerror (_("WARNING: Error while removing temorary directory"));
	}
	temp_mount.dir[0] = '\0';
	return 0;
}

/* Read the uuid of a ext3 filesystem */
int
read_ext2_uuid (const char *device, char *uuid)
{
	ext2_filsys fs;
	struct ext2_super_block *bs;
	ext2fs_open2 (device, NULL, EXT2_FLAG_RW | EXT2_FLAG_JOURNAL_DEV_OK,
				  0, 0, unix_io_manager, &fs);

	if (!fs)
	{
		msgerror (_("WARNING: uuid of %s could not be readed"), device);
		ext2fs_close (fs);
		return -1;
	}

	bs = fs->super;
	uuid_unparse (bs->s_uuid, uuid);

	ext2fs_close (fs);

	return 0;
}

/* Read the uuid of a linux-swap partition */
int
read_swap_uuid (const char *device, char *uuid)
{
	int fdr;

	fdr = open (device, O_RDONLY);

	lseek (fdr, 0x40c, SEEK_SET);

	read (fdr, uuid, 16);

	close (fdr);

	return 0;
}

/* Determine the filesystem and read its uuid */
int
read_uuid (const char *device, char *uuid, enum fs_codes fs_code)
{
	//At the moment, gnetic only knows how to read/write a ext2/3 and swap uuid's.
	switch (fs_code)
	{
	case ext2:
	case ext3:
	case ext4:
	{
		return read_ext2_uuid (device, uuid);
	}
	case swap:
	{
		return read_swap_uuid (device, uuid);
	}
	default:
	{
		msgerror (_("WARNING: Error reading the uuid of %s"), device);
		return -1;
	}
	}
}

/* Read the characteristics of a table to save it in the dna image. * 
 * This function is just as read_disk but read only one partition */
int
read_partition (const char *dev_path, struct part_info *part)
{
	PedDevice *device = NULL;
	PedDisk *disk = NULL;
	PedPartition *partition = NULL;
	char disk_ab_path[128]; //The (full) disk path
	int major, minor;
	int part_num;

	if (get_disk_path_from_part (dev_path, disk_ab_path) < 0)
	{
		msgerror (_
				  ("Error while trying to ascertain the disk where the current"
				   " partition resides"));
		return -1;
	}

	part_num = get_minor (dev_path) % (disk_max_partitions+1);

	device = ped_device_get (disk_ab_path);
	if (device)
		disk = ped_disk_new (device);
	if (disk)
		partition = ped_disk_get_partition (disk, part_num);
	if (!partition)
	{
		msgerror (_("Error while getting partition information"));
		return -1;
	}

	if (partition->type == PED_PARTITION_EXTENDED)
	{
		msgerror (_
				  ("Unable to make a genetic image from a extended partition"));
		return -1;
	}

	// - fs_code
	if (partition->fs_type)
		part->fs_code = get_fs_code (partition->fs_type->name);
	else
		part->fs_code = empty;
	// - min_size
	switch (part->fs_code)
	{
	case swap:
	case empty:
		msgerror (_("Attempted to make an image from a swap/empty partition"));
		return -1;
		break;
	default:
		part->min_size = used_space (partition);
	}

	// - used_part
	part->used_part =
		((partition->geom.length *
		  (device->sector_size / 1024.0)) / (float) dev_size (device)) * 100;
	/* FLAGS */
	// - bootable
	part->bootable = ped_partition_get_flag (partition, PED_PARTITION_BOOT);
	// - lba
	part->lba = ped_partition_get_flag (partition, PED_PARTITION_LBA);
	/* END FLAGS */

	// - boot sector (bs)
	if (read_bs (dev_path, part->bs) < 0)
	{
		msgerror (_("Error while reading the boot sector from device"));
		return -1;
	}
	// - label
	read_label (dev_path, part->fs_code, part->label);

	// - uuid
	read_uuid (dev_path, part->uuid, part->fs_code);

	return 0;
}

/* Read the partition table to save it in the dna image. This function is *
 * just as read_partition but read a whole disc */
int
read_disk (const char *dev_path, struct part_info parts[nlogicals + 4])
{
	PedDevice *device = NULL;
	PedDisk *disk = NULL;
	PedPartition *partition = NULL;
	int highest_part = 4;
	int i, ext_part = 4;
	device = ped_device_get (dev_path);
	disk = ped_disk_new (device);
	if (!disk)
	{
		msgerror (_("Unable to read partition table"));
		return -1;
	}

	printf (_("Reading partitions...\n"));
	fflush(stdout);

	bzero (parts, sizeof (struct part_info) * 4);
	while ((partition = ped_disk_next_partition (disk, partition)))
	{
		if (ped_partition_is_active (partition))
		{
			if (partition->type == PED_PARTITION_EXTENDED)
				ext_part = partition->num - 1;

			// - type
			switch (partition->type)
			{
			case PED_PARTITION_NORMAL:
				parts[partition->num - 1].type = primary;
				break;
			case PED_PARTITION_EXTENDED:
				parts[partition->num - 1].type = extended;
				break;
			case PED_PARTITION_LOGICAL:
				parts[partition->num - 1].type = logical;
				break;
			}

			// - fs_code
			if (partition->fs_type)
				parts[partition->num - 1].fs_code =
					get_fs_code (partition->fs_type->name);
			else
				parts[partition->num - 1].fs_code = empty;
			// - min_size
			switch (parts[partition->num - 1].fs_code)
			{
			case swap:
			case empty:
			{
				parts[partition->num - 1].min_size = 0;
				break;
			}
			default:
			{
				parts[partition->num - 1].min_size = used_space (partition);
			}
			}

			if (parts[partition->num - 1].min_size < 0)
			{
				msgerror (_("Unable to get the amount of kbytes used"));
				return -1;
			}

			// - used_part
			parts[partition->num - 1].used_part =
				((partition->geom.length *
				  (device->sector_size / 1024.0)) / (float) dev_size (device)) *
				100;
			/* FLAGS */
			// - bootable
			parts[partition->num - 1].bootable =
				ped_partition_get_flag (partition, PED_PARTITION_BOOT);
			// - lba
			parts[partition->num - 1].lba =
				ped_partition_get_flag (partition, PED_PARTITION_LBA);
			/* END FLAGS */
			if (partition->num > highest_part)
			{
				highest_part = partition->num;
			}
			// - mbr
			if (read_bs_part_n (dev_path, partition->num,
								parts[partition->num - 1].bs) < 0)
			{
				msgerror (_("Error while reading boot sector from device"));
				return -1;
			}
			// - label and uuid

			char part_ab_path[256];

			sprintf (part_ab_path, "%s%d", dev_path, partition->num);

			if (partition->type != PED_PARTITION_EXTENDED)
			{
				read_label (part_ab_path, parts[partition->num - 1].fs_code,
							parts[partition->num - 1].label);

				read_uuid (part_ab_path, parts[partition->num - 1].uuid,
						   parts[partition->num - 1].fs_code);
			}

		}
		sleep (1);
	}

	/* Write the limit reference */
	bzero ((void *) &parts[highest_part], sizeof (struct part_info));
	/* The min_size of the extended should be the sum of all its partitions */
	if (ext_part != 4)
	{
		for (i = 4; parts[i].used_part; i++)
		{
			parts[ext_part].min_size += parts[i].min_size;
		}
	}
	return 0;
}

// Determines whether the devica is SATA/SCSI or IDE
int set_disk_type(char *device)
{
	if(is_disk(device)) //For a disk
	{
		PedDevice *dev;
		dev=ped_device_get(device);
		if(!dev)
		{
			msgerror(_("Error while getting device information"));
			return -1;
		}
	
		switch(dev->type)
		{
			case PED_DEVICE_SCSI:
			{
				set_disk_is_sata();
				break;
			}
			case PED_DEVICE_IDE:
			{
				set_disk_is_ide();
				break;
			}
			default:
			{
				msgerror(_("gNetic only works on SCSI, SATA and IDE disks"));
				return -1;
			}
		}
	}
	else // For a partition
	{
		char dev_path[32];
		procpt (get_major(device), get_minor(device), dev_path, 0);
		
		// El següent codi ha estat escrit pel meu gat
		switch(dev_path[0])
		{
			case 's':
			{
				set_disk_is_sata();
				break;
			}
			case 'h':
			{
				set_disk_is_ide();
				break;
			}
			default:
			{
				// By default, the max number of partitions will be 16.
				// Sorry, this switch was writed for my cat.
				set_disk_is_sata();
				msgerror(_("gNetic only works on SCSI, SATA and IDE disks"));
			}
		}
	}
	
	return 0;
}

//Umount all the devices
int umount_all()
{
	FILE *f;
	char line[1024];
	char device_path[100], mount_point[100], type[100], options[512];
	int dump;
	int pass;
	
	if (!(f = fopen ("/proc/mounts", "r")))
	{
		msgerror (_("Error while opening /proc/mounts"));
		return -1;
	}
	
	while (fgets (line, sizeof (line), f)) // For each line in /proc/mounts
	{
		regex_t expr;
		// device path starts with /dev
		regcomp(&expr, "^/dev", REG_EXTENDED);
		int match=regexec(&expr, line, 0, NULL, 0);
		
		if( match==0) //If the line match
		{
			if (sscanf (line, "%s %s %s %s %d %d %[^\n ]", device_path, mount_point, type, options, &dump, &pass) !=6)
			{
				continue;
			}
			
			if(umount(device_path)<0)
			{
				msgerror(_("Error unmounting device: %s"), device_path);
				//return -1;
			}
		}
	}
	
	return 0;
}

//Determines whether a device is a disk or partition
bool is_disk(char *dev)
{
	PedDevice *device;
	PedDisk *disk;
	
	device=ped_device_get(dev);
	disk=ped_disk_new(device);
	
	if(!disk || !strcmp(disk->type->name, "loop"))
	{
		return false;
	}
	else
	{
		return true;
	}
}
