/*
 * `gnetic' is a program to use and enjoy by all alive being
 * Jesús Burgos <jburmac@gmail.com>
 * Joan Lledó <joanlluislledo@gmail.com>
 *
 *  Copyright (C) 2005 Jesús Burgos
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/* bootsector.c implements functions to save and restore the boot sector of 
 * disks and partitions
 */

#include "util.h"
#include "grub.h"
#include "disk.h"
#include "bootsector.h"

#define _FILE_OFFSET_BITS 64
#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include <parted/parted.h>

#include <libintl.h>
#include <locale.h>

#define _(x) gettext(x)
#define N_(x) (x)

/* Restores the boot sector of 'dev' (non-fat or ntfs) */
int
write_bs (const char *dev, unsigned char *ptr)
{
	int fd;
	if ((fd = open (dev, O_WRONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}
	if ((lseek (fd, 0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fd, ptr, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}
	if (close (fd) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

/* This read the number of heads of the disc */
char
find_out_number_of_heads (const char *part)
{
	char device_path[256];
	char ab_path[256];

	get_disk_path_from_part (part, device_path);

	sprintf (ab_path, "/dev/%s", device_path);

	PedDevice *dev = ped_device_get (ab_path);

	return dev->bios_geom.heads;
}

/* Restores the boot sector of a NTFS partition */
int
write_bs_ntfs (const char *dev, unsigned char *ptr)
{
	int fdw;
	int fdr;
	char backup[513];

	char heads = find_out_number_of_heads (dev);

	if ((fdw = open (dev, O_WRONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((fdr = open (dev, O_RDONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((lseek (fdw, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, ptr, 11) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x1A, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &heads, 1) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x48, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x48], 438) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x200, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x200], 7680) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	// Write a backup in the last sector
	if ((lseek (fdr, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (read (fdr, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, -512, SEEK_END)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if (close (fdr) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}

	if (close (fdw) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

/* Restores the boot sector of a FAT32 partition (win95,98,me)*/
int
write_bs_fat32 (const char *dev, unsigned char *ptr)
{
	int fdw;
	int fdr;
	char backup[513];

	if ((fdw = open (dev, O_WRONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((fdr = open (dev, O_RDONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((lseek (fdw, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, ptr, 11) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x52, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x52], 918) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x3f0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x3f0], 528) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	// Write a backup in the 6th sector
	if ((lseek (fdr, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (read (fdr, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0xc00, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if (close (fdr) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}

	if (close (fdw) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

/* Restores the boot sector of a FAT32 partition (winnt,winxp,2000,2003)*/
int
write_bs_fatnt (const char *dev, unsigned char *ptr)
{
	int fdw;
	int fdr;
	char backup[513];

	if ((fdw = open (dev, O_WRONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((fdr = open (dev, O_RDONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((lseek (fdw, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, ptr, 11) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x52, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x52], 918) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x3f0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x3f0], 528) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x1800, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x1800], 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	// Write a backup in the 6th sector
	if ((lseek (fdr, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (read (fdr, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0xc00, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, backup, 512) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if (close (fdr) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}

	if (close (fdw) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

/* Restores the boot sector of a FAT16 partition */
int
write_bs_fat16 (const char *dev, unsigned char *ptr)
{
	int fdw;
	if ((fdw = open (dev, O_WRONLY)) < 0)
	{
		msgerror (_("Unable to open device"));
		return -1;
	}

	if ((lseek (fdw, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, ptr, 11) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if ((lseek (fdw, 0x3e, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (write (fdw, &ptr[0x3e], 450) < 0)
	{
		msgerror (_("Unable to write on device"));
		return -1;
	}

	if (close (fdw) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

// FIXME: The 2 following functions should be cleaned

/* Puts the boot sector of 'dev' in 'ptr' (non-fat) */
int
read_bs (const char *dev, unsigned char *ptr)
{
	int fd;
	if ((fd = open (dev, O_RDONLY)) < 0)
	{
		msgerror (_("Unable to open device '%s'"), dev);
		return -1;
	}
	if ((lseek (fd, 0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if (read (fd, ptr, 8192) < 0)
	{
		msgerror (_("Unable to read from device"));
		return -1;
	}
	if (close (fd) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;
}

/* Puts the boot sector of 'part_num' partition of 'disk_path' in 'ptr' 
 * (non-fat) */
int
read_bs_part_n (const char *disk_path, int part_num, unsigned char *ptr)
{
	int fd;
	if ((fd = open (disk_path, O_RDONLY)) < 0)
	{
		msgerror (_("Unable to open device '%s'"), disk_path);
		return -1;
	}
	if (lseek (fd, get_part_offset (disk_path, part_num), SEEK_SET) < 0)
	{
		msgerror (_("Unable to seek to partition no. %i on %s"),
				  part_num, disk_path);
		goto err_close_all;
	}
	if (read (fd, ptr, 8192) < 0)
	{
		msgerror (_("Unable to read from device"));
		goto err_close_all;
	}
	if (close (fd) < 0)
	{
		msgerror (_("WARNING: Unable to close block device"));
	}
	return 0;

  err_close_all:
	close (fd);
	return -1;
}

/* Restores the lilo MBR (beta)*/
int
install_lilo (char *dev_path, unsigned char *mbr)
{
	int fd;

	if ((fd = open (dev_path, O_WRONLY)) < 0)
	{
		msgerror (_("Error openinig device"));
		return -1;
	}

	if ((lseek (fd, 0x0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}

	if ((write (fd, mbr, 438)) < 0)
	{
		msgerror (_("Error writing the disk Master Boot Record"));
		return -1;
	}

	if ((lseek (fd, 0x1fe, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	char magic[] = { 0x55, 0xaa };
	if ((write (fd, magic, 2)) < 0)
	{
		msgerror (_("Error writing the disk Master Boot Record"));
		return -1;
	}
	close (fd);

	return 0;
}

/* It determines if is a LILO mbr */
bool
is_lilo_mbr (unsigned char *ptr)
{
	char lilo_string[5];
	strncpy (lilo_string, &ptr[0x06], 4);
	lilo_string[4] = '\0';

	if (!strcmp (lilo_string, "LILO"))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/* Saves the master boot record of dev_path */
int
read_disk_mbr (const char *dev_path, unsigned char *mbr)
{
	int fd;

	if ((fd = open (dev_path, O_RDONLY)) < 0)
	{
		msgerror (_("Error openinig device"));
		return -1;
	}
	if ((lseek (fd, 0, SEEK_SET)) < 0)
	{
		msgerror (_("Error reading device"));
		return -1;
	}
	if ((read (fd, mbr, 446)) < 0)
	{
		msgerror (_("Error reading the disk Master Boot Record"));
		return -1;
	}
	close (fd);

	return 0;
}

/* Restores the master boot record of dev_path */
int
write_disk_mbr (const char *dev_path, unsigned char *mbr)
{
	if (info_grub.readed)
	{
		printf (_("\nRestoring GRUB... "));
		fflush (stdout);

		if (install_grub ())
		{
			printf (_("Error!\n"));
			msgerror (_("WARNING: Error restoring GRUB"));
			return -1;
		}
		else
		{
			printf (_("OK\n"));
		}
	}
	else if (is_lilo_mbr (mbr))
	{
		install_lilo (dev_path, mbr);
	}
	else
	{
		int fd;

		if ((fd = open (dev_path, O_WRONLY)) < 0)
		{
			msgerror (_("Error openinig device"));
			return -1;
		}
		if ((lseek (fd, 0, SEEK_SET)) < 0)
		{
			msgerror (_("Error reading device"));
			return -1;
		}
		if ((write (fd, mbr, 446)) < 0)
		{
			msgerror (_("Error writing the disk Master Boot Record"));
			return -1;
		}
		close (fd);
	}

	return 0;
}
