/* output_gstreamer.c - Output module for GStreamer
 *
 * Copyright (C) 2005   Ivo Clarysse
 *
 * This file is part of GMediaRender.
 *
 * GMediaRender is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GMediaRender is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GMediaRender; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA 02110-1301, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <gst/gst.h>

#include "upnp_connmgr.h"

/* returns all factories which have a maximum of maxtemplates GstPadTemplates in direction dir
 */
GList *gst_factories_at_most_templates(GList * factories,
				       GstPadDirection dir,
				       guint maxtemplates)
{
	GList *ret = NULL;

	while (factories) {
		guint count = 0;
		GList *templs =
		    ((GstElementFactory *) factories->data)->padtemplates;

		while (templs) {
			if (GST_PAD_TEMPLATE_DIRECTION(templs->data) ==
			    dir) {
				count++;
			}
			if (count > maxtemplates)
				break;
			templs = g_list_next(templs);
		}
		if (count <= maxtemplates)
			ret = g_list_prepend(ret, factories->data);

		factories = g_list_next(factories);
	}
	return ret;
}

static void scan_caps(const GstCaps * caps)
{
	guint i;

	g_return_if_fail(caps != NULL);

	if (gst_caps_is_any(caps)) {
		return;
	}
	if (gst_caps_is_empty(caps)) {
		return;
	}

	for (i = 0; i < gst_caps_get_size(caps); i++) {
		GstStructure *structure = gst_caps_get_structure(caps, i);
		register_mime_type(gst_structure_get_name(structure));
	}

}


static void scan_pad_templates_info(GstElement * element,
				    GstElementFactory * factory)
{
	const GList *pads;
	GstPadTemplate *padtemplate;

	if (!factory->numpadtemplates) {
		return;
	}

	pads = factory->padtemplates;
	while (pads) {
		padtemplate = (GstPadTemplate *) (pads->data);
		pads = g_list_next(pads);

		if ((padtemplate->direction == GST_PAD_SINK) &&
		    ((padtemplate->presence == GST_PAD_ALWAYS) ||
		     (padtemplate->presence == GST_PAD_SOMETIMES) ||
		     (padtemplate->presence == GST_PAD_REQUEST)) &&
		    (padtemplate->caps)) {
			scan_caps(padtemplate->caps);
		}
	}

}

static void
error_cb(GObject * object, GstObject * source, GError * error,
	 gchar * debug)
{
	gst_element_default_error(object, source, error, debug);
}




static void scan_mime_list(void)
{
	GList *plugins;

	printf("%s:ENTER\n", __FUNCTION__);
	plugins = gst_registry_pool_plugin_list();

	while (plugins) {
		GList *features;
		GstPlugin *plugin;

		plugin = (GstPlugin *) (plugins->data);
		plugins = g_list_next(plugins);

		features = gst_plugin_get_feature_list(plugin);

		while (features) {
			GstPluginFeature *feature;

			feature = GST_PLUGIN_FEATURE(features->data);

			if (GST_IS_ELEMENT_FACTORY(feature)) {
				GstElementFactory *factory;
				GstElement *element;
				factory = GST_ELEMENT_FACTORY(feature);
				element =
				    gst_element_factory_create(factory,
							       NULL);
				if (element) {
					scan_pad_templates_info(element,
								factory);
				}
			}

			features = g_list_next(features);
		}
	}
	printf("%s:LEAVE\n", __FUNCTION__);
}
static GstElement *play;

void output_set_uri(const char *uri)
{
	g_object_set(G_OBJECT(play), "uri", uri, NULL);
}

int output_play(void)
{
	if (gst_element_set_state(play, GST_STATE_PLAYING) ==
	    GST_STATE_FAILURE) {
		return -1;
	} else {
		return 0;
	}

}

int output_stop(void)
{
	if (gst_element_set_state(play, GST_STATE_READY) ==
	    GST_STATE_FAILURE) {
		return -1;
	} else {
		return 0;
	}

}

int output_pause(void)
{
	if (gst_element_set_state(play, GST_STATE_PAUSED) ==
	    GST_STATE_FAILURE) {
		return -1;
	} else {
		return 0;
	}

}



int output_gstreamer_init(int *argc, char ***argv)
{
	gst_init(argc, argv);
	scan_mime_list();

	play = gst_element_factory_make("playbin", "play");
	g_signal_connect(play, "error", G_CALLBACK(error_cb), NULL);
	if (gst_element_set_state(play, GST_STATE_READY) ==
	    GST_STATE_FAILURE) {
		fprintf(stderr,
			"Error: pipeline doesn't want to get ready.\n");
	}

	return 0;
}
