/* upnp.c - Generic UPnP handling routines
 *
 * Copyright (C) 2005   Ivo Clarysse
 *
 * This file is part of GMediaRender.
 *
 * GMediaRender is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GMediaRender is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GMediaRender; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA 02110-1301, USA.
 *
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <upnp/upnp.h>
#include <upnp/upnptools.h>
#include <errno.h>
#include <stdarg.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "webserver.h"
#include "upnp.h"

UpnpDevice_Handle device_handle;

static ithread_mutex_t device_mutex;

static struct device *upnp_device;

static const char *param_datatype_names[] = {
        [DATATYPE_STRING] =     "string",
        [DATATYPE_BOOLEAN] =    "boolean",
        [DATATYPE_I2] =         "i2",
        [DATATYPE_I4] =         "i4",
        [DATATYPE_UI2] =        "ui2",
        [DATATYPE_UI4] =        "ui4",
        [DATATYPE_UNKNOWN] =    NULL
};

static void add_value_attribute(IXML_Document *doc, IXML_Element *parent, char *attrname, char *value)
{
	IXML_Attr *attr;
	IXML_Node *child;
	attr = ixmlDocument_createAttribute(doc, attrname);
	child=ixmlDocument_createTextNode(doc, value);
	ixmlNode_appendChild((IXML_Node *)attr,(IXML_Node *)child);
	ixmlNode_appendChild((IXML_Node *)parent,(IXML_Node *)attr);
}

static void add_value_element(IXML_Document *doc, IXML_Element *parent, char *tagname, char *value)
{
	IXML_Element *top;
	IXML_Node *child;

	top=ixmlDocument_createElement(doc, tagname);
	child=ixmlDocument_createTextNode(doc, value);

	ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)child);

	ixmlNode_appendChild((IXML_Node *)parent,(IXML_Node *)top);
}
static void add_value_element_int(IXML_Document *doc, IXML_Element *parent, char *tagname, int value)
{
	char *buf;

	asprintf(&buf,"%d",value);
	add_value_element(doc, parent, tagname, buf);
	free(buf);
	
}

static IXML_Element *gen_specversion(IXML_Document *doc, int major, int minor)
{
	IXML_Element *top;

	top=ixmlDocument_createElement(doc, "specVersion");

	add_value_element_int(doc, top, "major", major);
	add_value_element_int(doc, top, "minor", minor);

	return top;
}

static IXML_Element *gen_scpd_action(IXML_Document *doc, struct action *act, struct argument **arglist, const char **varnames)
{
	IXML_Element *top;
	IXML_Element *parent,*child;

	top=ixmlDocument_createElement(doc, "action");

	add_value_element(doc, top, "name", (char *)act->action_name);
	if (arglist) {
		struct argument *arg;
		int j;
		parent=ixmlDocument_createElement(doc, "argumentList");
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)parent);
		for(j=0; (arg=arglist[j]); j++) {
			child=ixmlDocument_createElement(doc, "argument");
			ixmlNode_appendChild((IXML_Node *)parent,(IXML_Node *)child);
			add_value_element(doc,child,"name",(char *)arg->name);
			add_value_element(doc,child,"direction",(arg->direction==PARAM_DIR_IN)?"in":"out");
			add_value_element(doc,child,"relatedStateVariable",(char *)varnames[arg->statevar]);
		}
	}
	return top;
}

static IXML_Element *gen_scpd_actionlist(IXML_Document *doc, struct service *srv)
{
	IXML_Element *top;
	IXML_Element *child;
	int i;

	top=ixmlDocument_createElement(doc, "actionList");
	for(i=0; i<srv->command_count; i++) {
		struct action *act;
		struct argument **arglist;
		const char **varnames;
		act=&(srv->actions[i]);
		arglist=srv->action_arguments[i];
		varnames=srv->variable_names;
		if (act) {
			child=gen_scpd_action(doc, act, arglist, varnames);
			ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)child);
		}
	}
	return top;
}

static IXML_Element *gen_scpd_statevar(IXML_Document *doc, const char *name, struct var_meta *meta)
{
	IXML_Element *top,*parent;
	const char **valuelist;
	struct param_range *range;

	valuelist = meta->allowed_values;
	range = meta->allowed_range;

	top=ixmlDocument_createElement(doc, "stateVariable");

	add_value_attribute(doc, top, "sendEvents",(meta->sendevents==SENDEVENT_YES)?"yes":"no");
	add_value_element(doc,top,"name",(char *)name);
	add_value_element(doc,top,"dataType",(char *)param_datatype_names[meta->datatype]);

	if (valuelist) {
		const char *allowed_value;
		int i;
		parent=ixmlDocument_createElement(doc, "allowedValueList");
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)parent);
		for(i=0; (allowed_value=valuelist[i]); i++) {
			add_value_element(doc,parent,"allowedValue",(char *)allowed_value);
		} 
	}
	if (range) {
		parent=ixmlDocument_createElement(doc, "allowedValueRange");
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)parent);
		add_value_element_int(doc,parent,"minimum",range->min);
		add_value_element_int(doc,parent,"maximum",range->max);
		add_value_element_int(doc,parent,"step",range->step);
	}
	return top;
}

static IXML_Element *gen_scpd_servicestatetable(IXML_Document *doc, struct service *srv)
{
	IXML_Element *top;
	IXML_Element *child;
	int i;

	top=ixmlDocument_createElement(doc, "serviceStateTable");
	for(i=0; i<srv->variable_count; i++) {
		struct var_meta *meta = &(srv->variable_meta[i]);
		const char *name = srv->variable_names[i];
		child=gen_scpd_statevar(doc,name,meta);
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)child);
	}
	return top;
}

IXML_Document *generate_scpd(struct service *srv)
{
	IXML_Document *doc;
	IXML_Element *root;
	IXML_Element *child;

	doc = ixmlDocument_createDocument();

	root=ixmlDocument_createElementNS(doc, "urn:schemas-upnp-org:service-1-0","scpd");
	ixmlNode_appendChild((IXML_Node *)doc,(IXML_Node *)root);

	child=gen_specversion(doc,1,0);
	ixmlNode_appendChild((IXML_Node *)root,(IXML_Node *)child);

	child=gen_scpd_actionlist(doc,srv);
	ixmlNode_appendChild((IXML_Node *)root,(IXML_Node *)child);

	child=gen_scpd_servicestatetable(doc,srv);
	ixmlNode_appendChild((IXML_Node *)root,(IXML_Node *)child);
	
	
	return doc;
}

static IXML_Element *gen_desc_iconlist(IXML_Document *doc, struct icon **icons)
{
	IXML_Element *top;
	IXML_Element *parent;
	struct icon *icon_entry;
	int i;

	top=ixmlDocument_createElement(doc, "iconList");

	for (i=0; (icon_entry=icons[i]); i++) {
		parent=ixmlDocument_createElement(doc, "icon");
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)parent);
		add_value_element(doc,parent,"mimetype",(char *)icon_entry->mimetype);
		add_value_element_int(doc,parent,"width",icon_entry->width);
		add_value_element_int(doc,parent,"height",icon_entry->height);
		add_value_element_int(doc,parent,"depth",icon_entry->depth);
		add_value_element(doc,parent,"url",(char *)icon_entry->url);
	}

	return top;
}

static IXML_Element *gen_desc_servicelist(IXML_Document *doc)
{
	int i;
	struct service *srv;
	IXML_Element *top;
	IXML_Element *parent;

	top=ixmlDocument_createElement(doc, "serviceList");

        for (i=0; (srv = upnp_device->services[i]); i++) {
		parent=ixmlDocument_createElement(doc, "service");
		ixmlNode_appendChild((IXML_Node *)top,(IXML_Node *)parent);
		add_value_element(doc,parent,"serviceType",srv->type);
		add_value_element(doc,parent,"serviceId",(char *)srv->service_name);
		add_value_element(doc,parent,"SCPDURL",(char *)srv->scpd_url);
		add_value_element(doc,parent,"controlURL",(char *)srv->control_url);
		add_value_element(doc,parent,"eventSubURL",(char *)srv->event_url);
        }

	return top;
}


IXML_Document *generate_desc(void)
{
	IXML_Document *doc;
	IXML_Element *root;
	IXML_Element *child;
	IXML_Element *parent;

	doc = ixmlDocument_createDocument();

	root=ixmlDocument_createElementNS(doc, "urn:schemas-upnp-org:device-1-0","root");
	ixmlNode_appendChild((IXML_Node *)doc,(IXML_Node *)root);
	child=gen_specversion(doc,1,0);
	ixmlNode_appendChild((IXML_Node *)root,(IXML_Node *)child);
	parent=ixmlDocument_createElement(doc, "device");
	ixmlNode_appendChild((IXML_Node *)root,(IXML_Node *)parent);
	add_value_element(doc,parent,"deviceType",(char *)upnp_device->device_type);
	add_value_element(doc,parent,"friendlyName",(char *)upnp_device->friendly_name);
	add_value_element(doc,parent,"manufacturer",(char *)upnp_device->manufacturer);
	add_value_element(doc,parent,"manufacturerURL",(char *)upnp_device->manufacturer_url);
	add_value_element(doc,parent,"modelDescription",(char *)upnp_device->model_description);
	add_value_element(doc,parent,"modelName",(char *)upnp_device->model_name);
	add_value_element(doc,parent,"modelNumber",(char *)upnp_device->model_number);
	add_value_element(doc,parent,"modelURL",(char *)upnp_device->model_url);
	add_value_element(doc,parent,"serialNumber",(char *)upnp_device->serial_number);
	add_value_element(doc,parent,"UDN",(char *)upnp_device->udn);
	add_value_element(doc,parent,"UPC",(char *)upnp_device->upc);
	if (upnp_device->icons) {
		child=gen_desc_iconlist(doc,upnp_device->icons);
		ixmlNode_appendChild((IXML_Node *)parent,(IXML_Node *)child);
	}
	child=gen_desc_servicelist(doc);
	ixmlNode_appendChild((IXML_Node *)parent,(IXML_Node *)child);
	add_value_element(doc,parent,"presentationURL",(char *)upnp_device->presentation_url);

	return doc;
}

int
upnp_add_response(struct action_event *event, char *key, const char *value)
{
	char *val;
	int res;

	if (event->status)
		return -1;

	val = strdup(value);
	if (val == NULL) {
		/* report memory failure */
		event->status = -1;
		event->request->ActionResult = NULL;
		event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
		strcpy(event->request->ErrStr, strerror(errno));
		return -1;
	}

	res =
	    UpnpAddToActionResponse(&event->request->ActionResult,
				    event->request->ActionName,
				    event->service->type, key, val);
	if (res != UPNP_E_SUCCESS) {
		/* report custom error */
		free(val);
		event->request->ActionResult = NULL;
		event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
		strcpy(event->request->ErrStr, UpnpGetErrorMessage(res));
		return -1;
	}

	return 0;
}

int upnp_append_variable(struct action_event *event,
			 int varnum, char *paramname)
{
	char *value;
	struct service *service = event->service;
	int retval = 0;

	if (varnum >= service->variable_count) {
		upnp_set_error(event, UPNP_E_INTERNAL_ERROR,
			       "Internal Error - illegal variable number %d",
			       varnum);
		return -1;
	}

	ithread_mutex_lock(service->service_mutex);

	value = (char *) service->variable_values[varnum];
	if (value == NULL) {
		upnp_set_error(event, UPNP_E_INTERNAL_ERROR,
			       "Internal Error");
		retval = -1;
	} else {
		retval = upnp_add_response(event, paramname, value);
	}

	ithread_mutex_unlock(service->service_mutex);
	return retval;
}

void
upnp_set_error(struct action_event *event, int error_code,
	       const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	event->status = -1;
	event->request->ActionResult = NULL;
	event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
	vsnprintf(event->request->ErrStr, sizeof(event->request->ErrStr),
		  format, ap);
	va_end(ap);
	fprintf(stderr, "%s: %s\n", __FUNCTION__, event->request->ErrStr);
}


char *upnp_get_string(struct action_event *event, const char *key)
{
	IXML_Node *node;

	node = (IXML_Node *) event->request->ActionRequest;
	if (node == NULL) {
		upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
			       "Invalid action request document");
		return NULL;
	}
	node = ixmlNode_getFirstChild(node);
	if (node == NULL) {
		upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
			       "Invalid action request document");
		return NULL;
	}
	node = ixmlNode_getFirstChild(node);

	for (; node != NULL; node = ixmlNode_getNextSibling(node)) {
		if (strcmp(ixmlNode_getNodeName(node), key) == 0) {
			node = ixmlNode_getFirstChild(node);
			if (node == NULL) {
				/* Are we sure empty arguments are reported like this? */
				return strdup("");
			}
			return strdup(ixmlNode_getNodeValue(node));
		}
	}

	upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
		       "Missing action request argument (%s)", key);
	return NULL;
}

static struct service *find_service(char *service_name)
{
	struct service *event_service;
	int serviceNum = 0;
	while (event_service =
	       upnp_device->services[serviceNum], event_service != NULL) {
		if (strcmp(event_service->service_name, service_name) == 0)
			return event_service;
		serviceNum++;
	}
	return NULL;
}

static struct action *find_action(struct service *event_service,
				  char *action_name)
{
	struct action *event_action;
	int actionNum = 0;
	if (event_service == NULL)
		return NULL;
	while (event_action =
	       &(event_service->actions[actionNum]),
	       event_action->action_name != NULL) {
		if (strcmp(event_action->action_name, action_name) == 0)
			return event_action;
		actionNum++;
	}
	return NULL;
}

static int handle_subscription_request(struct
						Upnp_Subscription_Request
						*sr_event)
{
	struct service *subscription_service;

	printf("Subscription request\n");
	printf("  %s\n", sr_event->UDN);
	printf("  %s\n", sr_event->ServiceId);

	subscription_service = find_service(sr_event->ServiceId);
	if (subscription_service == NULL) {
		fprintf(stderr, "%s: Unknown service '%s'\n", __FUNCTION__,
			sr_event->ServiceId);
		return -1;
	}

	ithread_mutex_lock(&device_mutex);

	UpnpAcceptSubscription(device_handle,
			       sr_event->UDN, sr_event->ServiceId,
			       subscription_service->variable_names,
			       (const char **) (subscription_service->
						variable_values),
			       subscription_service->variable_count,
			       sr_event->Sid);

	ithread_mutex_unlock(&device_mutex);

	return 0;
}

static int handle_action_request(struct Upnp_Action_Request
					  *ar_event)
{
	struct service *event_service;
	struct action *event_action;

	event_service = find_service(ar_event->ServiceID);
	event_action = find_action(event_service, ar_event->ActionName);

	if (event_action == NULL) {
		fprintf(stderr, "Unknown action '%s' for service '%s'\n",
			ar_event->ActionName, ar_event->ServiceID);
		ar_event->ActionResult = NULL;
		ar_event->ErrCode = 401;
		return -1;
	}
	if (event_action->callback) {
		struct action_event event;
		int rc;
		event.request = ar_event;
		event.status = 0;
		event.service = event_service;

		rc = (event_action->callback) (&event);
		if (rc == 0) {
			ar_event->ErrCode = UPNP_E_SUCCESS;
			printf("Action was a success!\n");
		}
		if (ar_event->ActionResult == NULL) {
			ar_event->ActionResult =
			    UpnpMakeActionResponse(ar_event->ActionName,
						   ar_event->ServiceID, 0,
						   NULL);
		}
	} else {
		fprintf(stderr,
			"Got a valid action, but no handler defined (!)\n");
		fprintf(stderr, "  ErrCode:    %d\n", ar_event->ErrCode);
		fprintf(stderr, "  Socket:     %d\n", ar_event->Socket);
		fprintf(stderr, "  ErrStr:     '%s'\n", ar_event->ErrStr);
		fprintf(stderr, "  ActionName: '%s'\n",
			ar_event->ActionName);
		fprintf(stderr, "  DevUDN:     '%s'\n", ar_event->DevUDN);
		fprintf(stderr, "  ServiceID:  '%s'\n",
			ar_event->ServiceID);
		ar_event->ErrCode = UPNP_E_SUCCESS;
	}



	return 0;
}

static int event_handler(Upnp_EventType EventType, void *event,
			    void *Cookie)
{
	switch (EventType) {
	case UPNP_CONTROL_ACTION_REQUEST:
		handle_action_request(event);
		break;
	case UPNP_CONTROL_GET_VAR_REQUEST:
		printf("control get variable request\n");
		break;
	case UPNP_EVENT_SUBSCRIPTION_REQUEST:
		handle_subscription_request(event);
		break;

	default:
		printf("Unknown event type: %d\n", EventType);
		break;
	}
	return 0;
}




int init_upnp(struct device *device_def)
{
	int rc;
	int ret = -1;
	short int port = 0;
	char *ip_address = NULL;
	struct service *srv;
	struct icon *icon_entry;
	IXML_Document *doc;
	char *buf;
	int i;

	upnp_device = device_def;

	/* register icons in web server */
        for (i=0; (icon_entry = upnp_device->icons[i]); i++) {
		webserver_register_file(icon_entry->url);
        }

	/* generate and register service schemas in web server */
        for (i=0; (srv = upnp_device->services[i]); i++) {
		doc = generate_scpd(srv);
       		buf = ixmlDocumenttoString(doc);
		webserver_register_buf(srv->scpd_url,buf);
	}

	rc = UpnpInit(ip_address, port);
	if (UPNP_E_SUCCESS != rc) {
		printf("UpnpInit() Error: %d\n", rc);
		goto upnp_err_out;
	}
	rc = UpnpEnableWebserver(TRUE);
	if (UPNP_E_SUCCESS != rc) {
		printf("UpnpEnableWebServer() Error: %d\n", rc);
		goto upnp_err_out;
	}
	rc = UpnpSetVirtualDirCallbacks(&virtual_dir_callbacks);
	if (UPNP_E_SUCCESS != rc) {
		printf("UpnpSetVirtualDirCallbacks() Error: %d\n", rc);
		goto upnp_err_out;
	}
	rc = UpnpAddVirtualDir("/upnp");
	if (UPNP_E_SUCCESS != rc) {
		printf("UpnpAddVirtualDir() Error: %d\n", rc);
		goto upnp_err_out;
	}

	doc = generate_desc();
       	buf = ixmlDocumenttoString(doc);

	rc = UpnpRegisterRootDevice2(UPNPREG_BUF_DESC,
				     buf, strlen(buf), 1,
				     &event_handler, NULL,
				     &device_handle);
	if (UPNP_E_SUCCESS != rc) {
		printf("UpnpRegisterRootDevice2() Error: %d\n", rc);
		goto upnp_err_out;
	}

	rc = UpnpSendAdvertisement(device_handle, 100);
	if (UPNP_E_SUCCESS != rc) {
		fprintf(stderr, "Error sending advertisements: %d\n", rc);
		goto upnp_err_out;
	}

	return 0;

      upnp_err_out:
	UpnpFinish();
	return ret;
}
