/* contentdir.c - Implementation of UPnP ContentDirectory
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>
#include <stdint.h>		/* Gnulib/C99 */
#include <inttypes.h>		/* ? */
#include "gettext.h"		/* Gnulib/gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "xvasprintf.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "dirname.h"		/* Gnulib */
#include "strbuf.h"
#include "intutil.h"
#include "strutil.h"
#include "gmediaserver.h"
#include "search-parser.h"
#include "search-lexer.h"

extern int yyparse(yyscan_t scanner, SearchCriteriaParseData *data);

void
free_search_criteria(SearchCriteria *criteria)
{
    if (criteria->expr != NULL)
	free_search_expr(criteria->expr);

    free(criteria);
}

void
free_search_expr(SearchExpr *expr)
{
    switch (expr->type) {
    case T_AND:
    case T_OR:
	free_search_expr(expr->u.logical.expr1);
	free_search_expr(expr->u.logical.expr2);
	break;
    case T_EXISTS:
	free(expr->u.exists.property);
	break;
    default:
	free(expr->u.binary.property);
	free(expr->u.binary.value);
	break;
    }

    free(expr);
}

static bool
contentdir_get_search_capabilities(ActionEvent *event)
{
    upnp_add_response(event, "SearchCaps", "");
    return event->status;
}

static bool
contentdir_get_sort_capabilities(ActionEvent *event)
{
    upnp_add_response(event, "SortCaps", "");
    return event->status;
}

static bool
contentdir_get_system_update_id(ActionEvent *event)
{
    upnp_add_response(event, "Id", "0");
    return event->status;
}

static void
append_escaped_xml(StrBuf *out, const char *entity, const char *value)
{
    char *str;

    str = xsgmlescape(value);
    strbuf_appendf(out, "<%s>%s</%s>", entity, value, entity);
    free(str);
}

void
end_result(StrBuf *result)
{
    strbuf_append(result, "</DIDL-Lite>");
}

void
begin_result(StrBuf *result)
{
    strbuf_append(result, 
            "<DIDL-Lite xmlns=\"urn:schemas-upnp-org:metadata-1-0/DIDL-Lite/\""
            " xmlns:dc=\"http://purl.org/dc/elements/1.1/\""
            " xmlns:upnp=\"urn:schemas-upnp-org:metadata-1-0/upnp/\">");
}

static char *
get_entry_property(Entry *entry, char *property)
{
    EntryDetail *detail;

    detail = get_entry_detail(entry, DETAIL_TAG);

    /* XXX: should we allow namespace-less property names,
     * such as "title" rather than "dc:title"?
     * XXX: should we allow attribute properties,
     * such as @refID, @id, @parentID or @protocolInfo?
     */

    if (strcmp(property, "dc:title") == 0) {
	if (detail != NULL && detail->data.tag.title != NULL && strcmp(detail->data.tag.title, "") != 0)
	    return convert_string(detail->data.tag.title);
        return convert_string(entry->name);
    }

    if (strcmp(property, "upnp:class") == 0) {
	if (has_entry_detail(entry, DETAIL_CHILDREN))
	    return convert_string("object.container.storageFolder");
        /*if (has_entry_detail(entry, DETAIL_URL))
            return convert_string("object.item.audioItem.audioBroadcast");*/
	return convert_string("object.item.audioItem.musicTrack");
    }

    if (detail != NULL) {
	if (strcmp(property, "upnp:artist") == 0)
	    return convert_string(detail->data.tag.artist);
	if (strcmp(property, "upnp:album") == 0)
	    return convert_string(detail->data.tag.album);
	if (strcmp(property, "upnp:genre") == 0)
	    return convert_string(detail->data.tag.genre);
    }

    return NULL;
}

static bool
filter_includes(const char *filter, const char *field)
{
    if (strcmp(filter, "*") == 0)
        return true;

    if (string_in_csv(filter, ',', field))
	return true;

    return false;
}

static void
append_entry(StrBuf *result, Entry *entry, const char *filter)
{
    EntryDetail *detail;
    char *value;

    if (has_entry_detail(entry, DETAIL_CHILDREN)) {
        detail = get_entry_detail(entry, DETAIL_CHILDREN);
	strbuf_appendf(result,
		"<container id=\"%" PRIi32 "\" parentID=\"%" PRIi32 "\" restricted=\"true\" childCount=\"%" PRIi32 "\">",
		entry->id,
		entry->parent,
		detail->data.children.count);
    } else {
	strbuf_appendf(result,
		"<item id=\"%" PRIi32 "\" parentID=\"%" PRIi32 "\" restricted=\"true\">",
		entry->id,
		entry->parent);
        if (filter_includes(filter, "res") || filter_includes(filter, "@protocolInfo") || filter_includes(filter, "@size")) {
            detail = get_entry_detail(entry, DETAIL_FILE);
            if (detail != NULL) {
                strbuf_append(result, "<res");
                if (filter_includes(filter, "res") || filter_includes(filter, "@protocolInfo"))
                    strbuf_appendf(result, " protocolInfo=\"http-get:*:%s:*\"", detail->data.file.mime_type);
                /*if (detail->data.file.size != 0)*/
                if (filter_includes(filter, "res") || filter_includes(filter, "@size"))
                    strbuf_appendf(result, " size=\"%" PRIu32 "\"", detail->data.file.size);
                strbuf_appendf(result, ">http://%s:%d/files/%d</res>",
                        UpnpGetServerIpAddress(),
                        UpnpGetServerPort(),
                        entry->id);
            } else {
                detail = get_entry_detail(entry, DETAIL_URL);
                strbuf_append(result, "<res");
                if (filter_includes(filter, "res") || filter_includes(filter, "@protocolInfo"))
                    strbuf_appendf(result, " protocolInfo=\"http-get:*:audio/mpeg:*\""); /* FIXME: application/octet-stream? */
                strbuf_appendf(result, ">http://%s:%d/files/%d</res>",
                        UpnpGetServerIpAddress(),
                        UpnpGetServerPort(),
                        entry->id);
            }
        }
    }

    /* upnp:class is required and cannot be filtered out. */
    value = get_entry_property(entry, "upnp:class");
    strbuf_appendf(result, "<upnp:class>%s</upnp:class>", value);
    free(value);

    /* dc:title is required and cannot be filtered out. */
    value = get_entry_property(entry, "dc:title");
    if (value != NULL)
        append_escaped_xml(result, "dc:title", value);
    free(value);
    /* XXX: dc:creator instead or in combination with upnp:artist? */
    if (filter_includes(filter, "upnp:artist")) {
        value = get_entry_property(entry, "upnp:artist");
        if (value != NULL)
	    append_escaped_xml(result, "upnp:artist", value);
        free(value);
    }
    if (filter_includes(filter, "upnp:album")) {
        value = get_entry_property(entry, "upnp:album");
        if (value != NULL)
	    append_escaped_xml(result, "upnp:album", value);
        free(value);
    }
    if (filter_includes(filter, "upnp:genre")) {
        value = get_entry_property(entry, "upnp:genre");
        if (value != NULL)
	    append_escaped_xml(result, "upnp:genre", value);
        free(value);
    }

    if (has_entry_detail(entry, DETAIL_CHILDREN)) {
	strbuf_append(result, "</container>");
    } else {
	strbuf_append(result, "</item>");
    }
}

static char *
operator_name(int type)
{
    switch (type) {
    case T_AND: return _("Logical and");
    case T_OR: return _("Logical or");
    case T_EXISTS: return _("Exists");
    case T_EQ: return _("Binary =");
    case T_NE: return _("Binary !=");
    case T_LT: return _("Binary <");
    case T_LE: return _("Binary <=");
    case T_GT: return _("Binary >");
    case T_GE: return _("Binary >=");
    case T_CONTAINS: return _("Binary contains");
    case T_DOES_NOT_CONTAIN: return _("Binary doesNotContain");
    case T_DERIVED_FROM: return _("Binary derivedFrom");
    }

    return NULL;
}

static void
dump_search_expr(SearchExpr *e, int indent_size)
{
    char *indent;

    indent = xstrdupn(" ", indent_size);
    if (e == NULL) {
	say(4, _("%sMatch anything\n"), indent);
	free(indent);
	return;
    }

    if (e->type == T_AND || e->type == T_OR) {
	say(4, _("%s%s operator\n"), indent, operator_name(e->type));
	say(4, _("%sExpression 1:\n"), indent);
	dump_search_expr(e->u.logical.expr1, indent_size+2);
	say(4, _("%sExpression 2:\n"), indent);
	dump_search_expr(e->u.logical.expr2, indent_size+2);
    } else if (e->type == T_EXISTS) {
	say(4, _("%s%s operator\n"), indent, operator_name(e->type));
	say(4, _("%sProperty: %s\n"), indent, e->u.exists.property);
	say(4, _("%sExistence: %s\n"), indent, e->u.exists.must_exist ? _("true") : _("false"));
    } else {
	say(4, _("%s%s operator\n"), indent, operator_name(e->type));
	say(4, _("%sProperty: %s\n"), indent, e->u.binary.property);
	say(4, _("%sValue: %s\n"), indent, e->u.binary.value);
    }

    free(indent);
}

static SearchCriteria *
parse_search_criteria(const char *str, const char **error)
{
    SearchCriteriaParseData data;
    yyscan_t scanner;
    YY_BUFFER_STATE buffer;
    int result;

    if (yylex_init(&scanner) != 0) {
	*error = errstr;
	return NULL;
    }
    yyset_extra(&data, scanner);
    buffer = yy_scan_string(str, scanner);
    result = yyparse(scanner, &data);
    if (result != 0) {
	*error = data.error;
	return NULL;
    }
    yy_delete_buffer(buffer, scanner);
    yylex_destroy(scanner);

    return data.criteria;
}

static bool
match_search_expr(SearchExpr *expr, Entry *entry)
{
    char *value;
    bool result;
    
    switch (expr->type) {
    case T_AND:
	return match_search_expr(expr->u.logical.expr1, entry)
		&& match_search_expr(expr->u.logical.expr2, entry);
    case T_OR:
	return match_search_expr(expr->u.logical.expr1, entry)
		|| match_search_expr(expr->u.logical.expr2, entry);
    case T_EXISTS:
        value = get_entry_property(entry, expr->u.exists.property);
        result = (value != NULL) == (expr->u.exists.must_exist);
        free(value);
        return result;
    case T_EQ:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) == 0;
	free(value);
	return result;
    case T_NE:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) != 0;
	free(value);
	return result;
    case T_LT:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) > 0;
	free(value);
	return result;
    case T_LE:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) >= 0;
	free(value);
	return result;
    case T_GT:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) < 0;
	free(value);
	return result;
    case T_GE:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strcmp(value, expr->u.binary.value) <= 0;
	free(value);
	return result;
    case T_CONTAINS:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strstr(value, expr->u.binary.value) != NULL;
	free(value);
	return result;
    case T_DOES_NOT_CONTAIN:
	value = get_entry_property(entry, expr->u.binary.property);
	result = strstr(value, expr->u.binary.value) == NULL;
	free(value);
	return result;
    case T_DERIVED_FROM:
	/* XXX: clean up these comparisons, can be made simpler! */
	if (strcmp(expr->u.binary.value, "object") == 0)
	    return true;
	if (has_entry_detail(entry, DETAIL_CHILDREN)) {
	    if (strcmp(expr->u.binary.value, "object.container") == 0)
		return true;
	    if (strcmp(expr->u.binary.value, "object.container.storageFolder") == 0)
		return true;
	} else {
	    if (strcmp(expr->u.binary.value, "object.item") == 0)
		return true;
	    if (strcmp(expr->u.binary.value, "object.item.audioItem") == 0)
		return true;
	    if (strcmp(expr->u.binary.value, "object.item.audioItem.musicTrack") == 0)
		return true;
	}
	break;
    }

    /* Shouldn't get here */
    return false;
}

static bool
match_search_criteria(SearchCriteria *criteria, Entry *entry)
{
    if (criteria->expr == NULL)
	return true; /* match any */

    return match_search_expr(criteria->expr, entry);
}

static uint32_t
search_container(SearchCriteria *criteria, Entry *entry,
		 uint32_t *skip, uint32_t max_count,
		 const char *filter, StrBuf *result)
{
    EntryDetail *detail;
    uint32_t c;
    uint32_t match_count = 0;

    /* It has already been checked whether entry has children. */
    detail = get_entry_detail(entry, DETAIL_CHILDREN);

    for (c = 0; c < detail->data.children.count; c++) {
	Entry *child;

	child = get_entry_by_id(detail->data.children.list[c]);
	if (match_search_criteria(criteria, child)) {
	    if (*skip > 0) {
		(*skip)--;
	    } else {
	        /*if (result != NULL)*/
                append_entry(result, child, filter);
		match_count++;
		if (match_count > max_count)
		    return match_count;
	    }
	}

	if (has_entry_detail(child, DETAIL_CHILDREN)) {
	    /* Even if max_count is UINT32_MAX (for RequestedCount=0)
	     * we allow it to be decreased. This is because we never
	     * can return more than UINT32_MAX results anyway
	     * (the return value would overflow).
	     */
	    match_count += search_container(criteria, child, skip, max_count-match_count, filter, result);
	    if (match_count > max_count)
		return match_count;
	}
    }

    return match_count;
}

static bool
contentdir_search(ActionEvent *event)
{
    int32_t id;
    uint32_t index;
    uint32_t count;
    uint32_t match_count;
    char *search_criteria;
    char *sort_criteria;
    char *filter;
    const char *error = NULL;
    Entry *entry;
    EntryDetail *detail;
    SearchCriteria *criteria;
    StrBuf *result;

    /* Retrieve arguments */
    id = upnp_get_i4(event, "ContainerID"); /* See comment for ObjectID in browse */
    search_criteria = upnp_get_string(event, "SearchCriteria");
    sort_criteria = upnp_get_string(event, "SortCriteria");
    filter = upnp_get_string(event, "Filter");
    index = upnp_get_ui4(event, "StartingIndex");
    count = upnp_get_ui4(event, "RequestedCount");
    if (!event->status)
	return false;

    /* A RequestedCount of 0 means that all results are to be returned. */
    if (count == 0)
        count = UINT32_MAX;

    /* Check ContainerID argument */
    entry = get_entry_by_id(id);
    if (entry == NULL) {
        upnp_set_error(event, UPNP_CONTENTDIR_E_NO_CONTAINER,
            _("No such container"));
	return false;
    }
    detail = get_entry_detail(entry, DETAIL_CHILDREN);
    if (detail == NULL) {
        upnp_set_error(event, UPNP_CONTENTDIR_E_NO_CONTAINER,
            _("Not a container"));
	return false;
    }

    /* SortCriteria not supported at the moment */
    if (strcmp(sort_criteria, "") != 0) {
        upnp_set_error(event, UPNP_CONTENTDIR_E_BAD_SORT_CRITERIA,
            _("Sorting not supported"));
        return false;
    }

    /* Check SearchCriteria */
    criteria = parse_search_criteria(search_criteria, &error);
    if (criteria == NULL) {
        upnp_set_error(event, UPNP_CONTENTDIR_E_BAD_SORT_CRITERIA,
            _("Invalid search criteria: %s"), error);
	return false;
    }
    if (verbosity >= 4) {
	say(4, _("Search criteria:\n"));
	dump_search_expr(criteria->expr, 2);
    }

    /* Do the actual searching */
    result = strbuf_new();
    begin_result(result);
    match_count = search_container(criteria, entry, &index, count, filter, result);
    end_result(result);

    /* Make response. */
    upnp_add_response(event, "Result", strbuf_buffer(result));
    upnp_add_response(event, "NumberReturned", int32_str(match_count));
    upnp_add_response(event, "TotalMatches", "0");
    upnp_add_response(event, "UpdateID", "0");
    strbuf_free(result);

    return true;
}

static bool
contentdir_browse(ActionEvent *event)
{
    uint32_t index;
    uint32_t count;
    char *flag;
    char *filter;
    int32_t id;
    char *sort_criteria;
    bool metadata;
    Entry *entry;
    StrBuf *result;

    /* Retrieve arguments */
    index = upnp_get_ui4(event, "StartingIndex");
    count = upnp_get_ui4(event, "RequestedCount");
    /* ObjectID is a string according to ContentDir specification, but we use int32. */
    /* XXX: is this OK? maybe we should use a more appropriate error response if not int32. */
    id = upnp_get_i4(event, "ObjectID"); 
    filter = upnp_get_string(event, "Filter");
    flag = upnp_get_string(event, "BrowseFlag");
    sort_criteria = upnp_get_string(event, "SortCriteria");
    if (!event->status)
	return false;

    /* A RequestedCount of 0 means that all results are to be returned. */
    if (count == 0)
        count = UINT32_MAX;

    /* Check arguments */
    if (strcmp(sort_criteria, "") != 0) {
        upnp_set_error(event, UPNP_CONTENTDIR_E_BAD_SORT_CRITERIA,
            _("Sorting not supported"));
        return false;
    }

    if (strcmp(flag, "BrowseMetadata") == 0) {
	if (index != 0) {
	    upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
	        _("StartingIndex must be 0 when BrowseFlag is BrowseMetaData."));
	    return false;
	}
	metadata = true;
    } else if (strcmp(flag, "BrowseDirectChildren") == 0) {
	metadata = false;
    } else {
	upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
	    _("Invalid BrowseFlag argument value (%s)"), quotearg(flag));
	return false;
    }

    if (metadata) {
        entry = get_entry_by_id(id);
	if (entry == NULL) {
            upnp_set_error(event, UPNP_CONTENTDIR_E_NO_OBJECT,
                _("No such object"));
	    return false;
	}

	result = strbuf_new();
	begin_result(result);
	append_entry(result, entry, filter);
	end_result(result);
        upnp_add_response(event, "Result", strbuf_buffer(result));
        upnp_add_response(event, "NumberReturned", "1");
        upnp_add_response(event, "TotalMatches", "1");
        strbuf_free(result);
    } else {
        Entry *entry;
	EntryDetail *detail;
        StrBuf *result;
	uint32_t c;
	uint32_t end_index;
	uint32_t result_count = 0;

        entry = get_entry_by_id(id);
	if (entry == NULL) {
            upnp_set_error(event, UPNP_CONTENTDIR_E_NO_OBJECT,
                _("No such object"));
	    return false;
	}

	detail = get_entry_detail(entry, DETAIL_CHILDREN);
        if (detail == NULL) {
            upnp_set_error(event, UPNP_SOAP_E_INVALID_ARGS,
                _("BrowseDirectChildren only possible on containers"));
            return false;
        }

	result = strbuf_new();
	begin_result(result);
	end_index = detail->data.children.count;
	if (count != UINT32_MAX) /* hmm.. should figure out a better way to detect if index+count overflows (>UINT32_MAX) */
	    end_index = MIN(index+count, end_index);
        for (c = index; c < end_index; c++) {
	    Entry *child = get_entry_by_id(detail->data.children.list[c]);
	    append_entry(result, child, filter);
	    result_count++;
        }
	end_result(result);

        upnp_add_response(event, "Result", strbuf_buffer(result));
        upnp_add_response(event, "NumberReturned", int32_str(result_count));
        upnp_add_response(event, "TotalMatches", int32_str(detail->data.children.count));
        strbuf_free(result);
    }

    upnp_add_response(event, "UpdateID", "0");

    return event->status;
}

/*
ServiceVariable contentdir_service_variables[] = {
  { "A_ARG_TYPE_ObjectID"
  { "A_ARG_TYPE_Result"
//{ "A_ARG_TYPE_SearchCriteria"
  { "A_ARG_TYPE_BrowseFlag"
  { "A_ARG_TYPE_Filter"
  { "A_ARG_TYPE_SortCriteria"
  { "A_ARG_TYPE_Index"
  { "A_ARG_TYPE_Count"
  { "A_ARG_TYPE_UpdateID"
//{ "A_ARG_Type_TransferID"
//{ "A_ARG_Type_TransferLength"
//{ "A_ARG_Type_TransferTotal"
//{ "A_ARG_TYPE_TagValueList"
//{ "A_ARG_TYPE_URI"
  { "SearchCapabilities"
  { "SortCapabilities"
  { "SystemUpdateID"
//{ "ContainerUpdateIDs"
};
*/

ServiceAction contentdir_service_actions[] = {
  { "GetSearchCapabilities", contentdir_get_search_capabilities },
  { "GetSortCapabilities", contentdir_get_sort_capabilities },
  { "GetSystemUpdateID", contentdir_get_system_update_id },
  { "Browse", contentdir_browse },
  { "Search", contentdir_search },
/*{ "CreateObject", contentdir_create_object },*/
/*{ "DestroyObject", contentdir_destroy_object },*/
/*{ "UpdateObject", contentdir_update_object },*/
/*{ "ImportResource", contentdir_import_resource },*/
/*{ "GetTransferProgress", contentdir_get_transfer_progress },*/
/*{ "DeleteResource", contentdir_delete_resource },*/
/*{ "CreateReference", contentdir_create_reference },*/
  { NULL, NULL }
};
