/* metadata.c - Management of file metadata
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>
#include <config.h>
#ifdef HAVE_ID3LIB
#include <id3.h>                /* id3lib */
#undef false /* id3lib bug - it shouldn't mess with false and true! */
#undef true
#endif
#include <stdbool.h>            /* Gnulib, C99 */
#include <stdint.h>		/* Gnulib, C99 */
#include <dirent.h>		/* POSIX */
#include <sys/stat.h>		/* POSIX */
#include <sys/types.h>		/* POSIX */
#include <fcntl.h>		/* POSIX */
#include <unistd.h>		/* POSIX */
#include <ctype.h>		/* C89 */
#include "gettext.h"		/* Gnulib/gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "dirname.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "strbuf.h"
#include "gmediaserver.h"

#define DEFAULT_ENTRIES_SIZE 512
#define ROOT_ENTRY_NAME "(root)"

typedef enum {
    FILE_MP3,
    /*FILE_MP3_ID3,*/
    FILE_WMA,
    FILE_RIFF_WAVE,
    FILE_M4A,
    /*FILE_PLS,*/
    /*FILE_PLAYLIST_M3U,*/
    FILE_UNKNOWN,
} FileType;

static const char *mime_types[] = {
    /* FILE_MP3 */		"audio/mpeg",
    /* FILE_MP3_ID3 */		/*"audio/mpeg",*/
    /* FILE_WMA */		"audio/x-ms-wma",
    /* FILE_RIFF_WAVE */	"audio/x-wav",
    /* FILE_M4A */		"audio/mp4",  
    /* FILE_PLS */		/*"audio/x-scpls",*/
    /* FILE_M3U */		/*"audio/m3u",*/
    /* FILE_UNKNOWN */		"application/octet-stream",
};

static const char *file_type_names[] = {
    "mp3",
    /*"id3mp3",*/
    "wma",
    "wav",
    "m4a",
    /*"pls",*/
    /*"m3u",*/
    "unknown"
};

static Entry *root_entry;
static uint32_t entry_count = 0;
static uint32_t entries_size = 0;
static Entry **entries;
char *file_types = "mp3,wma";
#ifdef HAVE_ID3LIB
bool id3_enabled = true;
static ID3Tag *id3;
#endif

/* Concatenate two file names.
 * An empty path component ("") is equivalent to the current directory,
 * but will not yield "./" when concatenated.
 */
/* XXX: move to support */
static char *
concat_filenames(const char *p1, const char *p2)
{
    size_t l1;
    size_t l2;
    char *out;

    if (strcmp(p1, "") == 0) {
	if (strcmp(p2, "") == 0)
	    return ".";
        return xstrdup(p2);
    }
    if (strcmp(p2, "") == 0)
        return xstrdup(p1);

    l1 = strlen(p1);
    l2 = strlen(p2);

    if (p1[l1-1] == '/')
        l1--;

    out = xmalloc(l1+1+l2+1);
    memcpy(out, p1, l1);
    out[l1] = '/';
    memcpy(out+l1+1, p2, l2+1);
    return out;
}

EntryDetail *
get_entry_detail(Entry *entry, EntryDetailType type)
{
    EntryDetail *d;
    
    for (d = entry->details; d != NULL; d = d->next) {
	if (d->type == type)
	    return d;
    }

    return NULL;
}

bool
has_entry_detail(Entry *entry, EntryDetailType type)
{
    return get_entry_detail(entry, type) != NULL;
}

char *
get_entry_path(Entry *entry)
{
    StrBuf *path;
    char *full_path;
    char *local_path;

    path = strbuf_new();
    while (!has_entry_detail(entry, DETAIL_LOCAL_PATH)) {
	strbuf_prepend(path, entry->filename);
	if (entry->parent != 0)
	    strbuf_prepend(path, "/");
	entry = entries[entry->parent];
    }

    local_path = get_entry_detail(entry, DETAIL_LOCAL_PATH)->data.local_path;
    full_path = concat_filenames(local_path, strbuf_buffer(path));
    strbuf_free(path);
    return full_path;
}

static EntryDetail *
add_entry_detail(Entry *entry, EntryDetailType type)
{
    EntryDetail *detail;

    detail = xmalloc(sizeof(EntryDetail));
    detail->next = entry->details;
    detail->type = type;
    entry->details = detail;

    return detail;
}

static Entry*
make_entry(const char *name, int32_t parent, bool directory, uint32_t size)
{
    Entry *entry;

    entry = xmalloc(sizeof(Entry));
    entry->id = entry_count++;
    entry->parent = parent;
    entry->filename = xstrdup(name);
    entry->mime_type = "";
    entry->size = size;
    entry->details = NULL;

    if (directory) {
	EntryDetail *detail;
	detail = add_entry_detail(entry, DETAIL_CHILDREN);
	detail->data.children.count = 0;
	detail->data.children.list = NULL;
    }

    if (entry->id >= entries_size) {
        entries_size *= 2;
        entries = xrealloc(entries, entries_size * sizeof(Entry *));
    }
    entries[entry->id] = entry;
    
    return entry;
}

Entry *
get_entry_by_id(uint32_t id)
{
    if (id < 0 || id >= entry_count)
        return NULL;

    return entries[id];
}


static bool
attempt_read_at(int fd, size_t size, off_t pos, uint8_t *buf, const char *fullpath)
{
    size_t count;

    if (lseek(fd, pos, SEEK_SET) != pos) {
	warn(_("%s: cannot seek: %s\n"), fullpath, errstr);
	close(fd); /* Ignore errors since we opened for reading */
	return false;
    }

    count = full_read(fd, buf, size);
    if (count < 4) {
	if (errno != 0)
	    warn(_("%s: cannot read: %s\n"), fullpath, errstr);
	close(fd); /* Ignore errors since we opened for reading */
	return false;
    }
    return true;
}

static FileType
check_file_content_type(const char *fullpath)
{
    int fd;
    uint8_t buf[11];

    say(4, _("Check content type of file %s\n"), fullpath);

    fd = open(fullpath, O_RDONLY);
    if (fd < 0) {
	warn(_("%s: cannot open for reading: %s\n"), fullpath, errstr);
	return FILE_UNKNOWN;
    }

    if (!attempt_read_at(fd, 4, 0, buf, fullpath))
	return FILE_UNKNOWN;

    /* MPEG ADTS, layer III, v1 */
    if (buf[0] == 0xFF && (buf[1] & 0xFE) == 0xFA) {
	say(4, _("Matched type MP3 for %s\n"), fullpath);
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_MP3;
    }
    /* MP3 file with ID3 tag */
    if (buf[0] == 'I' && buf[1] == 'D' && buf[2] == '3') {
	say(4, _("Matched type MP3 with ID3 tag for %s\n"), fullpath);
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_MP3/*_ID3*/;
    }
    /* Microsoft ASF */
    if (buf[0] == 0x30 && buf[1] == 0x26 && buf[2] == 0xb2 && buf[3] == 0x75) {
	say(4, _("Matched type WMA for %s\n"), fullpath);
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_WMA;
    }
    /* RIFF (little-endian), WAVE audio */
    if (buf[0] == 'R' && buf[1] == 'I' && buf[2] == 'F' && buf[3] == 'F') {
	if (!attempt_read_at(fd, 4, 8, buf, fullpath))
	    return FILE_UNKNOWN;
	if (buf[0] == 'W' && buf[1] == 'A' && buf[2] == 'V' && buf[3] == 'E') {
	    say(4, _("Matched type RIFF WAVE for %s\n"), fullpath);
	    close(fd); /* Ignore errors since we opened for reading */
	    return FILE_RIFF_WAVE;
	}
    }

    /* ISO Media, MPEG v4 system, iTunes AAC-LC */
    if (!attempt_read_at(fd, 4, 4, buf, fullpath))
	return FILE_UNKNOWN;
    if (buf[0] == 'f' && buf[1] == 't' && buf[2] == 'y' && buf[3] == 'p') {
	if (!attempt_read_at(fd, 4, 8, buf, fullpath))
	    return FILE_UNKNOWN;
	if (buf[0] == 'M' && buf[1] == '4' && buf[2] == 'A') {
	    say(4, _("Matched type MPEG v4 iTunes AAC-LC for %s\n"), fullpath);
	    close(fd); /* Ignore errors since we opened for reading */
	    return FILE_M4A;
	}
    }

#if 0
    /* Playlist (PLS) */
    if (!attempt_read_at(fd, 11, 0, buf, fullpath))
	return FILE_UNKNOWN;
    if (strncasecmp((char *) buf, "[playlist]", 10) == 0 && (buf[10] == '\r' || buf[10] == '\n')) {
	say(4, _("Matched PLS playlist for %s\n"), fullpath);
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_PLS;
    }
#endif
    
    say(4, _("%s: skipping file - unrecognized content type\n"), fullpath);
    close(fd); /* Ignore errors since we opened for reading */
    return FILE_UNKNOWN;
}

#ifdef HAVE_ID3LIB
static char *
get_id3_string(ID3Tag *id3tag, ID3_FrameID frame_id)
{
    ID3Frame *frame;
    ID3Field *field;
    size_t size;
    char *buf;

    frame = ID3Tag_FindFrameWithID(id3tag, frame_id);
    if (frame != NULL) {
	field = ID3Frame_GetField(frame, ID3FN_TEXT);
	if (field != NULL) {
	    size = ID3Field_Size(field);
	    buf = xmalloc(size+1);
	    ID3Field_GetASCII(field, buf, size);
	    buf[size] = '\0';
	    return buf;
	}
    }

    return NULL;
}
#endif

static Entry *
scan_entry(const char *fullpath, const char *name, int32_t parent)
{
    struct stat sb;

    if (stat(fullpath, &sb) < 0) {
        warn(_("%s: cannot stat: %s\n"), fullpath, errstr);
        return NULL;
    }

    if (S_ISDIR(sb.st_mode)) {
        Entry *entry;
	EntryDetail *detail;
	int32_t *children;
	uint32_t child_count;
        struct dirent **dirents;
        int dirent_count;
        int c;

	say(4, _("Scanning directory %s\n"), fullpath);
        dirent_count = scandir(fullpath, &dirents, NULL, NULL);
        if (dirent_count < 0) {
            warn(_("%s: cannot scan directory: %s\n"), fullpath, errstr);
            return NULL;
        }

        entry = make_entry(name, parent, true, 0);
	children = xmalloc(sizeof(int32_t) * dirent_count);

	child_count = 0;
        for (c = 0; c < dirent_count; c++) {
            if (strcmp(dirents[c]->d_name, ".") != 0 && strcmp(dirents[c]->d_name, "..") != 0) {
		Entry *child;
		char *child_path;

		child_path = concat_filenames(fullpath, dirents[c]->d_name);
		child = scan_entry(child_path, dirents[c]->d_name, entry->id);
		if (child != NULL)
		    children[child_count++] = child->id;
		free(child_path);
	    }
        }

	detail = get_entry_detail(entry, DETAIL_CHILDREN);
	detail->data.children.count = child_count;
	detail->data.children.list = xmemdup(children, sizeof(int32_t) * child_count);
	free(children);

        return entry;
    }

    if (S_ISREG(sb.st_mode)) {
	Entry *entry;
	EntryDetail *detail;
	FileType type;

        if (parent == -1) {
            warn(_("%s: root must be a directory\n"), fullpath);
            return NULL;
        }
	type = check_file_content_type(fullpath);
	if (!string_in_csv(file_types, file_type_names[type]))
	    return NULL;

	say(4, _("Adding file %s\n"), fullpath);
	entry = make_entry(name, parent, false, sb.st_size);
	entry->mime_type = mime_types[type];

#ifdef HAVE_ID3LIB
	if (id3_enabled && (type == FILE_MP3/* || type == FILE_MP3_ID3*/)) {
	    ID3Tag_Clear(id3);
	    ID3Tag_Link(id3, fullpath);

	    detail = add_entry_detail(entry, DETAIL_TAG);
	    detail->data.tag.title = get_id3_string(id3, ID3FID_TITLE);
	    detail->data.tag.album = get_id3_string(id3, ID3FID_ALBUM);
	    detail->data.tag.artist = get_id3_string(id3, ID3FID_LEADARTIST);
	    detail->data.tag.genre = get_id3_string(id3, ID3FID_CONTENTTYPE);
	}
#endif
	
        return entry;
    }

    warn(_("%s: skipping file - unsupported file type\n"), fullpath);
    return NULL;
}

bool
init_metadata(char **pathv, int pathc)
{
    entries_size = DEFAULT_ENTRIES_SIZE;
    entries = xmalloc(sizeof(Entry *) * entries_size);

#ifdef HAVE_ID3LIB
    if (id3_enabled)
	id3 = ID3Tag_New();
#endif

    if (pathc > 1) {
	int32_t *children;
	EntryDetail *detail;
	uint32_t c;

	root_entry = make_entry(ROOT_ENTRY_NAME, -1, true, 0);
	children = xmalloc(sizeof(int32_t) * pathc);
	for (c = 0; c < pathc; c++) {
	    Entry *entry;
	    char *name;
	    uint32_t d;

	    name = base_name(pathv[c]);
	    if (*name == '\0' || *name == '/') {
	        name = xstrdup(pathv[c]);
            } else {
	        for (d = strlen(name)-1; name[d] == '/'; d--);
	        name = xstrndup(name, d+1);
            }

	    entry = scan_entry(pathv[c], name, -1);
	    children[c] = entry->id;
	    free(name);
	    detail = add_entry_detail(entry, DETAIL_LOCAL_PATH);
	    detail->data.local_path = xstrdup(pathv[c]);
	}
	detail = get_entry_detail(root_entry, DETAIL_CHILDREN);
	detail->data.children.count = pathc;
	detail->data.children.list = children;
    } else {
	EntryDetail *detail;
	
	root_entry = scan_entry(pathv[0], ROOT_ENTRY_NAME, -1);
	detail = add_entry_detail(root_entry, DETAIL_LOCAL_PATH);
	detail->data.local_path = xstrdup(pathv[0]);
    }

#ifdef HAVE_ID3LIB
    if (id3_enabled)
	ID3Tag_Delete(id3);
#endif

    return root_entry != NULL;
}

void
finish_metadata(void)
{
    uint32_t c;

    for (c = 0; c < entry_count; c++) {
	EntryDetail *d = entries[c]->details;

	while (d != NULL) {
	    EntryDetail *next;

	    if (d->type == DETAIL_CHILDREN) {
		free(d->data.children.list);
	    }
	    else if (d->type == DETAIL_LOCAL_PATH) {
		free(d->data.local_path);
	    }
	    else if (d->type == DETAIL_TAG) {
		free(d->data.tag.title);
		free(d->data.tag.artist);
		free(d->data.tag.album);
		free(d->data.tag.genre);
	    }

	    next = d->next;
	    free(d);
	    d = next;
	}

	free(entries[c]->filename);
        free(entries[c]);
    }

    free(entries);
}
