/* main.c - Main routines for the gmediaserver
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>
#include <upnp.h>		/* libupnp */
#include <upnptools.h>		/* libupnp */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include <getopt.h>		/* Gnulib, GNU */
#include <errno.h>		/* ? */
#include <stdio.h>		/* C89 */
#include <string.h>		/* C89 */
#include <stdlib.h>		/* C89 */
#include <stdarg.h>		/* C89 */
#include <stdbool.h>		/* Gnulib, C99 */
#include <signal.h>		/* ? */
#include "gettext.h"		/* Gnulib/gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "xalloc.h"		/* Gnulib */
#include "version-etc.h"	/* Gnulib */
#include "progname.h"		/* Gnulib */
#include "sig2str.h"		/* Gnulib */
#include "tempfailure.h"
#include "intutil.h"
#include "gmediaserver.h"

enum {
    OPT_DISABLE_ID3,
    OPT_FRIENDLY_NAME,
    OPT_PIDFILE,
    OPT_PORT,
    OPT_NOTIMESTAMP,
    OPT_TIMESTAMP,
    OPT_EXPIRE_TIME,
    OPT_FILE_TYPES,
    OPT_PROFILE,
    OPT_HELP,
    OPT_VERSION,
};

#define SHORT_OPTIONS "bv::i:o:p:"

static struct option long_options[] = {
#ifdef HAVE_ID3LIB
    { "disable-id3", no_argument, NULL, OPT_DISABLE_ID3 },
#endif
    { "friendly-name", required_argument, NULL, OPT_FRIENDLY_NAME },
    { "pid-file", required_argument, NULL, OPT_PIDFILE },
    { "profile", required_argument, NULL, OPT_PROFILE, },
    { "port", required_argument, NULL, 'p' },
    { "output", required_argument, NULL, 'o' },
    { "background", no_argument, NULL, 'b' },
    { "no-timestamp", no_argument, NULL, OPT_NOTIMESTAMP },
    { "timestamp", optional_argument, NULL, OPT_TIMESTAMP },
    { "interface", required_argument, NULL, 'i' },
    { "verbose", optional_argument, NULL, 'v' },
    { "expire-time", required_argument, NULL, OPT_EXPIRE_TIME },
    { "file-types", required_argument, NULL, OPT_FILE_TYPES },
    { "help", no_argument, NULL, OPT_HELP },
    { "version", no_argument, NULL, OPT_VERSION },
    { NULL, 0, NULL, 0 }
};

const char version_etc_copyright[] = "Copyright (C) 2005 Oskar Liljeblad.";

/* XXX: move this to strutil.c or something */
bool
string_in_csv(const char *csv, const char *value)
{
    const char *p0;
    const char *p1;

    for (p0 = csv; (p1 = strchr(p0, ',')) != NULL; p0 = p1+1) {
        if (strncmp(p0, value, p1-p0) == 0)
            return true;
    }
    if (strcmp(p0, value) == 0)
        return true;

    return false;
}

static void
dummy_signal_handler(int signal)
{
    /* No operation */
}

static void
write_pid_file(const char *pidfilename, pid_t pid, bool background)
{
    if (pidfilename != NULL) {
	FILE *pidfh;

	/* If we are going to background, we don't want to log this message
         * to the console. We only want to tell if the pid file failed to
         * write, which is pretty serious though.
         */
	if (!background)
	    say(2, _("Writing pid to pid file `%s'\n"), pidfilename);
	if ((pidfh = fopen(pidfilename, "w")) == NULL)
	    die(_("Cannot open pid file `%s' for writing - %s\n"), pidfilename, errstr);
	fprintf(pidfh, "%d", pid);
	if (fclose(pidfh) != 0)
	    die(_("Cannot close pid file `%s' - %s\n"), pidfilename, errstr);
    }
}

int
main(int argc, char **argv)
{
    struct sigaction signalaction;
    sigset_t signalset;
    char *pidfilename = NULL;
    char *listenip = NULL;
    uint16_t listenport = 0;
    bool background = false;
    bool timestamp_specified = false;
    char *logfilename = NULL;
    char *timestamp_format = NULL;
    uint32_t expire_time;

    set_program_name(argv[0]);

    for (;;) {
    	int c;

        c = getopt_long(argc, argv, SHORT_OPTIONS, long_options, NULL);
        if (c == -1)
            break;

        switch (c) {
        case 'i':
	    if ((listenip = get_ip_by_spec(optarg)) == NULL)
	    	die(_("invalid address or interface name `%s'\n"), optarg);
            break;
        case 'o':
            logfilename = optarg;
            break;
#ifdef HAVE_ID3LIB
	case OPT_DISABLE_ID3:
	    id3_enabled = false;
	    break;
#endif
	case OPT_FRIENDLY_NAME:
	    if (optarg[0] == '\0')
		die(_("friendly name cannot be empty\n"));
	    friendly_name = xstrdup(optarg);
	    break;
	case OPT_FILE_TYPES:
	    file_types = optarg;
	    /* FIXME: check valid */
	    break;
	case OPT_PIDFILE:
	    pidfilename = optarg;
	    break;
	case 'p':
	    if (!parse_uint16(optarg, &listenport))
		die(_("invalid port number `%s'\n"), optarg);
	    break;
        case OPT_EXPIRE_TIME:
            if (!parse_uint32(optarg, &expire_time) || expire_time > INT_MAX)
		die(_("invalid expire time `%s'\n"), optarg);
            ssdp_expire_time = expire_time;
            /* XXX: parse better, allow hours etc? */
            break;
	case OPT_NOTIMESTAMP:
	    timestamp_format = NULL;
	    timestamp_specified = true;
	    break;
	case OPT_TIMESTAMP:
	    timestamp_format = (optarg != NULL ? optarg : DEFAULT_TIMESTAMP_FORMAT);
	    timestamp_specified = true;
	    break;
	case OPT_PROFILE:
	    if (strcmp(optarg, "generic") == 0) {
		/* no operation */
	    } else if (strcmp(optarg, "mp101") == 0) {
	        file_types = "mp3,wma";
	    } else if (strcmp(optarg, "dms1") == 0) {
	        file_types = "mp3,wma,m4a,wav";
		expire_time = 1801;
	    } else {
		die(_("invalid profile name `%s'\n"), optarg);
	    }
	    break;
        case 'b':
	    background = true;
	    break;
        case 'v':
	    if (optarg != NULL) {
		if (optarg[0] == 'v') {
		    for (verbosity = 0; optarg[verbosity] != '\0'; verbosity++) {
			if (optarg[verbosity] != 'v')
			    die(_("invalid verbosity `%s'\n"), optarg);
		    }
		    verbosity++;
		} else {
		    if (!parse_uint32(optarg, &verbosity))
			die(_("invalid verbosity `%s'\n"), optarg);
		}
		if (verbosity > MAX_VERBOSITY)
		    die(_("maximum verbosity is %d\n"), MAX_VERBOSITY);
	    } else {
		verbosity++;
	    }
            break;
        case OPT_HELP:
            printf(_("Usage: %s [OPTIONS]... DIRECTORIES...\n"), argv[0]);
            printf(_("Run the UPnP media server.\n\n"));
	    printf(_("      --friendly-name=NAME   set display name for media server\n"));
#ifdef HAVE_ID3LIB
	    printf(_("      --disable-id3          do not scan files for ID3 information\n"));
#endif
	    printf(_("  -v, --verbose[=LEVEL]      set verbosity level (0-4)\n"));
            printf(_("      --pid-file=FILE        write pid to FILE when up and running\n"));
            printf(_("  -i, --interface=NAME       listen on a specific interface\n"));
            printf(_("  -p, --port=PORT            listen on a specific port\n"));
            printf(_("  -o, --output=LOGFILE       file for logging\n"));
            printf(_("  -b, --background           go to background (detach)\n"));
            printf(_("      --no-timestamp         do not prepend timestamp to log entries\n"));
            printf(_("      --timestamp[=FORMAT]   prepend timestamp with optional time format\n"));
	    printf(_("      --profile=NAME         specify profile (see below)\n"));
            printf(_("      --expire-time=SECONDS  advertisement expire time (default 100)\n"));
            printf(_("      --help                 display this help and exit\n"));
            printf(_("      --version              output version information and exit\n"));
	    printf(_("\nThe directories arguments specify where audio files are located.\n"
		     "See strftime(3) for description of the timestamp format.\n"));
	    printf(_("\nA profile specified with --profile enables settings for a certain device.\n"
		     "The following values are valid for --profile:\n\n"));
	    printf(_("  generic  no special settings\n"
	             "  mp101    Netgear MP101\n"
		     "  dms1     Omnifi DMS1\n"));
            printf(_("\nReport bugs to <%s>.\n"), PACKAGE_BUGREPORT);
            exit(EXIT_SUCCESS);
        case OPT_VERSION:
            version_etc(stdout, NULL, PACKAGE, VERSION, "Oskar Liljeblad", NULL);
            exit(EXIT_SUCCESS);
    	case '?':
	    exit(EXIT_FAILURE);
        }
    }

    if (argc - optind <= 0) {
        warn(_("missing directory argument\n"));
        fprintf(stderr, _("Try `%s --help' for more information.\n"), argv[0]);
        exit(1);
    }
    if (!timestamp_specified && logfilename != NULL)
        timestamp_format = DEFAULT_TIMESTAMP_FORMAT;

    if (background) {
        pid_t pid;

        /*say(2, "Moving process into background...\n");*/
    	if ((pid = fork()) < 0)
    	    die(_("Cannot fork new process - %s\n"), errstr);
        if (pid != 0) {
            /* Write pid before we return control to the shell. */
            write_pid_file(pidfilename, pid, true);
            exit(EXIT_SUCCESS);
        }

        fclose(stdin);	    	/* ignore errors */
	fclose(stdout);     	/* ignore errors */
	fclose(stderr);     	/* ignore errors */
        close(STDIN_FILENO);	/* ignore errors */
        close(STDOUT_FILENO);	/* ignore errors */
        close(STDERR_FILENO);	/* ignore errors */

        if (setsid() < 0)
	    die(_("Cannot create process session - %s\n"), errstr);

	/* XXX: The following is necessary to work around a bug in
	 *      libupnp 1.2.1 - fd 0 must not be closed!
	 */
	if (open("/dev/null", O_RDONLY) < 0)
	    die(_("Cannot open /dev/null - %s\n"), errstr);
    }

    if (!timestamp_specified && logfilename != NULL)
    	timestamp_format = DEFAULT_TIMESTAMP_FORMAT;

    init_logging(logfilename, timestamp_format);

    /* We could write pid before initiating logging too.
     */
    if (!background)
        write_pid_file(pidfilename, getpid(), false);

    signalaction.sa_handler = dummy_signal_handler;
    if (sigemptyset(&signalaction.sa_mask) < 0)
    	die(_("Cannot empty signal set - %s\n"), errstr);
    signalaction.sa_flags = SA_RESTART;
#ifdef HAVE_STRUCT_SIGACTION_SA_RESTORER
    signalaction.sa_restorer = NULL;
#endif
    if (sigaction(SIGINT, &signalaction, NULL) < 0)
    	die(_("Cannot register signal handler - %s\n"), errstr);
    if (sigaction(SIGTERM, &signalaction, NULL) < 0)
    	die(_("Cannot register signal handler - %s\n"), errstr);
    if (sigaction(SIGUSR1, &signalaction, NULL) < 0)
    	die(_("Cannot register signal handler - %s\n"), errstr);

    if (sigemptyset(&signalset) < 0)
    	die(_("Cannot empty signal set - %s\n"), errstr);
    if (sigaddset(&signalset, SIGTERM) < 0)
    	die(_("Cannot add signal to set - %s\n"), errstr);
    if (sigaddset(&signalset, SIGINT) < 0)
    	die(_("Cannot add signal to set - %s\n"), errstr);
    if (sigaddset(&signalset, SIGUSR1) < 0)
    	die(_("Cannot add signal to set - %s\n"), errstr);
    if (sigprocmask(SIG_BLOCK, &signalset, NULL) < 0)
    	die(_("Cannot block signals - %s\n"), errstr);

    if (!init_metadata(argv+optind, argc-optind)) {
        finish_logging(false);
	exit(EXIT_FAILURE);
    }

    init_upnp(listenip, listenport);

    while (true) {
    	int sig;
    	char signame[SIG2STR_MAX];
      	
      	if ((sig = TEMP_FAILURE_RETRY(sigwaitinfo(&signalset, NULL))) < 0)
    	    die(_("sigwaitinfo failed - %s\n"), errstr);
        if (sig2str(sig, signame) == 0)
      	    say(2, _("Received %s signal\n"), signame);
      	else
      	    say(2, _("Received signal %d\n"), sig);
      	if (sig == SIGUSR1) {
      	    /*say(1, "Reloading configuration file...\n");*/ /* XXX: not implemented */
      	}
      	if (sig == SIGTERM || sig == SIGINT)
            break;
    }
    
    finish_upnp();
    finish_metadata();

    if (pidfilename != NULL)
    	unlink(pidfilename);	/* ignore errors */

    finish_logging(true);

    exit(EXIT_SUCCESS);
}
