/* webserver.c - Serving files through the libupnp built in web server
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <sys/types.h>		/* POSIX */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include "gettext.h"            /* Gnulib/gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "xalloc.h"
#include "intutil.h"
#include "gmediaserver.h"
#include "schemas/ConnectionManager.h"
#include "schemas/ContentDirectory.h"

typedef struct {
    char *fullpath;
    size_t pos;
    enum {
	FILE_LOCAL,
	FILE_MEMORY,
    } type;
    union {
	struct {
	    int fd;
	    Entry *entry;
	} local;
	struct {
	    const char *contents;
	    size_t len;
	} memory;
    } detail;
} WebServerFile;

static Entry *
get_file_entry(const char *filename)
{
    int32_t id;

    if (filename[0] != '/')
	return NULL;
    filename = strchr(filename+1, '/');
    if (filename == NULL)
	return NULL;
    filename++;
    if (filename[0] == '\0')
	return NULL;
    if (!parse_int32(filename, &id))
	return NULL;

    return get_entry_by_id(id);
}

static int
webserver_get_info(const char *filename, struct File_Info *info)
{
    Entry *entry;
    char *fullpath;
    struct stat sb;

    say(4, _("Request file information for %s\n"), filename);

    if (strcmp(filename, "/upnp/ContentDirectory.xml") == 0) {
	info->file_length = CONTENTDIRECTORY_DESC_LEN;
	info->last_modified = 0;
	info->is_directory = 0;
	info->is_readable = 1;
	info->content_type = ixmlCloneDOMString("text/xml");
	return 0;
    }
    if (strcmp(filename, "/upnp/ConnectionManager.xml") == 0) {
	info->file_length = CONNECTIONMANAGER_DESC_LEN;
	info->last_modified = 0;
	info->is_directory = 0;
	info->is_readable = 1;
	info->content_type = ixmlCloneDOMString("text/xml");
	return 0;
    }

    entry = get_file_entry(filename);
    if (entry == NULL)
	return -1;

    fullpath = get_entry_path(entry);
    if (stat(fullpath, &sb) < 0) {
        warn(_("%s: cannot stat: %s\n"), fullpath, errstr);
	free(fullpath);
	return -1;
    }

    if (access(fullpath, R_OK) < 0) {
	if (errno != EACCES) {
	    warn(_("%s: cannot check access: %s\n"), fullpath, errstr);
	    free(fullpath);
	    return -1;
	}
	info->is_readable = 0;
    } else {
	info->is_readable = 1;
    }
    info->file_length = sb.st_size;
    info->last_modified = sb.st_mtime;
    info->is_directory = S_ISDIR(sb.st_mode);
    if (info->is_directory) {
	/* XXX: not really sure what content type to set for directories */
	info->content_type = ixmlCloneDOMString("");
    } else {
	/* XXX: safe guess always?! */
	info->content_type = ixmlCloneDOMString("audio/mpeg");
    }

    say(4, _("Returned info for file %s successfully\n"), fullpath);

    return 0;
}

static UpnpWebFileHandle
webserver_open(const char *filename, enum UpnpOpenFileMode mode)
{
    Entry *entry;
    int fd;
    char *fullpath;
    WebServerFile *file;

    if (mode != UPNP_READ) {
	warn(_("%s: ignoring request to open file for writing\n"), filename);
	return NULL;
    }

    if (strcmp(filename, "/upnp/ContentDirectory.xml") == 0) {
	file = xmalloc(sizeof(WebServerFile));
	file->fullpath = xstrdup("ContentDirectory.xml in memory");
	file->pos = 0;
	file->type = FILE_MEMORY;
	file->detail.memory.contents = CONTENTDIRECTORY_DESC;
	file->detail.memory.len = CONTENTDIRECTORY_DESC_LEN;
	return file;
    }
    if (strcmp(filename, "/upnp/ConnectionManager.xml") == 0) {
	file = xmalloc(sizeof(WebServerFile));
	file->fullpath = xstrdup("ConnectionManager.xml in memory");
	file->pos = 0;
	file->type = FILE_MEMORY;
	file->detail.memory.contents = CONNECTIONMANAGER_DESC;
	file->detail.memory.len = CONNECTIONMANAGER_DESC_LEN;
	return file;
    }

    entry = get_file_entry(filename);
    if (entry == NULL)
	return NULL;

    fullpath = get_entry_path(entry);

    fd = open(fullpath, O_RDONLY);
    if (fd < 0) {
	warn(_("%s: cannot open for reading: %s\n"), fullpath, errstr);
	free(fullpath);
	return NULL;
    }

    file = xmalloc(sizeof(WebServerFile));
    file->fullpath = fullpath;
    file->pos = 0;
    file->type = FILE_LOCAL;
    file->detail.local.entry = entry;
    file->detail.local.fd = fd;

    return (UpnpWebFileHandle) file;
}

static int
webserver_read(UpnpWebFileHandle fh, char *buf, size_t buflen)
{
    WebServerFile *file = (WebServerFile *) fh;
    ssize_t len = -1;

    say(4, _("Attempting to read %d bytes at %d from %s\n"), buflen, file->pos, file->fullpath);

    switch (file->type) {
    case FILE_LOCAL:
	len = read(file->detail.local.fd, buf, buflen);
	break;
    case FILE_MEMORY:
	len = min(buflen, file->detail.memory.len - file->pos);
	memcpy(buf, file->detail.memory.contents + file->pos, len);
	break;
    }

    if (len < 0)
	warn(_("%s: cannot read: %s\n"), file->fullpath, errstr);
    else
	file->pos += len;

    return len;
}

static int
webserver_write(UpnpWebFileHandle fh, char *buf, size_t buflen)
{
    WebServerFile *file = (WebServerFile *) fh;

    warn(_("%s: ignoring request to write to file\n"), file->fullpath);

    return -1;
}

static int
webserver_seek(UpnpWebFileHandle fh, long offset, int origin)
{
    WebServerFile *file = (WebServerFile *) fh;
    long newpos = -1;

    switch (origin) {
    case SEEK_SET:
	say(4, _("Attempting to seek to %ld (was at %d) in %s\n"), offset, file->pos, file->fullpath);
	newpos = offset;
	break;
    case SEEK_CUR:
	say(4, _("Attempting to seek by %ld from %d in %s\n"), offset, file->pos, file->fullpath);
	newpos = file->pos + offset;
	break;
    case SEEK_END:
	say(4, _("Attempting to seek by %ld from end (was at %d) in %s\n"), offset, file->pos, file->fullpath);
	if (file->type == FILE_LOCAL) {
	    struct stat sb;
	    if (stat(file->fullpath, &sb) < 0) {
		warn(_("%s: cannot stat: %s\n"), file->fullpath, errstr);
		return -1;
	    }
	    newpos = sb.st_size + offset;
	} else if (file->type == FILE_MEMORY) {
	    newpos = file->detail.memory.len + offset;
	}
	break;
    }

    switch (file->type) {
    case FILE_LOCAL:
	/* Just make sure we cannot seek before start of file. */
	if (newpos < 0) {
	    warn(_("%s: cannot seek: %s\n"), file->fullpath, strerror(EINVAL));
	    return -1;
	}
	/* Don't seek with origin as specified above, as file may have
	 * changed in size since our last stat.
	 */
	if (lseek(file->detail.local.fd, newpos, SEEK_SET) == -1) {
	    warn(_("%s: cannot seek: %s\n"), file->fullpath, errstr);
	    return -1;
	}
	break;
    case FILE_MEMORY:
	if (newpos < 0 || newpos > file->detail.memory.len) {
	    warn(_("%s: cannot seek: %s\n"), file->fullpath, strerror(EINVAL));
	    return -1;
	}
	break;
    }

    file->pos = newpos;
    return 0;
}

static int
webserver_close(UpnpWebFileHandle fh)
{
    WebServerFile *file = (WebServerFile *) fh;

    switch (file->type) {
    case FILE_LOCAL:
	close(file->detail.local.fd); /* Ignore errors since only reading */
	break;
    case FILE_MEMORY:
	/* no operation */
	break;
    }
    free(file->fullpath);
    free(file);

    return 0;
}

struct UpnpVirtualDirCallbacks virtual_dir_callbacks = {
    webserver_get_info,
    webserver_open,
    webserver_read,
    webserver_write,
    webserver_seek,
    webserver_close
};
