/* upnp.c - Routines dealing interfacing with the upnp library
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <stdbool.h>		/* Gnulib, C99 */
#include <upnp.h>		/* libupnp */
#include <upnptools.h>		/* libupnp */
#include <assert.h>		/* C89 */
#include <netinet/in.h>		/* ?; inet_ntoa */
#include <arpa/inet.h>		/* ?; inet_ntoa */
#include "xvasprintf.h"		/* Gnulib */
#include "xgethostname.h"	/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "gmediaserver.h"
#include "intutil.h"
#include "web/MediaServer.h"

char *friendly_name = NULL;
static UpnpDevice_Handle device;

static Service services[] = {
  {
    "urn:upnp-org:serviceId:ContentDirectory",
    "urn:schemas-upnp-org:service:ContentDirectory:1",
    contentdir_service_actions
  },
  {
    "urn:upnp-org:serviceId:ConnectionManager",
    "urn:schemas-upnp-org:service:ConnectionManager:1",
    connectmgr_service_actions
  },
  { NULL, NULL, NULL }
};

static const char *
upnp_errmsg(int res)
{
    return UpnpGetErrorMessage(res);

/*    switch (res) {
    case UPNP_E_SUCCESS:
    	return "Success";   	    	    // strerror(ENOMEM);
    case UPNP_E_INVALID_HANDLE:
    	return "Invalid UPnP handle";
    case UPNP_E_INVALID_PARAM:
    	return "Invalid argument";  	    // strerror(EINVAL);
    case UPNP_E_OUTOF_HANDLE:
    	return "Out of handles";
    case UPNP_E_OUTOF_CONTEXT:
    case UPNP_E_BUFFER_TOO_SMALL:
    case UPNP_E_INVALID_SID:
    case UPNP_E_INIT_FAILED:
    	return UpnpGetErrorMessage(res);
    case UPNP_E_OUTOF_MEMORY:
    	return "Cannot allocate memory";    // strerror(ENOMEM);
    case UPNP_E_INIT:
    	return "UPnP already initialized";
    case UPNP_E_INVALID_DESC:
    	return "Invalid description document";
    case UPNP_E_INVALID_URL:
    	return "Invalid URL";
    case UPNP_E_INVALID_DEVICE:
    	return "Invalid device";
    case UPNP_E_INVALID_SERVICE:
    	return "Invalid service";
    case UPNP_E_BAD_RESPONSE:
    	return "Bad response";
    case UPNP_E_BAD_REQUEST:
    	return "Bad request";
    case UPNP_E_INVALID_ACTION:
    	return "Invalid action";
    case UPNP_E_FINISH:
    	return "UPnP not initialized";
    case UPNP_E_URL_TOO_BIG:
    	return "URL too long";
    case UPNP_E_BAD_HTTPMSG:
    	return "Invalid HTTP message";
    case UPNP_E_ALREADY_REGISTERED:
    	return "Client/device already registered";
    case UPNP_E_NETWORK_ERROR:
    	return "Network error";
    case UPNP_E_SOCKET_WRITE:
    case UPNP_E_SOCKET_READ:
    case UPNP_E_SOCKET_BIND:
    case UPNP_E_SOCKET_CONNECT:
    case UPNP_E_OUTOF_SOCKET:
    case UPNP_E_LISTEN:
    case UPNP_E_TIMEDOUT:
    case UPNP_E_SOCKET_ERROR:
    case UPNP_E_FILE_WRITE_ERROR:
    case UPNP_E_EVENT_PROTOCOL:
    case UPNP_E_SUBSCRIBE_UNACCEPTED:
    case UPNP_E_UNSUBSCRIBE_UNACCEPTED:
    case UPNP_E_NOTIFY_UNACCEPTED:
    case UPNP_E_INVALID_ARGUMENT:
    case UPNP_E_FILE_NOT_FOUND:
    case UPNP_E_FILE_READ_ERROR:
    case UPNP_E_EXT_NOT_XML:
    case UPNP_E_NO_WEB_SERVER:
    case UPNP_E_OUTOF_BOUNDS:
    case UPNP_E_NOT_FOUND:
    case UPNP_E_INTERNAL_ERROR:
    case UPNP_SOAP_E_INVALID_ACTION:
    case UPNP_SOAP_E_INVALID_ARGS:
    case UPNP_SOAP_E_OUT_OF_SYNC:
    case UPNP_SOAP_E_INVALID_VAR:
    case UPNP_SOAP_E_ACTION_FAILED:
    }
*/
}

static void
handle_subscription_request(struct Upnp_Subscription_Request *event)
{
    say(2, "Event received: Subscription request\n");
    /* XXX: not implemented */
}

static void
handle_get_var_request(struct Upnp_State_Var_Request *event)
{
    say(2, "Event received: Get variable request\n");
    /* XXX: not implemented */
}

static bool
find_service_action(struct Upnp_Action_Request *request, Service **service, ServiceAction **action)
{
    int c;
    int d;

    *service = NULL;
    *action = NULL;

    for (c = 0; services[c].id != NULL; c++) {
        if (strcmp(services[c].id, request->ServiceID) == 0) {
	    *service = &services[c];
            for (d = 0; services[c].actions[d].name != NULL; d++) {
                if (strcmp(services[c].actions[d].name, request->ActionName) == 0) {
		    *action = &services[c].actions[d];
		    return true;
		}
            }
            return false;
        }
    }

    return false;
}


static void
handle_action_request(struct Upnp_Action_Request *request)
{
    Service *service;
    ServiceAction *action;

    say(2, "Event received: Action request\n");
    say(3, "Event device UDN: %s\n", request->DevUDN);
    say(3, "Event service ID: %s\n", request->ServiceID);
    say(3, "Event action name: %s\n", request->ActionName);
    if (request->ErrCode != UPNP_E_SUCCESS)
	say(3, "Event error: %s (%d)\n", upnp_errmsg(request->ErrCode), request->ErrCode);
    say(3, "Event source: %s\n", inet_ntoa(request->CtrlPtIPAddr));

    if (verbosity >= 4) {
        DOMString dump;
        char *line;
        char *line2;

        dump = ixmlPrintDocument(request->ActionRequest);
        say(4, "Event action request:\n");
        for (line = dump; (line2 = strstr(line, "\n")) != NULL; line = line2+1) {
            *line2 = '\0';
            say(4, "  %s\n", line);
        }
	if (*line)
            say(4, "%s\n", line);
        ixmlFreeDOMString(dump);
    }

    if (strcmp(request->DevUDN, DEVICE_UDN) != 0) {
        say(1, "Discarding event - event device UDN not recognized\n");
        return;
    }

    if (find_service_action(request, &service, &action)) {
	ActionEvent event;

	event.request = request;
	event.status = true;
	event.service = service;
	if (action->function(&event) && event.status) {
	    request->ErrCode = UPNP_E_SUCCESS;


	    if (verbosity >= 4) {
		DOMString dump;
		char *line;
		char *line2;
		
		dump = ixmlPrintDocument(request->ActionResult);
		say(4, "Event action result:\n");
		for (line = dump; (line2 = strstr(line, "\n")) != NULL; line = line2+1) {
		    *line2 = '\0';
		    say(4, "  %s\n", line);
		}
		if (*line)
		    say(4, "%s\n", line);
		ixmlFreeDOMString(dump);
	    }

	    say(2, "Action executed successfully\n");
	} else {
	    say(2, "Action failed\n");
	}
	return;
    }
    
    if (service != NULL) {
	say(1, "Discarding event - action name not recognized\n");
	strcpy(request->ErrStr, "Unrecognized action");
    } else {
	say(1, "Discarding event - service ID not recognized\n");
	strcpy(request->ErrStr, "Unrecognized service");
    }
    request->ActionResult = NULL;
    request->ErrCode = UPNP_SOAP_E_INVALID_ACTION;
}

static int
device_callback_event_handler(Upnp_EventType eventtype, void *event, void *cookie)
{
    switch (eventtype) {
    case UPNP_EVENT_SUBSCRIPTION_REQUEST:
        handle_subscription_request((struct Upnp_Subscription_Request *) event);
        break;
    case UPNP_CONTROL_GET_VAR_REQUEST:
        handle_get_var_request((struct Upnp_State_Var_Request *) event);
        break;
    case UPNP_CONTROL_ACTION_REQUEST:
        handle_action_request((struct Upnp_Action_Request *) event);
        break;
    default:
        say(2, "Received an unknown event (type 0x%x)\n", eventtype);
        break;
    }

    return 0; /* return valid is ignored by libupnp */
}

void
upnp_set_error(ActionEvent *event, const char *format, ...)
{
    va_list ap;
    
    va_start(ap, format);
    event->status = false;
    event->request->ActionResult = NULL;
    event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
    vsnprintf(event->request->ErrStr, sizeof(event->request->ErrStr), format, ap);
    va_end(ap);
}

int32_t
upnp_get_i4(ActionEvent *event, const char *key)
{
    char *value;
    int32_t out;

    value = upnp_get_string(event, key);
    if (value == NULL)
	return 0;
    if (parse_int32(value, &out))
	return out;

    upnp_set_error(event, "Invalid value for %s argument (%s)", key, value);
    return 0;
}

uint32_t
upnp_get_ui4(ActionEvent *event, const char *key)
{
    char *value;
    uint32_t out;

    value = upnp_get_string(event, key);
    if (value == NULL)
	return 0;
    if (parse_uint32(value, &out))
	return out;

    upnp_set_error(event, "Invalid value for %s argument (%s)", key, value);
    return 0;
}

char *
upnp_get_string(ActionEvent *event, const char *key)
{
    IXML_Node *node;
    
    node = (IXML_Node *) event->request->ActionRequest;
    if (node == NULL) {
	upnp_set_error(event, "Invalid action request document");
	return NULL;
    }
    node = ixmlNode_getFirstChild(node);
    if (node == NULL) {
	upnp_set_error(event, "Invalid action request document");
	return NULL;
    }
    node = ixmlNode_getFirstChild(node);

    for (; node != NULL; node = ixmlNode_getNextSibling(node)) {
	if (strcmp(ixmlNode_getNodeName(node), key) == 0) {
	    node = ixmlNode_getFirstChild(node);
	    if (node == NULL) {
		/* Are we sure empty arguments are reported like this? */
		return "";
	    }
	    return ixmlNode_getNodeValue(node);
	}
    }

    upnp_set_error(event, "Missing action request argument (%s)", key);
    return NULL;
}

/* XXX: key is really const here, but UpnpAddToActionResponse doesn't take const key */
bool
upnp_add_response(ActionEvent *event, char *key, const char *value)
{
    char *val;
    int res;

    if (!event->status)
	return false;

    val = strdup(value);
    if (val == NULL) {
	/* report memory failure */
	event->status = false;
        event->request->ActionResult = NULL;
        event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
        strcpy(event->request->ErrStr, errstr);
        return false;
    }

    res = UpnpAddToActionResponse(&event->request->ActionResult, event->request->ActionName, event->service->type, key, val);
    if (res != UPNP_E_SUCCESS) {
	/* report custom error */
        free(val);
        event->request->ActionResult = NULL;
        event->request->ErrCode = UPNP_SOAP_E_ACTION_FAILED;
        strcpy(event->request->ErrStr, upnp_errmsg(res));
        return false;
    }

    return true;
}

void
init_upnp(const char *listenip, uint16_t listenport)
{
    int res;
    char *mediaserver_desc;

    say(3, "Initializing UPnP subsystem...\n");
    res = UpnpInit(listenip, listenport);
    if (res != UPNP_E_SUCCESS)
        die("Cannot initialize UPnP subsystem - %s\n", upnp_errmsg(res));

    say(1, "UPnP MediaServer listening on %s:%d\n", UpnpGetServerIpAddress(), UpnpGetServerPort());

    say(3, "Enabling UPnP web server...\n");
    res = UpnpEnableWebserver(TRUE);
    if (res != UPNP_E_SUCCESS)
        die("Cannot enable UPnP web server - %s\n", upnp_errmsg(res));
    res = UpnpSetVirtualDirCallbacks(&virtual_dir_callbacks);
    if (res != UPNP_E_SUCCESS)
        die("Cannot set virtual directory callbacks - %s\n", upnp_errmsg(res));
    res = UpnpAddVirtualDir("/audio");
    if (res != UPNP_E_SUCCESS)
        die("Cannot add virtual directory for web server - %s\n", upnp_errmsg(res));

    say(3, "Registering UPnP root device...\n");

    if (friendly_name != NULL)
	friendly_name = xasprintf("GMediaServer on %s", xgethostname());
    mediaserver_desc = xasprintf(MEDIASERVER_DESC, friendly_name);
    free(friendly_name);
    res = UpnpRegisterRootDevice2(UPNPREG_BUF_DESC, mediaserver_desc,
              strlen(mediaserver_desc), 1, device_callback_event_handler, NULL, &device);
    free(mediaserver_desc);
    if (res != UPNP_E_SUCCESS)
        die("Cannot register root device - %s\n", upnp_errmsg(res));

    say(1, "Sending UPnP advertisement for device...\n");
    res = UpnpSendAdvertisement(device, SSDP_PAUSE);
    if (res != UPNP_E_SUCCESS)
        die("Cannot send device advertisement - %s\n", upnp_errmsg(res));

    say(1, "Listening for control point connections...\n");
}

void
finish_upnp(void)
{
    int res;

    say(1, "Shutting down UPnP MediaServer...\n");
    res = UpnpUnRegisterRootDevice(device);
    if (res != UPNP_E_SUCCESS)
        die("Cannot unregister root device - %s\n", upnp_errmsg(res));

    say(3, "Deinitializing UPnP subsystem...\n");
    res = UpnpFinish();
    if (res != UPNP_E_SUCCESS)
        die("Cannot deinitialize UPnP library - %s\n", upnp_errmsg(res));
}
