/* strbuf.h - The string buffer data-structure.
 *
 * Copyright (C) 2004 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef STRBUF_H
#define STRBUF_H

#include <stdint.h>
#include <stdarg.h>

typedef struct _StrBuf StrBuf;

struct _StrBuf {
    char *buf;
    uint32_t len;
    uint32_t capacity;
};

StrBuf *strbuf_new(void);
StrBuf *strbuf_new_with_capacity(uint32_t capacity);
StrBuf *strbuf_new_from_substring(const char *str, int32_t sp, int32_t ep);
#define strbuf_new_from_string(str) 	    	    strbuf_new_from_substring(str,0,-1)

void strbuf_free(StrBuf *sb);
char *strbuf_free_to_string(StrBuf *sb);

#define strbuf_append_char(sb,chr)   	    	        strbuf_append_char_n(sb,1,chr)
#define strbuf_append(sb,str) 	    	    	        strbuf_append_n(sb,1,str)
#define strbuf_append_substring(sb,str,ssp,sep)	        strbuf_append_substring(sb,1,str,ssp,sep)
#define strbuf_append_data(sb,mem,len)   	    	strbuf_append_data_n(sb,1,mem,len)
#define strbuf_appendf(sb,fmt...)   	    	        strbuf_appendf_n(sb,1,fmt)
#define strbuf_vappendf(sb,fmt,ap)  	    	        strbuf_vappendf_n(sb,1,fmt,ap)
#define strbuf_append_char_n(sb,n,chr)                  strbuf_replace_char_n(sb,-1,-1,n,chr)
#define strbuf_append_n(sb,n,str)                       strbuf_replace_n(sb,-1,-1,n,str)
#define strbuf_append_data_n(sb,n,mem,len)   	    	strbuf_replace_data_n(sb,-1,-1,n,mem,len)
#define strbuf_append_substring_n(sb,n,str,ssp,sep)     strbuf_replace_substring_n(sb,-1,-1,n,str,ssp,sep)
#define strbuf_appendf_n(sb,n,fmt...)                   strbuf_replacef_n(sb,-1,-1,n,fmt)
#define strbuf_vappendf_n(sb,n,fmt,ap)                  strbuf_vreplacef_n(sb,-1,-1,n,fmt,ap)

#define strbuf_prepend_char(sb,chr)   	    	        strbuf_prepend_char_n(sb,1,chr)
#define strbuf_prepend(sb,str) 	    	    	        strbuf_prepend_n(sb,1,str)
#define strbuf_prepend_data(sb,mem,len) 	        strbuf_prepend_data_n(sb,1,mem,len)
#define strbuf_prepend_substring(sb,str,ssp,sep)        strbuf_prepend_substring_n(sb,1,str,ssp,sep)
#define strbuf_prependf(sb,fmt...)   	    	        strbuf_prependf_n(sb,1,fmt)
#define strbuf_vprependf(sb,fmt,ap)  	    	        strbuf_vprependf_n(sb,1,fmt,ap)
#define strbuf_prepend_char_n(sb,n,chr)                 strbuf_replace_char_n(sb,0,0,n,chr)
#define strbuf_prepend_n(sb,n,str)                      strbuf_replace_n(sb,0,0,n,str)
#define strbuf_prepend_data_n(sb,n,mem,len) 	        strbuf_replace_data_n(sb,0,0,n,mem,len)
#define strbuf_prepend_substring_n(sb,n,str,ssp,sep)    strbuf_replace_substring_n(sb,0,0,n,str,ssp,sep)
#define strbuf_prependf_n(sb,n,fmt...)                  strbuf_replacef_n(sb,0,0,n,fmt)
#define strbuf_vprependf_n(sb,n,fmt,ap)                 strbuf_vreplacef_n(sb,0,0,n,fmt,ap)

#define strbuf_insert_char(sb,sp,chr)   	        strbuf_insert_char_n(sb,sp,1,chr)
#define strbuf_insert(sb,sp,str) 	    	        strbuf_insert_n(sb,sp,1,str)
#define strbuf_insert_data(sb,sp,mem,len) 	        strbuf_insert_data_n(sb,sp,1,mem,len)
#define strbuf_insert_substring(sb,sp,str,ssp,sep)      strbuf_insert_substring_n(sb,sp,1,str,ssp,sep)
#define strbuf_insertf(sb,sp,fmt...)   	    	        strbuf_insertf_n(sb,sp,1,fmt)
#define strbuf_vinsertf(sb,sp,fmt,ap)  	    	        strbuf_vinsertf_n(sb,sp,1,fmt,ap)
#define strbuf_insert_char_n(sb,sp,n,chr)               strbuf_replace_char_n(sb,sp,sp,n,chr)
#define strbuf_insert_n(sb,sp,n,str)                    strbuf_replace_n(sb,sp,sp,n,str)
#define strbuf_insert_data_n(sb,sp,n,mem,len) 	        strbuf_replace_data_n(sb,sp,sp,n,mem,len)
#define strbuf_insert_substring_n(sb,sp,n,str,ssp,sep)  strbuf_replace_substring_n(sb,sp,sp,n,str,ssp,sep)
#define strbuf_insertf_n(sb,sp,n,fmt...)                strbuf_replacef_n(sb,sp,sp,n,fmt)
#define strbuf_vinsertf_n(sb,sp,n,fmt,ap)               strbuf_vreplacef_n(sb,sp,sp,n,fmt,ap)

#define strbuf_set_char(sb,chr) 	    	        strbuf_set_char_n(sb,1,chr)
#define strbuf_set(sb,str)  	    	    	        strbuf_set_n(sb,1,str)
#define strbuf_set_data(sb,mem,len)  	            	strbuf_set_data_n(sb,1,mem,len)
#define strbuf_set_substring(sb,str,ssp,sep)  	        strbuf_set_substring_n(sb,1,str,ssp,sep)
#define strbuf_setf(sb,fmt...)  	    	        strbuf_setf_n(sb,1,fmt)
#define strbuf_vsetf(sb,fmt,ap)  	    	        strbuf_vsetf_n(sb,1,fmt,ap)
#define strbuf_set_char_n(sb,chr)                       strbuf_replace_char_n(sb,0,-1,n,chr)
#define strbuf_set_n(sb,str)                            strbuf_replace_n(sb,0,-1,n,str)
#define strbuf_set_data_n(sb,mem,len)  	            	strbuf_replace_data_n(sb,0,-1,n,mem,len)
#define strbuf_set_substring_n(sb,str,ssp,sep)          strbuf_replace_substring_n(sb,0,-1,n,str,ssp,sep)
#define strbuf_setf_n(sb,fmt...)                        strbuf_replacef_n(sb,0,-1,n,fmt)
#define strbuf_vsetf_n(sb,fmt,ap)                       strbuf_vreplacef_n(sb,0,-1,n,fmt,ap)

#define strbuf_replace_char(sb,sp,ep,ch)                strbuf_replace_char_n(sb,sp,ep,1,ch)
#define strbuf_replace(sb,sp,ep,str)	    	        strbuf_replace_n(sb,sp,ep,1,str)
#define strbuf_replace_data(sb,sp,ep,mem,len)	    	strbuf_replace_data_n(sb,sp,ep,1,mem,len)
#define strbuf_replace_substring(sb,sp,ep,str,ssp,sep)  strbuf_replace_substring_n(sb,sp,ep,1,str,ssp,sep)
#define strbuf_replacef(sb,sp,ep,fmt...)                strbuf_replacef_n(sb,sp,ep,1,fmt)
#define strbuf_vreplacef(sb,sp,ep,fmt,ap)               strbuf_vreplacef_n(sb,sp,ep,1,fmt,ap)
void strbuf_replace_char_n(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times, char ch);
#define strbuf_replace_n(sb,sp,ep,n,str)	    	strbuf_replace_substring_n(sb,sp,ep,n,str,0,-1)
void strbuf_replace_substring_n(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times, const char *str, int32_t ssp, int32_t sep);
void strbuf_replace_data_n(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times, const void *mem, uint32_t len);
int strbuf_replacef_n(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times, const char *fmt, ...) __attribute__ ((format (printf, 5, 6)));
int strbuf_vreplacef_n(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times, const char *fmt, va_list ap) __attribute__ ((format (printf, 5, 0)));

#define strbuf_set_char_at(sb,sp,chr)	    	        strbuf_replace(sb,sp,sp+1,chr)
#define strbuf_delete_char(sb,sp)   	    	        strbuf_replace(sb,sp,sp+1,NULL)
#define strbuf_delete(sb,sp,ep)   	    	        strbuf_replace(sb,sp,ep,NULL)
#define strbuf_clear(sb)   	    	    	        strbuf_replace(sb,0,-1,NULL)

void strbuf_reverse_substring(StrBuf *sb, int32_t sp, int32_t ep);
#define strbuf_reverse(sb)  	    	    	        strbuf_reverse_substring(sb,0,-1)

void strbuf_repeat_substring(StrBuf *sb, int32_t sp, int32_t ep, uint32_t times);
#define strbuf_repeat(sb,n)  	    	    	        strbuf_repeat_substring(sb,0,-1,n)

uint32_t strbuf_length(StrBuf *sb);
uint32_t strbuf_capacity(StrBuf *sb);
char *strbuf_buffer(StrBuf *sb);

void strbuf_set_length(StrBuf *sb, uint32_t new_length);
void strbuf_ensure_capacity(StrBuf *sb, uint32_t minimum_capacity);

char *strbuf_substring(StrBuf *sb, int32_t sp, int32_t ep);
#define strbuf_to_string(sb)	    	    	        strbuf_substring(sb,0,-1)
char strbuf_char_at(StrBuf *sb, int32_t index);

#endif
