/* metadata.c - Management of file metadata
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <stdint.h>		/* Gnulib, C99 */
#include <dirent.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <fcntl.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <ctype.h>		/* C89 */
#include <id3.h>		/* id3lib */
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "strbuf.h"
#include "gmediaserver.h"

#define DEFAULT_ENTRIES_SIZE 512

typedef enum {
    FILE_UNKNOWN,
    FILE_MPEG_ADTS,
    FILE_MP3_ID3,
    FILE_WMA
} FileType;

char *rootpath = NULL;
static Entry *root_entry;
static uint32_t entry_count = 0;
static uint32_t entries_size = 0;
static Entry **entries;
#ifdef HAVE_ID3LIB
bool id3_enabled = true;
static ID3Tag *id3;
#endif

/* Concatenate two file names.
 * An empty path component ("") is equivalent to the current directory,
 * but will not yield "./" when concatenated.
 */
/* XXX: move to support */
static char *
concat_filenames(const char *p1, const char *p2)
{
    size_t l1;
    size_t l2;
    char *out;

    if (strcmp(p1, "") == 0) {
	if (strcmp(p2, "") == 0)
	    return ".";
        return xstrdup(p2);
    }
    if (strcmp(p2, "") == 0)
        return xstrdup(p1);

    l1 = strlen(p1);
    l2 = strlen(p2);

    if (p1[l1-1] == '/')
        l1--;

    out = xmalloc(l1+1+l2+1);
    memcpy(out, p1, l1);
    out[l1] = '/';
    memcpy(out+l1+1, p2, l2+1);
    return out;
}

char *
get_entry_path(Entry *entry)
{
    StrBuf *path;
    char *fullpath;

    path = strbuf_new();
    while (entry->id != 0) {
	strbuf_prepend(path, entry->filename);
	if (entry->parent != 0)
	    strbuf_prepend(path, "/");
	entry = entries[entry->parent];
    }

    fullpath = concat_filenames(rootpath, strbuf_buffer(path));
    strbuf_free(path);
    return fullpath;
}

static Entry*
make_entry(const char *name, int32_t parent, bool directory)
{
    Entry *entry;

    entry = xmalloc(sizeof(Entry));
    entry->id = entry_count++;
    entry->child_count = directory ? 0 : -1;
    entry->children = NULL;
    entry->parent = parent;
    entry->filename = xstrdup(name);
    entry->title = NULL;
    entry->artist = NULL;
    entry->album = NULL;
    entry->genre = NULL;

    if (entry->id >= entries_size) {
        entries_size *= 2;
        entries = xrealloc(entries, entries_size * sizeof(Entry *));
    }
    entries[entry->id] = entry;
    
    return entry;
}

Entry *
get_entry_by_id(uint32_t id)
{
    if (id < 0 || id >= entry_count)
        return NULL;

    return entries[id];
}

static FileType
check_file_content_type(const char *fullpath)
{
    int fd;
    uint8_t buf[4];
    size_t count;

    say(4, "Check content type of file %s\n", fullpath);

    fd = open(fullpath, O_RDONLY);
    if (fd < 0) {
	warn("%s: cannot open for reading: %s\n", fullpath, errstr);
	return FILE_UNKNOWN;
    }

    count = full_read(fd, buf, 4);
    if (count < 4) {
	if (errno != 0)
	    warn("%s: cannot read: %s\n", fullpath, errstr);
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_UNKNOWN;
    }

    close(fd); /* Ignore errors since we opened for reading */
    
    if (buf[0] == 0xFF && (buf[1] & 0xFE) == 0xFA) {
	say(4, "Matched type MPEG ADTS for %s\n", fullpath);
	return FILE_MPEG_ADTS;
    }
    if (buf[0] == 'I' && buf[1] == 'D' && buf[2] == '3') {
	say(4, "Matched type MP3 with ID3 tag for %s\n", fullpath);
	return FILE_MP3_ID3;
    }
    if (buf[0] == 0x30 && buf[1] == 0x26 && buf[2] == 0xb2 && buf[3] == 0x75) {
	say(4, "Matched type WMA for %s\n", fullpath);
	return FILE_WMA;
    }

    say(4, "%s: skipping file - unrecognized content type\n", fullpath);
    return FILE_UNKNOWN;
}

#ifdef HAVE_ID3LIB
static char *
get_id3_string(ID3Tag *id3tag, ID3_FrameID frame_id)
{
    ID3Frame *frame;
    ID3Field *field;
    size_t size;
    char *buf;

    frame = ID3Tag_FindFrameWithID(id3tag, frame_id);
    if (frame != NULL) {
	field = ID3Frame_GetField(frame, ID3FN_TEXT);
	if (field != NULL) {
	    size = ID3Field_Size(field);
	    buf = xmalloc(size+1);
	    ID3Field_GetASCII(field, buf, size);
	    buf[size] = '\0';
	    return buf;
	}
    }

    return NULL;
}
#endif

static Entry *
scan_entry(const char *fullpath, const char *name, int32_t parent)
{
    struct stat sb;

    if (stat(fullpath, &sb) < 0) {
        warn("%s: cannot stat: %s\n", fullpath, errstr);
        return NULL;
    }

    if (S_ISDIR(sb.st_mode)) {
        Entry *entry;
	int32_t *children;
        struct dirent **dirents;
        int dirent_count;
        int c;

	say(4, "Scanning directory %s\n", fullpath);
        dirent_count = scandir(fullpath, &dirents, NULL, NULL);
        if (dirent_count < 0) {
            warn("%s: cannot scan directory: %s\n", fullpath, errstr);
            return NULL;
        }

        entry = make_entry(name, parent, true);
	children = xmalloc(sizeof(int32_t) * dirent_count);

	entry->child_count = 0;
        for (c = 0; c < dirent_count; c++) {
            if (strcmp(dirents[c]->d_name, ".") != 0 && strcmp(dirents[c]->d_name, "..") != 0) {
		Entry *child;
		char *child_path;

		child_path = concat_filenames(fullpath, dirents[c]->d_name);
		child = scan_entry(child_path, dirents[c]->d_name, entry->id);
		if (child != NULL)
		    children[entry->child_count++] = child->id;
		free(child_path);
	    }
        }

	entry->children = xmemdup(children, sizeof(int32_t) * entry->child_count);
	free(children);

        return entry;
    }

    if (S_ISREG(sb.st_mode)) {
	Entry *entry;
	FileType type;

        if (parent == -1) {
            warn("%s: root must be a directory\n", fullpath);
            return NULL;
        }
	type = check_file_content_type(fullpath);
	if (type == FILE_UNKNOWN)
	    return NULL;

	say(4, "Adding file %s\n", fullpath);
	entry = make_entry(name, parent, false);

#ifdef HAVE_ID3LIB
	if (id3_enabled && (type == FILE_MPEG_ADTS || type == FILE_MP3_ID3)) {
	    ID3Tag_Clear(id3);
	    ID3Tag_Link(id3, fullpath);
	    entry->title = get_id3_string(id3, ID3FID_TITLE);
	    entry->album = get_id3_string(id3, ID3FID_ALBUM);
	    entry->artist = get_id3_string(id3, ID3FID_LEADARTIST);
	    entry->genre = get_id3_string(id3, ID3FID_CONTENTTYPE);
	}
#endif
	
        return entry;
    }

    warn("%s: skipping file - unsupported file type\n", fullpath);
    return NULL;
}

bool
init_metadata(void)
{
    entries_size = DEFAULT_ENTRIES_SIZE;
    entries = xmalloc(sizeof(Entry *) * entries_size);
#ifdef HAVE_ID3LIB
    if (id3_enabled)
	id3 = ID3Tag_New();
#endif
    root_entry = scan_entry(rootpath, "", -1);
#ifdef HAVE_ID3LIB
    if (id3_enabled)
	ID3Tag_Delete(id3);
#endif
    return root_entry != NULL;
}

void
finish_metadata(void)
{
    uint32_t c;

    for (c = 0; c < entry_count; c++) {
	free(entries[c]->children);
	free(entries[c]->title);
	free(entries[c]->artist);
	free(entries[c]->album);
	free(entries[c]->genre);
	free(entries[c]->filename);
        free(entries[c]);
    }

    free(entries);
}
