/* main.c - Main routines for the gmediaserver
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <upnp.h>		/* libupnp */
#include <upnptools.h>		/* libupnp */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include <getopt.h>		/* Gnulib, GNU */
#include <errno.h>		/* ? */
#include <stdio.h>		/* C89 */
#include <string.h>		/* C89 */
#include <stdlib.h>		/* C89 */
#include <stdarg.h>		/* C89 */
#include <stdbool.h>		/* Gnulib, C99 */
#include <signal.h>		/* ? */
#include "xalloc.h"		/* Gnulib */
#include "version-etc.h"	/* Gnulib */
#include "intutil.h"
#include "gmediaserver.h"

enum {
    OPT_DISABLE_ID3,
    OPT_FRIENDLY_NAME,
    OPT_PIDFILE,
    OPT_PORT,
    OPT_NOTIMESTAMP,
    OPT_TIMESTAMP,
    OPT_HELP,
    OPT_VERSION,
};

#define SHORT_OPTIONS "bvi:o:p:"

static struct option long_options[] = {
#ifdef HAVE_ID3LIB
    { "disable-id3", no_argument, NULL, OPT_DISABLE_ID3 },
#endif
    { "friendly-name", required_argument, NULL, OPT_FRIENDLY_NAME },
    { "pid-file", required_argument, NULL, OPT_PIDFILE },
    { "port", required_argument, NULL, 'p' },
    { "output", required_argument, NULL, 'o' },
    { "background", no_argument, NULL, 'b' },
    { "no-timestamp", no_argument, NULL, OPT_NOTIMESTAMP },
    { "timestamp", optional_argument, NULL, OPT_TIMESTAMP },
    { "interface", required_argument, NULL, 'i' },
    { "verbose", no_argument, NULL, 'v' },
    { "help", no_argument, NULL, OPT_HELP },
    { "version", no_argument, NULL, OPT_VERSION },
    { NULL, 0, NULL, 0 }
};

const char version_etc_copyright[] = "Copyright (C) 2004 Oskar Liljeblad.";

static void
dummy_signal_handler(int signal)
{
    /* No operation */
}

int
main(int argc, char **argv)
{
    struct sigaction signalaction;
    sigset_t signalset;
    char *pidfilename = NULL;
    char *listenip = NULL;
    uint16_t listenport = 0;
    bool background = false;
    bool timestamp_specified = false;

    while (1) {
    	int c;

        c = getopt_long(argc, argv, SHORT_OPTIONS, long_options, NULL);
        if (c == -1)
            break;

        switch (c) {
        case 'i':
	    if ((listenip = get_ip_by_spec(optarg)) == NULL)
	    	die("invalid address or interface name `%s'\n", optarg);
            break;
        case 'o':
            if ((logfh = fopen(optarg, "a")) == NULL)
                die("cannot open log file `%s' for appending\n", optarg);
	    setlinebuf(logfh);
            break;
#ifdef HAVE_ID3LIB
	case OPT_DISABLE_ID3:
	    id3_enabled = false;
	    break;
#endif
	case OPT_FRIENDLY_NAME:
	    if (optarg[0] == '\0')
		die("friendly name cannot be empty\n");
	    friendly_name = xstrdup(optarg);
	    break;
	case OPT_PIDFILE:
	    pidfilename = optarg;
	    break;
	case 'p':
	    if (!parse_uint16(optarg, &listenport))
		die("invalid port number `%s'\n", optarg);
	    break;
	case OPT_NOTIMESTAMP:
	    timestamp_format = NULL;
	    timestamp_specified = true;
	    break;
	case OPT_TIMESTAMP:
	    timestamp_format = (optarg != NULL ? optarg : DEFAULT_TIMESTAMP_FORMAT);
	    timestamp_specified = true;
	    break;
        case 'b':
	    background = true;
	    break;
        case 'v':
	    if (optarg != NULL) {
		if (optarg[0] == 'v') {
		    for (verbosity = 0; optarg[verbosity] != '\0'; verbosity++) {
			if (optarg[verbosity] != 'v')
			    die("invalid verbosity `%s'\n", optarg);
		    }
		    verbosity++;
		} else {
		    if (!parse_uint32(optarg, &verbosity))
			die("invalid verbosity `%s'\n", optarg);
		}
		if (verbosity > MAX_VERBOSITY)
		    die("maximum verbosity is %d\n", MAX_VERBOSITY);
	    } else {
		verbosity = 1;
	    }
            break;
        case OPT_HELP:
            printf("Usage: %s [OPTIONS]... DIRECTORY\n", argv[0]);
            printf("Run the UPnP media server.\n");
            printf("\n");
	    printf("      --friendly-name=NAME  set display name for media server\n");
#ifdef HAVE_ID3LIB
	    printf("      --disable-id3         do not scan ID3 information on files\n");
#endif
	    printf("  -v, --verbose[=LEVEL]     set verbosity level (0-4)\n");
            printf("      --pid-file=FILE       write pid to FILE when up and running\n");
            printf("  -i, --interface=NAME      listen on a specific interface\n");
            printf("  -p, --port=PORT           listen on a specific port\n");
            printf("  -o, --output=LOGFILE      file for logging\n");
            printf("  -b, --background          go to background (detach)\n");
            printf("      --no-timestamp        do not prepend timestamp to log entries\n");
            printf("      --timestamp[=FORMAT]  prepend timestamp with optional time format\n");
            printf("      --help                display this help and exit\n");
            printf("      --version             output version information and exit\n");
            printf("\n");
	    printf("The directory argument specifies where audio files are located.\n");
            printf("See strftime(3) for description of the timestamp format.\n");
	    printf("\n");
            printf("Report bugs to <%s>.\n", PACKAGE_BUGREPORT);
            exit(EXIT_SUCCESS);
        case OPT_VERSION:
            version_etc(stdout, NULL, PACKAGE, VERSION, "Oskar Liljeblad", NULL);
            exit(EXIT_SUCCESS);
    	case '?':
	    exit(EXIT_FAILURE);
        }
    }

    if (argc - optind <= 0)
	die("missing directory argument\nTry `%s --help' for more information.\n", argv[0]);
    if (argc - optind > 1)
	die("too many arguments\nTry `%s --help' for more information.\n", argv[0]);
    rootpath = argv[optind];

    if (listenip != NULL)
	say(2, "Using IP address `%s'.\n", listenip);

    if (!timestamp_specified && logfh != NULL)
    	timestamp_format = DEFAULT_TIMESTAMP_FORMAT;

    if (background) {
        pid_t pid;

        /*say(2, "Moving process into background...\n");*/
    	if ((pid = fork()) < 0)
    	    die("Cannot fork new process - %s\n", errstr);
        if (pid != 0)
            exit(EXIT_SUCCESS);

        fclose(stdin);	    	/* ignore errors */
	fclose(stdout);     	/* ignore errors */
	fclose(stderr);     	/* ignore errors */
        close(STDIN_FILENO);	/* ignore errors */
        close(STDOUT_FILENO);	/* ignore errors */
        close(STDERR_FILENO);	/* ignore errors */

        if (setsid() < 0)
	    die("Cannot create process session - %s\n", errstr);

	/* XXX: The following is to work around a bug in 
	 *      libupnp 1.2.1 - fd 0 must not be closed!
	 */
	if (open("/dev/null", O_RDONLY) < 0)
	    die("Cannot open /dev/null - %s\n", errstr);
    }

    signalaction.sa_handler = dummy_signal_handler;
    if (sigemptyset(&signalaction.sa_mask) < 0)
    	die("Cannot empty signal set - %s\n", errstr);
    signalaction.sa_flags = SA_RESTART;
    signalaction.sa_restorer = NULL;
    if (sigaction(SIGINT, &signalaction, NULL) < 0)
    	die("Cannot register signal handler - %s\n", errstr);
    if (sigaction(SIGTERM, &signalaction, NULL) < 0)
    	die("Cannot register signal handler - %s\n", errstr);
    if (sigaction(SIGUSR1, &signalaction, NULL) < 0)
    	die("Cannot register signal handler - %s\n", errstr);

    if (sigemptyset(&signalset) < 0)
    	die("Cannot empty signal set - %s\n", errstr);
    if (sigaddset(&signalset, SIGTERM) < 0)
    	die("Cannot add signal to set - %s\n", errstr);
    if (sigaddset(&signalset, SIGINT) < 0)
    	die("Cannot add signal to set - %s\n", errstr);
    if (sigaddset(&signalset, SIGUSR1) < 0)
    	die("Cannot add signal to set - %s\n", errstr);
    if (sigprocmask(SIG_BLOCK, &signalset, NULL) < 0)
    	die("Cannot block signals - %s\n", errstr);

    if (!init_metadata()) {
	if (logfh != NULL)
	    fclose(logfh);	/* ignore errors */
	exit(EXIT_FAILURE);
    }

    log_starting();
    
    if (pidfilename != NULL) {
	FILE *pidfh;
	say(2, "Writing pid to pid file `%s'\n", pidfilename);
	if ((pidfh = fopen(pidfilename, "w")) == NULL)
	    die("Cannot open pid file `%s' for writing - %s\n", pidfilename, errstr);
	fprintf(pidfh, "%d", getpid());
	if (fclose(pidfh) != 0)
	    die("Cannot close pid file `%s' - %s\n", pidfilename, errstr);
    }

    init_upnp(listenip, listenport);

    while (true) {
    	int sig;
      	
      	if ((sig = TEMP_FAILURE_RETRY(sigwaitinfo(&signalset, NULL))) < 0)
    	    die("sigwaitinfo failed - %s\n", errstr);
      	say(2, "Received signal %d\n", sig);
      	if (sig == SIGUSR1) {
      	    //say(1, "Reloading configuration file...\n"); /* XXX: not implemented */
      	}
      	if (sig == SIGTERM || sig == SIGINT)
            break;
    }
    
    finish_upnp();
    finish_metadata();

    if (pidfilename != NULL)
    	unlink(pidfilename);	/* ignore errors */

    say(2, "%s process id %d terminated normally\n", PACKAGE, getpid());

    if (logfh != NULL)
        fclose(logfh);	/* ignore errors */

    exit(EXIT_SUCCESS);
}
