/* log.c - Logging and user information routines
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include "gmediaserver.h"

#define MAX_TIMESTAMP_LENGTH 1024

uint32_t verbosity = 0;
FILE *logfh = NULL;
char *timestamp_format = NULL;
static bool started = false;

static void
log_timestamp(FILE *out)
{
    if (timestamp_format != NULL) {
        time_t curtime;
        struct tm *loctime;
        size_t len;

        if (time(&curtime) < 0) {
            /* XXX: report error somehow? */
            return;
        }
	loctime = localtime(&curtime);
        len = strftime(NULL, -1, timestamp_format, loctime);

        if (len > 0) {
            char buf[len+1];

            strftime(buf, len+1, timestamp_format, loctime);
            fprintf(out, "%s ", buf);
        }
    }
}

void
log_starting(void)
{
    started = true;
    say(2, "%s process id %d starting\n", PACKAGE, getpid());
}

void
say(int level, const char *format, ...)
{
    if (verbosity >= level) {
    	va_list args;

    	va_start(args, format);
    	if (logfh != NULL) {
    	    log_timestamp(logfh);
    	    vfprintf(logfh, format, args);
        } else {
    	    log_timestamp(stdout);
    	    vfprintf(stdout, format, args);
        }
	va_end(args);
    }
}

void
die(const char *format, ...)
{
    va_list args;

    va_start(args, format);
    if (logfh != NULL) {
    	log_timestamp(logfh);
      	fputs("Error: ", logfh);
      	vfprintf(logfh, format, args);
    } else {
        log_timestamp(stderr);
        fprintf(stderr, "%s: ", program_invocation_name);
    	vfprintf(stderr, format, args);
    }
    va_end(args);

    if (started)
	say(2, PACKAGE " process id %d terminated due to error\n", getpid());

    exit(1);
}

void
warn(const char *format, ...)
{
    va_list args;

    va_start(args, format);
    if (logfh != NULL) {
      log_timestamp(logfh);
      fprintf(logfh, "Warning: ");
      vfprintf(logfh, format, args);
    } else {
      log_timestamp(stderr);
      fprintf(stderr, "%s: ", program_invocation_name);
      vfprintf(stderr, format, args);
    }
    va_end(args);
}
