/* gmediaserver.h - Common definitions
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef GMEDIASERVER_H
#define GMEDIASERVER_H

#include <errno.h>		/* C89; errno */
#include <stdarg.h>		/* C89; (variadict) */
#include <stdio.h>		/* C89; FILE */
#include <string.h>		/* C89 */
#include <stdbool.h>		/* Gnulib, C99; bool */
#include <upnp.h>		/* libupnp */

typedef struct _ServiceAction ServiceAction;
typedef struct _Service Service;
typedef struct _ActionEvent ActionEvent;
typedef struct _Entry Entry;

struct _ServiceAction {
    char *name;
    bool (*function)(ActionEvent *);
};

struct _Service {
    char *id;
    char *type;
    ServiceAction *actions;
};

struct _ActionEvent {
    struct Upnp_Action_Request *request;
    bool status;
    Service *service;
};

struct _Entry {
    int32_t id;
    int32_t child_count;
    int32_t parent;
    char *filename;
    char *title;
    char *artist;
    char *album;
    char *genre;
    int32_t *children;
};

/* log.h */
#define DEFAULT_TIMESTAMP_FORMAT "%Y-%m-%d %H:%M"
#define MAX_VERBOSITY 4
#define errstr strerror(errno)
extern FILE *logfh;
extern uint32_t verbosity;
extern char *timestamp_format;
void say(int level, const char *format, ...);
void die(const char *format, ...);
void warn(const char *format, ...);
void log_starting(void);

/* main.c */

/* interface.h */
char *get_ip_by_spec(char *spec);

/* metadata.c */
extern bool id3_enabled;
extern char *rootpath;
bool init_metadata(void);
void finish_metadata(void);
Entry *get_entry_by_id(uint32_t id);
char *get_entry_path(Entry *entry);

/* upnp.c */
extern char *friendly_name;
void init_upnp(const char *listenip, uint16_t listenport);
void finish_upnp(void);
void upnp_set_error(ActionEvent *event, const char *format, ...);
char *upnp_get_string(ActionEvent *event, const char *item);
uint32_t upnp_get_ui4(ActionEvent *event, const char *key);
bool upnp_add_response(ActionEvent *event, char *key, const char *value);
int32_t upnp_get_i4(ActionEvent *event, const char *key);

/* webserver.c */
extern struct UpnpVirtualDirCallbacks virtual_dir_callbacks;

/* contentdir.c */
extern ServiceAction contentdir_service_actions[];

/* connectmgr.c */
extern ServiceAction connectmgr_service_actions[];

/* url.c */
void sgmlescape(const char *str, char *target, uint32_t *length);
char *xsgmlescape(const char *str);
void urlencode(const char *str, char *target, uint32_t *length);
char *xurlencode(const char *str);

#endif
