/* contentdir.c - Implementation of UPnP ContentDirectory
 *
 * Copyright (C) 2005  Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>
#include <stdint.h>
#include <inttypes.h>
#include "xvasprintf.h"
#include "minmax.h"
#include "strbuf.h"
#include "intutil.h"
#include "gmediaserver.h"

static bool
contentdir_get_search_capabilities(ActionEvent *event)
{
    upnp_add_response(event, "SearchCaps", "");
    return event->status;
}

static bool
contentdir_get_sort_capabilities(ActionEvent *event)
{
    upnp_add_response(event, "SortCaps", "");
    return event->status;
}

static bool
contentdir_get_system_update_id(ActionEvent *event)
{
    upnp_add_response(event, "Id", "0");
    return event->status;
}

static void
append_escaped_xml(StrBuf *out, const char *entity, const char *value)
{
    char *str;

    str = xsgmlescape(value);
    strbuf_appendf(out, "<%s>%s</%s>", entity, value, entity);
    free(str);
}

static void
append_entry_details(StrBuf *out, Entry *entry)
{
    if (entry->title != NULL)
	append_escaped_xml(out, "dc:title", entry->title);
    else
	append_escaped_xml(out, "dc:title", entry->filename);
    if (entry->artist != NULL)
	append_escaped_xml(out, "upnp:artist", entry->artist); /* XXX: or dc:creator? */
    if (entry->album != NULL)
	append_escaped_xml(out, "upnp:album", entry->album);
    if (entry->genre != NULL)
	append_escaped_xml(out, "upnp:genre", entry->genre);
}

static bool
contentdir_browse(ActionEvent *event)
{
    uint32_t index;
    uint32_t count;
    char *flag;
    char *filter;
    int32_t id;
    char *sort_criteria;
    bool metadata;

    /* Retrieve arguments */
    index = upnp_get_ui4(event, "StartingIndex");
    count = upnp_get_ui4(event, "RequestedCount");
    /* ObjectID is a string according to ContentDir specification, but we use int32. */
    /* XXX: is this OK? maybe we should use a more appropriate error response if not int32. */
    id = upnp_get_i4(event, "ObjectID"); 
    filter = upnp_get_string(event, "Filter");
    flag = upnp_get_string(event, "BrowseFlag");
    sort_criteria = upnp_get_string(event, "SortCriteria");
    if (!event->status)
	return false;

    /* Check arguments */
    if (strcmp(flag, "BrowseMetadata") == 0) {
	if (index != 0) {
	    upnp_set_error(event, "StartingIndex must be 0 when BrowseFlag is BrowseMetaData.");
	    return false;
	}
	metadata = true;
    } else if (strcmp(flag, "BrowseDirectChildren") == 0) {
	metadata = false;
    } else {
	upnp_set_error(event, "Invalid BrowseFlag argument value (%s)", flag);
	return false;
    }

    if (metadata) {
        Entry *entry;

        entry = get_entry_by_id(id);
	if (entry == NULL) {
            upnp_set_error(event, "No such object");
	    return false;
	}
    } else {
        Entry *entry;
        StrBuf *out;
	uint32_t c = 0;
	uint32_t result_count = 0;

        entry = get_entry_by_id(id);
	if (entry == NULL) {
            upnp_set_error(event, "No such object");
	    return false;
	}

        if (entry->child_count == -1) {
            upnp_set_error(event, "BrowseDirectChildren only possible on containers");
            return false;
        }

        out = strbuf_new();
        strbuf_append(out, 
            "<DIDL-Lite xmlns=\"urn:schemas-upnp-org:metadata-1-0/DIDL-Lite\""
            " xmlns:dc=\"http://purl.org/dc/elements/1.1/\""
            " xmlns:upnp=\"urn:schemas-upnp-org:metadata-1-0/upnp/\">");

        for (c = index; c < MIN(index+count, entry->child_count); c++) {
	    Entry *child = get_entry_by_id(entry->children[c]);

            if (child->child_count >= 0) {
                strbuf_appendf(out,
                        "<container id=\"%" PRIi32 "\" parentID=\"%" PRIi32 "\" restricted=\"true\" childCount=\"%" PRIi32 "\">"
                        "<class>object.container.storageFolder</class>",
                        child->id,
                        child->parent,
                        child->child_count);
		append_entry_details(out, child);
		strbuf_append(out, "</container>");
            } else {
                strbuf_appendf(out,
                        "<item id=\"%" PRIi32 "\" parentID=\"%" PRIi32 "\" restricted=\"true\">"
			"<class>object.item.audioItem.musicTrack</class>"
			"<res protocolInfo=\"http-get:*:audio/mpeg:*\">http://%s:%d/audio/%d</res>",
			child->id,
			child->parent,
			UpnpGetServerIpAddress(),
			UpnpGetServerPort(),
			child->id);
		append_entry_details(out, child);
		strbuf_append(out, "</item>");
	    }
	    result_count++;
        }
	strbuf_append(out, "</DIDL-Lite>");

        upnp_add_response(event, "Result", strbuf_buffer(out));
        upnp_add_response(event, "NumberReturned", int32_str(result_count));
        upnp_add_response(event, "TotalMatches", int32_str(entry->child_count));
        strbuf_free(out);
    }

    upnp_add_response(event, "UpdateID", "0");
    upnp_add_response(event, "SessionID", "0");

    return event->status;
}

/*
ServiceVariable contentdir_service_variables[] = {
  { "A_ARG_TYPE_ObjectID"
  { "A_ARG_TYPE_Result"
//{ "A_ARG_TYPE_SearchCriteria"
  { "A_ARG_TYPE_BrowseFlag"
  { "A_ARG_TYPE_Filter"
  { "A_ARG_TYPE_SortCriteria"
  { "A_ARG_TYPE_Index"
  { "A_ARG_TYPE_Count"
  { "A_ARG_TYPE_UpdateID"
//{ "A_ARG_Type_TransferID"
//{ "A_ARG_Type_TransferLength"
//{ "A_ARG_Type_TransferTotal"
//{ "A_ARG_TYPE_TagValueList"
//{ "A_ARG_TYPE_URI"
  { "SearchCapabilities"
  { "SortCapabilities"
  { "SystemUpdateID"
//{ "ContainerUpdateIDs"
};
*/

ServiceAction contentdir_service_actions[] = {
  { "GetSearchCapabilities", contentdir_get_search_capabilities },
  { "GetSortCapabilities", contentdir_get_sort_capabilities },
  { "GetSystemUpdateID", contentdir_get_system_update_id },
  { "Browse", contentdir_browse },
/*{ "Search", contentdir_search },*/
/*{ "CreateObject", contentdir_create_object },*/
/*{ "DestroyObject", contentdir_destroy_object },*/
/*{ "UpdateObject", contentdir_update_object },*/
/*{ "ImportResource", contentdir_import_resource },*/
/*{ "GetTransferProgress", contentdir_get_transfer_progress },*/
/*{ "DeleteResource", contentdir_delete_resource },*/
/*{ "CreateReference", contentdir_create_reference },*/
  { NULL, NULL }
};
