/* metadata.c - Management of file metadata
 *
 * Copyright (C) 2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <config.h>
#ifdef HAVE_ID3LIB
#include <id3.h>                /* id3lib */
#undef false /* id3lib bug - it shouldn't mess with false and true! */
#undef true
#endif
#ifdef HAVE_TAGLIB
#include <tag_c.h>
#undef BOOL /* conflicts with libupnp */
#endif
#include <stdbool.h>            /* Gnulib, C99 */
#include <stdint.h>		/* Gnulib, C99 */
#include <dirent.h>		/* POSIX */
#include <sys/stat.h>		/* POSIX */
#include <sys/types.h>		/* POSIX */
#include <fcntl.h>		/* POSIX */
#include <unistd.h>		/* POSIX */
#include <ctype.h>		/* C89 */
#include "gettext.h"		/* Gnulib/gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "dirname.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "quote.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "getnline.h"		/* Gnulib */
#include "strbuf.h"
#include "strutil.h"
#include "gmediaserver.h"

#define DEFAULT_ENTRIES_SIZE 		512
#define DEFAULT_FILE_TYPES 		"mp3,wma,m3u,pls"
#define ROOT_ENTRY_NAME 		"(root)"
#define DEFAULT_PLAYLIST_ENTRIES	16
#define MAX_INVALID_M3U_FILES		16
#define MAX_INVALID_EXTM3U_FILES	16
#define MAX_INVALID_PLS_FILES		16
#define MAX_INVALID_PLS_LINES		16

typedef enum {
    FILE_MP3,
    FILE_MP3_ID3,
    FILE_WMA,
    FILE_RIFF_WAVE,
    FILE_M4A,
    FILE_PLS,
    FILE_M3U,
    FILE_EXTM3U,
    FILE_UNKNOWN,
    FILE_TYPES_COUNT,
} FileType;

static const char *file_type_mime_types[] = {
    [FILE_MP3] 		= "audio/mpeg",
    [FILE_MP3_ID3]	= "audio/mpeg",
    [FILE_WMA] 		= "audio/x-ms-wma",
    [FILE_RIFF_WAVE] 	= "audio/x-wav",
    [FILE_M4A] 		= "audio/mp4",  
    [FILE_PLS] 		= "audio/x-scpls",
    [FILE_M3U] 		= "audio/m3u",
    [FILE_EXTM3U] 	= "audio/m3u",
    [FILE_UNKNOWN] 	= "application/octet-stream",
};

static const char *file_type_names[] = {
    [FILE_MP3] 		= "mp3",
    [FILE_MP3_ID3]	= "mp3", /* possibly id3mp3 or mp3id3 in the future */
    [FILE_WMA] 		= "wma",
    [FILE_RIFF_WAVE] 	= "wav",
    [FILE_M4A] 		= "m4a",
    [FILE_PLS] 		= "pls",
    [FILE_M3U] 		= "m3u",
    [FILE_EXTM3U] 	= "m3u", /* possibly extm3u in the future */
    [FILE_UNKNOWN] 	= "unknown",
};

static const char *file_type_descs[] = {
    [FILE_MP3] 		= "MPEG-1 Audio Layer 3 (MP3)",
    [FILE_MP3_ID3]	= "MPEG-1 Audio Layer 3 (MP3) with leading ID3 tag",
    [FILE_WMA] 		= "Windows Media Audio (WMA)",
    [FILE_RIFF_WAVE] 	= "RIFF Wave",
    [FILE_M4A] 		= "MPEG v4 iTunes AAC-LC (M4A)",
    [FILE_PLS]		= "PLS playlist",
    [FILE_M3U]		= "Simple M3U playlist",
    [FILE_EXTM3U]	= "Extended M3U playlist",
};

static Entry *scan_entry(const char *fullpath, const char *name, int32_t parent);
static Entry *scan_playlist_file(const char *fullpath, const char *name, int32_t parent, FileType type);

static Entry *root_entry;
static uint32_t entry_count = 0;
static uint32_t entries_size = 0;
static Entry **entries;
char *file_types = DEFAULT_FILE_TYPES;
static ithread_mutex_t metadata_mutex;
bool tags_enabled = true;
#ifdef HAVE_ID3LIB
static ID3Tag *id3;
#endif

EntryDetail *
get_entry_detail(Entry *entry, EntryDetailType type)
{
    EntryDetail *d;
    
    for (d = entry->details; d != NULL; d = d->next) {
	if (d->type == type)
	    return d;
    }

    return NULL;
}

bool
has_entry_detail(Entry *entry, EntryDetailType type)
{
    return get_entry_detail(entry, type) != NULL;
}

static EntryDetail *
add_entry_detail(Entry *entry, EntryDetailType type)
{
    EntryDetail *detail;

    detail = xmalloc(sizeof(EntryDetail));
    detail->next = entry->details;
    detail->type = type;
    entry->details = detail;

    return detail;
}

static Entry *
make_entry(const char *name, int32_t parent, bool directory)
{
    Entry *entry;

    entry = xmalloc(sizeof(Entry));
    entry->id = entry_count++;
    entry->parent = parent;
    entry->name = xstrdup(name);
    entry->details = NULL;

    if (directory) {
	EntryDetail *detail;
	detail = add_entry_detail(entry, DETAIL_CHILDREN);
	detail->data.children.count = 0;
	detail->data.children.list = NULL;
    }

    if (entry->id >= entries_size) {
        entries_size *= 2;
        entries = xrealloc(entries, entries_size * sizeof(Entry *));
    }
    entries[entry->id] = entry;
    
    return entry;
}

Entry *
get_entry_by_id(uint32_t id)
{
    if (id < 0 || id >= entry_count)
        return NULL;

    return entries[id];
}


static bool
attempt_read_at(int fd, size_t size, off_t pos, uint8_t *buf, const char *fullpath)
{
    size_t count;

    if (lseek(fd, pos, SEEK_SET) != pos) {
	warn(_("%s: cannot seek: %s\n"), quotearg(fullpath), errstr);
	close(fd); /* Ignore errors since we opened for reading */
	return false;
    }

    count = full_read(fd, buf, size);
    if (count < size) {
	if (errno != 0)
	    warn(_("%s: cannot read: %s\n"), quotearg(fullpath), errstr);
	close(fd); /* Ignore errors since we opened for reading */
	return false;
    }
    return true;
}

static FileType
check_file_content_type(const char *fullpath)
{
    int fd;
    uint8_t buf[11];

    /* XXX: Use stdio instead of fd syscalls */
    fd = open(fullpath, O_RDONLY);
    if (fd < 0) {
	warn(_("%s: cannot open for reading: %s\n"), quotearg(fullpath), errstr);
	return FILE_UNKNOWN;
    }

    if (!attempt_read_at(fd, 4, 0, buf, fullpath))
	return FILE_UNKNOWN;

    /* MPEG ADTS, layer III, v1 */
    if (buf[0] == 0xFF && (buf[1] & 0xFE) == 0xFA) {
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_MP3;
    }
    /* MP3 file with ID3 tag */
    if (buf[0] == 'I' && buf[1] == 'D' && buf[2] == '3') {
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_MP3_ID3;
    }
    /* Microsoft ASF */
    if (buf[0] == 0x30 && buf[1] == 0x26 && buf[2] == 0xb2 && buf[3] == 0x75) {
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_WMA;
    }
    /* RIFF (little-endian), WAVE audio */
    if (buf[0] == 'R' && buf[1] == 'I' && buf[2] == 'F' && buf[3] == 'F') {
	if (!attempt_read_at(fd, 4, 8, buf, fullpath))
	    return FILE_UNKNOWN;
	if (buf[0] == 'W' && buf[1] == 'A' && buf[2] == 'V' && buf[3] == 'E') {
	    close(fd); /* Ignore errors since we opened for reading */
	    return FILE_RIFF_WAVE;
	}
    }

    /* ISO Media, MPEG v4 system, iTunes AAC-LC */
    if (!attempt_read_at(fd, 4, 4, buf+4, fullpath))
	return FILE_UNKNOWN;
    if (buf[4] == 'f' && buf[5] == 't' && buf[6] == 'y' && buf[7] == 'p') {
	if (!attempt_read_at(fd, 4, 8, buf+8, fullpath))
	    return FILE_UNKNOWN;
	if (buf[8] == 'M' && buf[9] == '4' && buf[10] == 'A') {
	    close(fd); /* Ignore errors since we opened for reading */
	    return FILE_M4A;
	}
    }

    /* Extended M3U */
    if (memcmp(buf, "#EXTM3U", 7) == 0 && (buf[7] == '\r' || buf[7] == '\n')) {
        close(fd); /* Ignore errors since we opened for reading */
        return FILE_EXTM3U;
    }
    /* Playlist (PLS) */
    if (!attempt_read_at(fd, 4, 8, buf+8, fullpath))
	return FILE_UNKNOWN;
    if (memcmp(buf, "[playlist]", 10) == 0 && (buf[10] == '\r' || buf[10] == '\n')) {
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_PLS;
    }
    /* Simple M3U */
    if (ends_with_nocase(fullpath, ".m3u") == 0) {
	close(fd); /* Ignore errors since we opened for reading */
	return FILE_M3U;
    }

    close(fd); /* Ignore errors since we opened for reading */
    return FILE_UNKNOWN;
}

#ifdef HAVE_ID3LIB
static char *
get_id3_string(ID3Tag *id3tag, ID3_FrameID frame_id)
{
    ID3Frame *frame;
    ID3Field *field;
    size_t size;
    char *buf;

    frame = ID3Tag_FindFrameWithID(id3tag, frame_id);
    if (frame != NULL) {
	field = ID3Frame_GetField(frame, ID3FN_TEXT);
	if (field != NULL) {
	    size = ID3Field_Size(field);
	    buf = xmalloc(size+1);
	    ID3Field_GetASCII(field, buf, size);
	    buf[size] = '\0';
	    return buf;
	}
    }

    return NULL;
}
#endif

static bool
add_playlist_child(Entry *parent, size_t *children_size, char *buf, const char *playlist_dir)
{
    Entry *child;
    char *child_path = buf;

    if (buf[0] != '/' && strncasecmp(buf, "http://", 7) != 0)
        child_path = concat_filenames(playlist_dir, buf);
    /* XXX: using base_name here may not be entire appropriate -
     * maybe child_path is an url?
     */
    child = scan_entry(child_path, base_name(child_path), parent->id); 
    if (child != NULL) {
        EntryChildren *children;

        children = &(get_entry_detail(parent, DETAIL_CHILDREN)->data.children);
        if (children->list == NULL) {
            children->list = xmalloc((*children_size) * sizeof(int32_t));
        } else if (children->count >= (*children_size)) {
            *children_size *= 2;
            children->list = xrealloc(children->list, (*children_size) * sizeof(int32_t));
        }
        children->list[children->count++] = child->id;
    }
    if (child_path != buf)
        free(child_path);

    return child != NULL;
}

static Entry *
scan_playlist_file(const char *fullpath, const char *name, int32_t parent, FileType type)
{
    FILE *fh;
    size_t alloc_max;
    char *buf = NULL;
    size_t bufsize = 0;
    ssize_t len;
    Entry *entry;
    char *playlist_dir;
    uint32_t children_size = DEFAULT_PLAYLIST_ENTRIES; /* just some basic initial size */
    int ln;
    int bad = 0;

    say(4, _("Scanning playlist %s\n"), quote(fullpath));
    fh = fopen(fullpath, "r");
    if (fh == NULL) {
        warn(_("%s: cannot open file for reading: %s\n"), quotearg(fullpath), errstr);
        return NULL;
    }

    playlist_dir = dir_name(fullpath);
    entry = make_entry(name, parent, true);

    /* Note: we don't handle playlists with MAC-like newlines (CR only),
     * simply because getline or getdelim doesn't handle that easily.
     */
    alloc_max = MAX(FILENAME_MAX+3, FILENAME_MAX); /* to handle overflow */
    if (type == FILE_PLS || type == FILE_EXTM3U) {
        /* We know the first line is OK, we tested that in
         * check_file_content_type. The read can still fail,
         * that's why we test using ferror below.
         */
        getnline(&buf, &bufsize, alloc_max, fh);
        ln = 2;
    } else {
        ln = 1;
    }
    if (!ferror(fh)) {
        for (; (len = getnline(&buf, &bufsize, alloc_max, fh)) >= 0; ln++) {
            len = chomp(buf, len);
            if (type == FILE_PLS) {
                if (len >= 7 && strncasecmp(buf, "Title", 5) == 0) {
                    /* Ignore Title lines. */
                } else if (len >= 8 && strncasecmp(buf, "Length", 6) == 0) {
                    /* Ignore Length lines. */
                } else if (len >= 17 && strncasecmp(buf, "NumberOfEntries=", 16) == 0) {
                    /* Ignore NumberOfEntries lines. */
                } else if (len >= 9 && strncasecmp(buf, "Version=", 8) == 0) {
                    /* Ignore Version lines. */
                } else if (len >= 7 && strncasecmp(buf, "File", 4) == 0) {
                    int c;

                    for (c = 4; c < len && buf[c] >= '0' && buf[c] <= '9'; c++);
                    if (c == 4 || c >= len || buf[c] != '=') {
                        warn(_("%s: invalid line %d\n"), quotearg(fullpath), ln);
                        break;
                    }
                    /* Ignore the numbering of files in PLS playlists. */
                    if (!add_playlist_child(entry, &children_size, buf+c+1, playlist_dir)) {
                        bad++;
                        if (bad > MAX_INVALID_PLS_FILES) {
                            /* For now, just break out of the loop. */
                            warn(_("%s: too many invalid files (max allowed %d)\n"), quotearg(fullpath), MAX_INVALID_PLS_FILES);
                            break;
                        }
                    }
                } else {
                    warn(_("%s: invalid line %d\n"), quotearg(fullpath), ln);
                    bad++;
                    if (bad > MAX_INVALID_PLS_LINES) {
                        /* For now, just break out of the loop. */
                        warn(_("%s: too many invalid lines (max allowed %d)\n"), quotearg(fullpath), MAX_INVALID_PLS_LINES);
                        break;
                    }
                }
            } else if (type == FILE_EXTM3U) {
                if (len >= 8 && strncmp(buf, "#EXTINF:", 8) == 0) {
                    /* No operation - we're not interested in EXTINF */
                } else if (len > 0) {
                    if (!add_playlist_child(entry, &children_size, buf, playlist_dir)) {
                        bad++;
                        if (bad > MAX_INVALID_EXTM3U_FILES) {
                            /* For now, just break out of the loop. */
                            warn(_("%s: too many invalid files (max allowed %d)\n"), quotearg(fullpath), MAX_INVALID_EXTM3U_FILES);
                            break;
                        }
                    }
                }
            } else if (type == FILE_M3U) {
                if (!add_playlist_child(entry, &children_size, buf, playlist_dir)) {
                    bad++;
                    if (bad > MAX_INVALID_M3U_FILES) {
                        /* For now, just break out of the loop. */
                        warn(_("%s: too many invalid files (max allowed %d)\n"), quotearg(fullpath), MAX_INVALID_M3U_FILES);
                        break;
                    }
                }
            }
        }
    }

    /* A read error here means that we may still have read some playlist
     * entries.
     */
    if (ferror(fh))
        warn(_("%s: cannot read from file: %s\n"), quotearg(fullpath), errstr);

    fclose(fh); /* Ignore errors because we opened for reading */
    free(buf);
    free(playlist_dir);
    return entry;
}
    
static Entry *
scan_entry(const char *fullpath, const char *name, int32_t parent)
{
    struct stat sb;

    if (strncasecmp(fullpath, "http://", 7) == 0) {
        Entry *entry;
        EntryDetail *detail;

        say(4, _("Adding URL %s as %s\n"), quote_n(0, fullpath), quote_n(1, name));
        entry = make_entry(name, parent, false);
        detail = add_entry_detail(entry, DETAIL_URL);
        detail->data.url.url = xstrdup(fullpath);

        return entry;
    }

    if (stat(fullpath, &sb) < 0) {
        warn(_("%s: cannot stat: %s\n"), quotearg(fullpath), errstr);
        return NULL;
    }

    if (S_ISDIR(sb.st_mode)) {
        Entry *entry;
	EntryDetail *detail;
	int32_t *children;
	uint32_t child_count;
        struct dirent **dirents;
        int dirent_count;
        int c;

	say(4, _("Scanning directory %s\n"), quote(fullpath));
        dirent_count = scandir(fullpath, &dirents, NULL, alphasort);
        if (dirent_count < 0) {
            warn(_("%s: cannot scan directory: %s\n"), quotearg(fullpath), errstr);
            return NULL;
        }

        entry = make_entry(name, parent, true);
	children = xmalloc(sizeof(int32_t) * dirent_count);

	child_count = 0;
        for (c = 0; c < dirent_count; c++) {
            if (strcmp(dirents[c]->d_name, ".") != 0 && strcmp(dirents[c]->d_name, "..") != 0) {
		Entry *child;
		char *child_path;

		child_path = concat_filenames(fullpath, dirents[c]->d_name);
		child = scan_entry(child_path, dirents[c]->d_name, entry->id);
		if (child != NULL)
		    children[child_count++] = child->id;
		free(child_path);
	    }
        }

	detail = get_entry_detail(entry, DETAIL_CHILDREN);
	detail->data.children.count = child_count;
	detail->data.children.list = xmemdup(children, sizeof(int32_t) * child_count);
	free(children);

        return entry;
    }

    if (S_ISREG(sb.st_mode)) {
	Entry *entry;
	EntryDetail *detail;
	FileType type;

        if (parent == -1) {
            warn(_("%s: root must be a directory\n"), quotearg(fullpath));
            return NULL;
        }
        say(4, _("Checking content type of file %s\n"), quote(fullpath));
	type = check_file_content_type(fullpath);
	if (type == FILE_UNKNOWN) {
	    say(4, _("Matched no type for %s\n"), quote(fullpath));
            /*say(4, _("%s: skipping file - unrecognized content type\n"), quotearg(fullpath));*/
        } else if (type == FILE_M3U) {
	    say(4, _("Assuming type %s for %s\n"), file_type_descs[type], quote(fullpath));
        } else {
	    say(4, _("Matched type %s for %s\n"), file_type_descs[type], quote(fullpath));
        }
	if (!string_in_csv(file_types, ',', file_type_names[type]))
	    return NULL;

        if (type == FILE_PLS || type == FILE_M3U || type == FILE_EXTM3U)
            return scan_playlist_file(fullpath, name, parent, type);

        say(4, _("Adding file %s as %s\n"), quote_n(0, fullpath), quote_n(1, name));
        entry = make_entry(name, parent, false);

        detail = add_entry_detail(entry, DETAIL_FILE);
        detail->data.file.size = sb.st_size;
        detail->data.file.mtime = sb.st_mtime;
        detail->data.file.filename = xstrdup(fullpath);
	detail->data.file.mime_type = file_type_mime_types[type];

	if (tags_enabled && (type == FILE_MP3 || type == FILE_MP3_ID3)) {
#ifdef HAVE_ID3LIB
	    /*char *title;*/
	
	    ID3Tag_Clear(id3);
	    ID3Tag_Link(id3, fullpath);

            if (ID3Tag_NumFrames(id3) != 0) {
                detail = add_entry_detail(entry, DETAIL_TAG);
                /*title = get_id3_string(id3, ID3FID_TITLE);
                if (title != NULL && strcmp(title, "") != 0) {
                    free(entry->name);
                    entry->name = title;
                } else {
                    free(title);
                }*/
                detail->data.tag.title = get_id3_string(id3, ID3FID_TITLE);
                detail->data.tag.album = get_id3_string(id3, ID3FID_ALBUM);
                detail->data.tag.artist = get_id3_string(id3, ID3FID_LEADARTIST);
                detail->data.tag.genre = get_id3_string(id3, ID3FID_CONTENTTYPE);
	    }
#endif
#ifdef HAVE_TAGLIB
            {
                TagLib_File *tl_file;

		/* FIXME: change TagLib_File_MPEG when OGG support added */
                tl_file = taglib_file_new_type(fullpath, TagLib_File_MPEG); 
                if (tl_file != NULL) {
                    TagLib_Tag *tl_tag;

                    tl_tag = taglib_file_tag(tl_file);
                    detail = add_entry_detail(entry, DETAIL_TAG);
                    detail->data.tag.title = xstrdup(taglib_tag_title(tl_tag));

                    detail->data.tag.album = xstrdup(taglib_tag_album(tl_tag));
                    detail->data.tag.artist = xstrdup(taglib_tag_artist(tl_tag));
                    detail->data.tag.genre = xstrdup(taglib_tag_genre(tl_tag));
                    taglib_tag_free_strings();
                    taglib_file_free(tl_file);
                }
            }
#endif
        }

        return entry;
    }

    warn(_("%s: skipping file - unsupported file type\n"), quotearg(fullpath));
    return NULL;
}

void
init_metadata(void)
{
    ithread_mutex_init(&metadata_mutex, NULL);

    entries_size = DEFAULT_ENTRIES_SIZE;
    entries = xmalloc(sizeof(Entry *) * entries_size);

#ifdef HAVE_ID3LIB
    if (tags_enabled)
	id3 = ID3Tag_New();
#endif
}

bool
scan_entries(char **pathv, int pathc)
{
    if (pathc > 1) {
	int32_t *children;
	EntryDetail *detail;
	uint32_t c;

	root_entry = make_entry(ROOT_ENTRY_NAME, -1, true);
	children = xmalloc(sizeof(int32_t) * pathc);
	for (c = 0; c < pathc; c++) {
	    Entry *entry;
	    char *name;
	    uint32_t d;

	    name = base_name(pathv[c]);
	    if (*name == '\0' || *name == '/') {
	        name = xstrdup(pathv[c]);
            } else {
	        for (d = strlen(name)-1; name[d] == '/'; d--);
	        name = xstrndup(name, d+1);
            }

	    entry = scan_entry(pathv[c], name, -1);
	    children[c] = entry->id;
	    free(name);
	}
	detail = get_entry_detail(root_entry, DETAIL_CHILDREN);
	detail->data.children.count = pathc;
	detail->data.children.list = children;
    } else {
	root_entry = scan_entry(pathv[0], ROOT_ENTRY_NAME, -1);
    }


    return root_entry != NULL;
}

void
clear_entries(void)
{
    uint32_t c;

    for (c = 0; c < entry_count; c++) {
	EntryDetail *d = entries[c]->details;

	while (d != NULL) {
	    EntryDetail *next;

	    if (d->type == DETAIL_CHILDREN) {
		free(d->data.children.list);
	    } else if (d->type == DETAIL_FILE) {
	        free(d->data.file.filename);
	    } else if (d->type == DETAIL_TAG) {
		free(d->data.tag.title);
		free(d->data.tag.artist);
		free(d->data.tag.album);
		free(d->data.tag.genre);
	    }

	    next = d->next;
	    free(d);
	    d = next;
	}

	free(entries[c]->name);
        free(entries[c]);
    }

    entry_count = 0;
}

void
finish_metadata(void)
{
    clear_entries();
#ifdef HAVE_ID3LIB
    if (tags_enabled)
	ID3Tag_Delete(id3);
#endif
    free(entries);
    entries_size = 0;

    ithread_mutex_destroy(&metadata_mutex);
}

void
lock_metadata(void)
{
    ithread_mutex_lock(&metadata_mutex);
}

void
unlock_metadata(void)
{
    ithread_mutex_unlock(&metadata_mutex);
}
