#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2008
#       Rafael Cunha de Almeida <almeidaraf@gmail.com>. All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#    1. Redistributions of source code must retain the above copyright notice,
#       this list of conditions and the following disclaimer.
#    2. Redistributions in binary form must reproduce the above copyright
#       notice, this list of conditions and the following disclaimer in the
#       documentation and/or other materials provided with the distribution.
#    3. The name of the author may not be used to endorse or promote products
#       derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
# EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
# BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
# IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import sys
import readline
import subprocess
import os
import os.path
import email
import select
import urllib2
import shutil
import HTMLParser

from getpass import getpass
from htmlentitydefs import entitydefs

import libgmail

# Time (in seconds) to wait between e-mail checks
TIMEOUT = 10

LIST_FOLDERS = 'lf'
LIST_EMAILS = 'lm'
ENTER_FOLDER = 'cd'
ARCHIVE = 'ar'
READ_EMAIL = 'o'
COMPOSE = 'c'
SEND_DRAFT = 's'
REPORT_SPAM = '!'
WAIT_EMAIL = 'wait'
HELP = 'help'
QUIT = 'q'


class NoCommandError(Exception):
    pass


class ExecutionError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return repr(self.message)


class AuthorFieldParser(HTMLParser.HTMLParser):
    def __init__(self):
        HTMLParser.HTMLParser.__init__(self)
        self.inside_span = False
        self.email = ''
        self.text = ''

    def handle_starttag(self, tag, attrs):
        if tag == 'span':
            self.inside_span = True
            self.email = dict(attrs)['id'].replace('_upro_', '')

    def handle_data(self, data):
        if self.inside_span:
            self.text += data + (' <%s>' % self.email)
        else:
            self.text += data

    def handle_endtag(self, tag):
        if tag == 'span':
            self.inside_span = False


class AccountState:
    """This class have variables that must be shared along the commands. The
    ReadEmail command must know what were the last messages displayed to the
    user, for instance."""
    def __init__(self):
        # starting directory is not a label
        self.current_dir = 'inbox'
        self.isLabel = False
        self.labels = []
        # threads from the last lm command
        self.active_threads = []


class Command:
    """This is the basic unit of the program. All the user does is type down
    commands which will give him messages on the screen."""
    def __init__(self, s, state, acc):
        self.state = state
        self.acc = acc

    def execute(self):
        pass


class ListFolders(Command):
    def execute(self):
        if self.acc.getLabelNames():
            all_labels = libgmail.STANDARD_FOLDERS + self.acc.getLabelNames()
        else:
            all_labels = libgmail.STANDARD_FOLDERS
        self.state.labels = list(all_labels)

        for i, name in enumerate(all_labels):
            print i, name


class Archive(Command):
    def __init__(self, s, state, acc):
        Command.__init__(self, s, state, acc)
        self.arg = s.strip()

    def execute(self):
        try:
            self.acc.archiveThread(self.state.active_threads[int(self.arg)])
        except AttributeError:
            print "Version %s of libgmail doesn't support archiving" %\
                  libgmail.Version
        except ValueError:
            raise ExecutionError("`ar' expects a number as parameter")


class ReportSpam(Command):
    def __init__(self, s, state, acc):
        Command.__init__(self, s, state, acc)
        self.arg = s.strip()

    def execute(self):
        try:
            self.acc.reportSpam(self.state.active_threads[int(self.arg)])
        except AttributeError:
            print "Version %s of libgmail doesn't support spam reporting" %\
                  libgmail.Version
        except ValueError:
            raise ExecutionError("`!' expects a number as parameter")


class EnterFolder(Command):
    def __init__(self, s, state, acc):
        Command.__init__(self, s, state, acc)
        self.arg = s.strip()

    def execute(self):
        if not self.arg:
            raise ExecutionError('cd command expects a number ' +
                                 'or value as parameter')

        if self.state.labels:
            all_labels = self.state.labels
        elif self.acc.getLabelNames():
            all_labels = list(libgmail.STANDARD_FOLDERS +
                              self.acc.getLabelNames())
        else:
            all_labels = list(libgmail.STANDARD_FOLDERS)

        try:
            i = int(self.arg)
            self.state.current_dir = all_labels[i]
        except IndexError:
            raise ExecutionError("Label %s doesn't exist" % i)
        except ValueError:
            label = self.arg
            if label not in all_labels:
                raise ExecutionError("Label %s doesn't exist" % label)
            else:
                self.state.current_dir = label

        self.state.active_threads = []
        self.state.isLabel = not self.state.current_dir in\
                                 libgmail.STANDARD_FOLDERS


class ListEmails(Command):
    def execute(self):
        if self.state.isLabel:
            conversations = self.acc.getMessagesByLabel(self.state.current_dir)
        else:
            conversations = self.acc.getMessagesByFolder(self.state.current_dir)

        self.state.active_threads = list(conversations)

        t = []
        for i, c in enumerate(conversations):
            t.append((str(i),
                      ['', 'N'][bool(c.unread)],
                      self.__fix_html(self.__fix_encoding(c.authors)),
                      self.__fix_html(self.__fix_encoding(c.subject)),
                     )
                    )

        for line in self.__tabler(t):
            print line

    #XXX: helper functions. They're ok for now, but I might want to revist them,
    #     maybe write something better as -- if -- the project moves
    def __entitytoletter(self, s):
        newdefs = []
        for k, v in entitydefs.items():
            newdefs.append(('&'+k+';', v))

        for k, v in newdefs:
            s = s.replace(k, v)

        return s

    def __fix_html(self, s):
        parser = AuthorFieldParser()
        try:
            parser.feed(s)
            parser.close()
        except HTMLParser.HTMLParseError:
            #XXX: I gotta make some better html handling some day. The shouldn't
            #be any HTML in the title, really, maybe if I don't change the
            #HTML entities to actual characters before hand, who knows...
            parser.text = s

        return parser.text

    def __fix_encoding(self, s):
        final = []
        i = 0
        while i < len(s):
            if s[i] == '\\' and s[i+1] == 'u':
                num = int(s[i+2:i+6], 16)
                final.append(unichr(num).encode('utf-8'))
                i += 6
            else:
                final.append(s[i])
                i += 1

        return self.__entitytoletter(''.join(final).strip())

    def __greatest_len(self, l):
        return max(map(len, l))

    def __compose_field(self, lines):
        x = self.__greatest_len(lines) + 1
        fields = []
        for line in lines:
            fields.append(line+((x-len(line))*' '))

        return fields

    def __concat(self, t1, t2):
        t = []
        for a, b in zip(t1,t2):
            t.append(a+b)

        return t

    def __tabler(self, t):
        if not t:
            return []

        table = []
        table = self.__compose_field([x[0] for x in t])
        table = self.__concat(table, self.__compose_field([x[1] for x in t]))
        table = self.__concat(table, self.__compose_field([x[2] for x in t]))

        indentsize = self.__greatest_len([x[0] for x in t]) +\
                     self.__greatest_len([x[1] for x in t])
        table = self.__concat(table,
                              ['\n'+((2+indentsize)*' ')+x[3] for x in t])

        return table


class ReadEmail(Command):
    def __init__(self, s, state, acc):
        Command.__init__(self, s, state, acc)
        self.arg = s.strip()

    def __select_payload(self, payload):
        # looking for the real message
        for msg in payload:
            if msg.get_content_type() == 'multipart/alternative':
                payload = msg.get_payload()
                break

        # let's search for html and plain stuff
        html = None
        plain = None
        for msg in payload:
            if msg.get_content_type() == 'text/html':
                html = msg.get_payload(decode=True)
                charset = msg.get_content_charset()
            elif msg.get_content_type() == 'text/plain':
                plain = msg.get_payload(decode=True)
                charset = msg.get_content_charset()

        p = subprocess.Popen(['html2text'],
                             stdin=subprocess.PIPE,
                             stdout=subprocess.PIPE)
        if html:
            (output, err) = p.communicate(html)
            if charset:
                try:
                    return output.decode(charset).encode('utf-8')
                except UnicodeDecodeError:
                    # We can't always trust the information we're given :-/
                    return output
            else:
                return output
        elif plain:
            if charset:
                try:
                    return plain.decode(charset).encode('utf-8')
                except UnicodeDecodeError:
                    # We can't always trust the information we're given :-/
                    return plain
            else:
                return plain
        else:
            tmp = payload[0].get_payload(decode=True)
            charset = payload[0].get_content_charset()
            if charset:
                try:
                    return tmp.decode(charset).encode('utf-8')
                except UnicodeDecodeError:
                    # We can't always trust the information we're given :-/
                    return tmp
            else:
                return tmp

    def __print_field(self, msg, field):
        value = msg.get(field)
        if value:
            return field.capitalize()+': '+value
        else:
            return ''

    EMAIL_DIVISOR = '\n\n'+(80*'-')+'\n\n'

    def __formating(self, text, msgid):
        msg = email.message_from_string(text)
        mget = lambda field: self.__print_field(msg, field)
        if msg.is_multipart():
            body = self.__select_payload(msg.get_payload())
        else:
            body = msg.get_payload(decode=True)
            charset = msg.get_content_charset()
            if charset:
                try:
                    body = body.decode(charset).encode('utf-8')
                except UnicodeDecodeError:
                    # We can't always trust the information we're given :-/
                    return body
        fields = [mget('to'),
                  mget('cc'),
                  mget('from'),
                  mget('date'),
                  'Message-ID: %s' % msgid,
                  mget('subject'),
                  '\n',
                  body,
                  self.EMAIL_DIVISOR]

        return '\n'.join([x for x in fields if x])

    def __add_quote(self, text):
        buffer = []
        for line in text.split('\n'):
            line = '> ' + line
            buffer.append(line)
        return '\n'.join(buffer)

    def __reply_maker(self, text):
        msg = email.message_from_string(text)

        #extracting information
        frm = msg.get('From', '')
        cc = msg.get('CC', '').split(',')
        to = msg.get('To', '').split(',')
        id = msg.get('Message-id', '')
        body = msg.get_payload()
        subject = msg.get('Subject', '')
        if 're:' not in subject.lower():
            subject = 'Re: ' + subject

        #setting up the reply
        body = self.__add_quote(body)
        body = ("On %s, %s wrote:\n" % (msg.get('Date'), frm)) + body
        newmsg = email.message_from_string(body)
        newmsg['To'] = frm
        receivers = [x for x in to if self.acc.name not in x]
        receivers.extend(cc)
        newmsg['CC'] =  ', '.join(receivers)
        newmsg['In-reply-to'] = id
        newmsg['Subject'] = subject

        return newmsg.as_string()

    def execute(self):
        try:
            conversation = self.state.active_threads[int(self.arg)]
        except ValueError:
            raise ExecutionError("`o' expects a number as parameter")
        except IndexError:
            raise ExecutionError("Invalid thread number")

        f = open(TMP, 'w')
        for msg in conversation:
            print>>f, self.__formating(msg.source, msg.id)
        f.close()
        mtime = os.path.getmtime(TMP)

        subprocess.call([EDITOR, TMP])

        if mtime != os.path.getmtime(TMP):
            shutil.copy(TMP, DRAFT)
            text = self.__reply_maker(open(DRAFT).read())
            f = open(DRAFT, 'w')
            f.write(text)
            f.close()


class WaitEmail(Command):
    def __init__(self, s, state, acc):
        Command.__init__(self, s, state, acc)
        self.arg = [x.strip() for x in s.split()]

    def __search_new(self):
        for folder in self.arg:
            try:
                if folder in libgmail.STANDARD_FOLDERS:
                    conversations = self.acc.getMessagesByFolder(folder)
                else:
                    conversations = self.acc.getMessagesByLabel(folder)
            except urllib2.URLError:
                return
            for msg in conversations:
                if msg.unread:
                    return folder

    def execute(self):
        folder = None
        while 1:
            rl, wl, xl = select.select([sys.stdin],[],[],TIMEOUT)
            if sys.stdin in rl:
                sys.stdin.read(1)
                break
            folder = self.__search_new()
            if folder:
                break

        if folder:
            conf = Config(os.path.expanduser('~/.gmailreader/config'))
            script = os.path.expanduser(conf.get('script'))
            if script:
                subprocess.call([script])
            CommandFactory.generate('cd %s' % folder, self.acc).execute()
            CommandFactory.generate('lm', self.acc).execute()


class SendEmail(Command):
    def execute(self):
        text = open(DRAFT).read()
        attrs = email.message_from_string(text)
        msg = libgmail.GmailComposedMessage(attrs.get('to'),
                                            attrs.get('subject'),
                                            attrs.get_payload(),
                                            attrs.get('cc'),
                                            attrs.get('bcc'))
        self.acc.sendMessage(msg, replyTo=attrs.get('in-reply-to'))


class ComposeEmail(Command):
    def execute(self):
        subprocess.call([EDITOR, DRAFT])


class Help(Command):
    def execute(self):
        s = """Help:
lf              - List folders
lm              - List e-mails
cd <num>|<name> - Go inside the folder indicated by `num'
                  (as shown by lf) or by the folder's name
o <num>         - Open e-mail of the number `num' indicated
                  when `lm' was executed
wait <name1> <name2> ... - Keeps on waiting for the named folders
                           if new email arrives it executes a script
                           pointed out on .gmailreader/config and
                           prints the folder contents on screen
c               - Edit draft file
s               - Send draft
ar <num>        - Archive e-mail indicated by the number `num'
! <num>         - Report e-mail indicated by the number `num' as spam
help            - Prints this message
q               - Quit (c-d and c-c also work)"""
        print s


class CommandFactory:
    """Factory class used to generate new Commands and keep the execution state
    throught the AccountState class"""
    accstate = AccountState()

    @classmethod
    def generate(cls, s, acc):
        """generate(str, GmailAccout) -> Command

        This method serves as a generator for an executable command"""
        tmp = s.split()
        cmdtype = tmp[0]
        rest = ''.join(tmp[1:])

        if cmdtype == LIST_FOLDERS:
            return ListFolders(rest, cls.accstate, acc)
        elif cmdtype == LIST_EMAILS:
            return ListEmails(rest, cls.accstate, acc)
        elif cmdtype == ENTER_FOLDER:
            return EnterFolder(rest, cls.accstate, acc)
        elif cmdtype == READ_EMAIL:
            return ReadEmail(rest, cls.accstate, acc)
        elif cmdtype == COMPOSE:
            return ComposeEmail(rest, cls.accstate, acc)
        elif cmdtype == SEND_DRAFT:
            return SendEmail(rest, cls.accstate, acc)
        elif cmdtype == ARCHIVE:
            return Archive(rest, cls.accstate, acc)
        elif cmdtype == REPORT_SPAM:
            return ReportSpam(rest, cls.accstate, acc)
        elif cmdtype == WAIT_EMAIL:
            return WaitEmail(rest, cls.accstate, acc)
        elif cmdtype == HELP:
            return Help(rest, cls.accstate, acc)
        elif cmdtype == QUIT:
            raise SystemExit
        else:
            raise NoCommandError()


class Config:
    """This is a readonly config file handler."""
    def __init__(self, fname):
        self.attrs = {}
        try:
            lines = open(fname).read().split('\n')
        except:
            open(fname, 'w').close()
            os.chmod(fname, 0600)
        else:
            for line in lines:
                arg = line.split('=')
                key = arg[0].strip()
                self.attrs[key] = reduce(str.__add__, arg[1:], '').strip()
    
    def get(self, key, default = lambda: None):
        """get(key, defaultfun) -> string|None

        This method was created allowing for lazy evaluation, you should pass a
        function as a parameter which, when evaluated will return the default
        value. This allow for very small code for when we want to read user
        input as a default."""
        if self.attrs.has_key(key):
            return self.attrs[key]
        else:
            return default()


def main():
    conf = Config(os.path.expanduser('~/.gmailreader/config'))
    email = conf.get('username', lambda: raw_input("Username: "))
    email += '@gmail.com'
    pw = conf.get('password', lambda: getpass("Password: "))

    acc = libgmail.GmailAccount(email, pw)

    print 'Please wait while logging in ...'

    try:
        acc.login()
    except libgmail.GmailLoginFailure,e:
        print "Login failed: %s" % e.message
        raise SystemExit

    # Start by printing the inbox contents
    CommandFactory.generate(LIST_EMAILS, acc).execute()

    while 1:
        try:
            cmd = raw_input('gmail> ').strip()
            if cmd:
                command = CommandFactory.generate(cmd, acc)
            else:
                continue
        except EOFError:
            print
            raise SystemExit
        except NoCommandError:
            print "what?!"
            continue

        try:
            command.execute()
        except ExecutionError, e:
            print "Error: ", e.message


if __name__ == '__main__':
    try:
        # global variables setup
        if os.system('mkdir -p ~/.gmailreader'):
            sys.stderr.write('Unable to create ~/.gmailreader\n')
            raise SystemExit, 1

        DRAFT = os.path.expanduser('~/.gmailreader/draft')
        TMP = os.path.expanduser('~/.gmailreader/tmp')

        EDITOR=Config(os.path.expanduser('~/.gmailreader/config')).get('editor')
        if not EDITOR:
            EDITOR = os.getenv('EDITOR')
        if not EDITOR:
            EDITOR = 'vi'

        main()
    except KeyboardInterrupt:
        print
