/* 
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _SMARTPTR_HH_
#define _SMARTPTR_HH_

#include <string>
/** @memo reference counted objects implementation
    used to handle instances in containers (like vectors)
    motivated by \URL[Mumit's  STL Newbie guide]{http://www.xraylith.wisc.edu/~khan/software/stl/STL.newbie.html#class_pointer}
    inspired by \URL[David Harvey]{http://web.ftech.net/~honeyg/articles/smartp.htm}
*/
template <class T>
class SmartPtr
{
public:
  /// Default constructor
  SmartPtr(void) : _counted(0) {};
  /** Constructor based on a pointer to the desired object
      After this call, the pointer is under control !
  */
  SmartPtr(T * ptr) : _counted(0) {
    _counted = new Counted<T>(ptr);
    _counted->getRef();
  };
  /** Destructor : free a reference of the counted object
      If the references counter == 0, free the object
   */
  ~SmartPtr(void) { unBind(); };
  /// Copy-constructor : get a new reference to the counted object
  SmartPtr(SmartPtr<T> const & ref) : 
    _counted(ref._counted) 
  {
    if (!Null())
      {
	_counted->getRef();
      }
  };
  /** Assignment operator : release the current counted object 
      and get a new reference to the counted object
  */
  SmartPtr<T> & operator=(SmartPtr<T> const &ref)
    {
      if (!ref.Null())
	{
	  ref._counted->getRef();
	}
      unBind();
      _counted = ref._counted;
      return *this;
    };
  /** Dereferencement operator : provide consistent use of counted objects
      obj->member_of_class_T even if obj is in fact SmartPtr<T> obj;
   */
  T * operator->() 
  {
    if (!Null())
      {
	return _counted->_t;
      }
  };
  /// Const dereferencement operator
  T const * operator->() const
  {
    if (!Null())
    {
      return _counted->_t;
    }
  } ;
  /** Returns the pointer, without any control
      @doc Returns the pointer, without any control.
      Warning : the following sequence may lead to segfault !
      ##
      #AClass * my_new_ptr = new AClass();#
      #SmartPtr<AClass> reference_counted_object = new SmartPtr<my_new_ptr>;#
      #// ...#
      #AClass * my_ptr = reference_counted_object.getPtr();#
      #delete my_ptr; // freeing memory of a counted pointer ! ARGHHHH !#
      #reference_counted_object->member(); // SEGFAULT :(#
  */
  T * getPtr(void) const
  {
      return _counted->_t;
  };

  ///equality operator
  friend bool operator==(SmartPtr<T> const & lhs, SmartPtr<T> const & rhs)
    {
      bool res;
      res = false;
      if ((lhs.Null()) && (rhs.Null()))
	{
	  res = true;
	}
      else
	{
	  if ((lhs.Null()) || (rhs.Null()))
	    {
	      res = false;
	    }
	  else
	    {
	      res = (lhs._counted->_t == rhs._counted->_t);
	    }
	}
      return res;
    };
  /** Equality operator between SmartPtr<T> and string
      To be consistent, class T has to handle the equality operator with string
  */
  friend bool operator==(SmartPtr<T> const & lhs, std::string const & r_id)
    {
      bool res;
      res = false;
      if (lhs.Null())
	{
	  res = false;
	}
      else
	{
	  res = (*(lhs._counted->_t)) == r_id;
	}
      return res;
    };
  ///
 /** Equality operator between SmartPtr<T> and int
      To be consistent, class T has to handle the equality operator with int
  */
  friend bool operator==(SmartPtr<T> const & lhs, int value)
    {
      bool res;
      res = false;
      if (lhs.Null())
	{
	  res = false;
	}
      else
	{
	  res = (*(lhs._counted->_t)) == value;
	}
      return res;
    };
  friend bool operator!=(SmartPtr<T> const & lhs, SmartPtr<T> const & rhs)
    {
      return !(lhs==rhs);
    };
  /// true if the SmartPtr object does not reference any pointer
  bool Null(void) const { return _counted == 0; };
  /// frees the referenced pointer
  void setNull(void) { unBind(); };
private:
  template <class T2>
  class Counted
  {
  public:
    Counted(T2 *  ptr) : _count(0), _t(ptr) {};
    ~Counted(void) { delete _t; };
    unsigned getRef(void) { return ++_count; };
    unsigned freeRef(void) { return --_count; };
    T * operator->() { return &_t; };
    T * _t;
    unsigned _count;
  };
  void unBind(void) 
  {
    if (!Null() && _counted->freeRef() == 0)
      {
	delete _counted;
      }
    _counted = 0;
  };
  Counted<T> * _counted;
};

#endif
