#include "GEDCOMParser/Visitor/GEDCOMSerializerVisitor.hh"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include "gettext.h"
#define _(String) dgettext(PACKAGE, String)

#include <iostream>
#include "string_utility/Converter.hh"

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitAddress (GEDCOMParser::Address *
						     object)
{
  _writer->WriteText (toString < int >(_ref_level) + " ADDR",
		      object->getAddr (), _ref_level);
  _writer->WriteString (toString < int >(_ref_level + 1) + " ADR1",
			object->getAdr1 ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " ADR2",
			object->getAdr2 ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " CITY",
			object->getCity ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " STAE",
			object->getState ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " POST",
			object->getPost ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " CTRY",
			object->getCountry ());
  _writer->WriteString (toString < int >(_ref_level) + " PHON",
			object->getPhone ());
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitAssociation (GEDCOMParser::
							 Association * object)
{

  _writer->WriteStrings (toString < int >(_ref_level) + " ASSO",
			 object->getIndiXrefs ());
  _writer->WriteString (toString < int >(_ref_level) + " TYPE",
			object->getType ());
  _writer->WriteString (toString < int >(_ref_level) + " RELA",
			object->getRela ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitChangeDate (GEDCOMParser::
							ChangeDate * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " CHAN", " ");
  _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
		      object->getDateObject ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " TIME",
			object->getTime ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitChildToFamilyLink (GEDCOMParser::
							       ChildToFamilyLink
							       * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " FAMC",
			object->getFamXref ());
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " PEDI",
			 object->getPedis ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitData (GEDCOMParser::Data * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " DATA",
			object->getData ());
  _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
		      object->getDateObject ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " COPR",
			object->getCopr ());
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitEvent (GEDCOMParser::Event *
						   object)
{
  _writer->WriteString (toString < int >(_ref_level) + " TYPE",
			object->getType ());
  _writer->WriteDate (toString < int >(_ref_level) + " DATE",
		      object->getDateObject ());
  if (!object->getPlace ().Null ())
    {
      VisitPlace (object->getPlace ().getPtr ());
    }
  if (!object->getAddress ().Null ())
    {
      VisitAddress (object->getAddress ().getPtr ());
    }
  _writer->WriteString (toString < int >(_ref_level) + " AGE",
			object->getAge ());
  _writer->WriteString (toString < int >(_ref_level) + " AGNC",
			object->getAgnc ());
  _writer->WriteString (toString < int >(_ref_level) + " CAUS",
			object->getCaus ());
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
    object->getMultimediaLinks ();
  VisitMultimediaLinks (&mm_links);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitFamilyEvent (GEDCOMParser::
							 FamilyEvent * object)
{
  std::string tag = "";
  switch (object->getRawType ())
    {
    case GEDCOMParser::FamilyEvent::e_Anul:
      {
	tag = " ANUL";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Cens:
      {
	tag = " CENS";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Div:
      {
	tag = " DIV";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Divf:
      {
	tag = " DIVF";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Enga:
      {
	tag = " ENGA";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Marr:
      {
	tag = " MARR";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Marb:
      {
	tag = " MARB";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Marc:
      {
	tag = " MARC";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Marl:
      {
	tag = " MARL";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Mars:
      {
	tag = " MARS";
	break;
      }
    case GEDCOMParser::FamilyEvent::e_Even:
      {
	tag = " EVEN";
	break;
      }
    default:
      {
	tag = " EVEN";
	std::
	  cerr << _("Found FamilyEvent #") << object->getType () <<
	  _("Warning - GEDCOM writing : Event unknown (tagging as EVEN)") <<
	  std::endl;
	break;
      }
    }

  if (!object->getEvent ().Null ())
    {
      _writer->WriteString (toString < int >(_ref_level) + tag, " ");
      _ref_level++;
      VisitEvent (object->getEvent ().getPtr ());
      _ref_level--;
    }

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitFamilyRecord (GEDCOMParser::
							  FamilyRecord *
							  object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"FAM");
  GEDCOMParser::FamilyEvents_t family_event_records =
    object->getFamilyEvents ();
  _ref_level++;
  VisitFamilyEvents (&family_event_records);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " HUSB",
			object->getHusb ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " WIFE",
			object->getWife ());
  GEDCOMParser::ChildXrefs_t child_xrefs = object->getChilXrefs ();
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " CHIL",
			 child_xrefs);
  _writer->WriteString (toString < int >(_ref_level + 1) + " NCHI",
			object->getNchi ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " SUBM",
			object->getSubm ());
  _ref_level++;

  GEDCOMParser::LdsSpouseSealings_t lds_spouse_sealings =
    object->getLdsSpouseSealings ();
  VisitLdsSpouseSealings (&lds_spouse_sealings);
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
    object->getMultimediaLinks ();
  VisitMultimediaLinks (&mm_links);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitHeader (GEDCOMParser::Header *
						    object)
{
  _writer->WriteString (toString < int >(_ref_level) + " HEAD", " ");
  _writer->WriteString (toString < int >(_ref_level + 1) + " SOUR",
			object->getSour ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " VERS",
			object->getVers ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " NAME",
			object->getName ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " CORP",
			object->getCorp ());
  if (!object->getAddress ().Null ())
    {
      _ref_level = _ref_level + 3;
      object->getAddress ().getPtr ()->Accept (this);
      _ref_level = _ref_level - 3;
    }
  if (!object->getData ().Null ())
    {
      _ref_level = _ref_level + 2;
      object->getData ().getPtr ()->Accept (this);
      _ref_level = _ref_level - 2;
    }
  _writer->WriteString (toString < int >(_ref_level + 1) + " DEST",
			object->getDest ());
  _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
		      object->getDateObject ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " TIME",
			object->getTime ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " SUBM",
			object->getSubm ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " SUBN",
			object->getSubn ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " FILE",
			object->getFile ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " COPR",
			object->getCopr ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " GEDC", " ");
  _writer->WriteString (toString < int >(_ref_level + 2) + " VERS",
			object->getGedcVers ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " FORM",
			object->getGedcForm ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " CHAR",
			object->getCharset ());
  _writer->WriteString (toString < int >(_ref_level + 2) + " VERS",
			object->getCharsetVersion ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " LANG",
			object->getLang ());
  if (object->getPlaceForm () != "")
    {
      _writer->WriteString (toString < int >(_ref_level + 1) + " PLAC", "", true);
      _writer->WriteString (toString < int >(_ref_level + 2) + " FORM",
			    object->getPlaceForm ());
    }
  _writer->WriteText (toString < int >(_ref_level + 1) + " NOTE",
		      object->getNote (), _ref_level + 1);
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitIndividualAttribute (GEDCOMParser::IndividualAttribute * object)
{
  std::string tag = "";
  switch (object->getRawType ())
    {
    case GEDCOMParser::IndividualAttribute::e_Cast:
      tag = " CAST";
      break;
    case GEDCOMParser::IndividualAttribute::e_Dscr:
      tag = " DSCR";
      break;
    case GEDCOMParser::IndividualAttribute::e_Educ:
      tag = " EDUC";
      break;
    case GEDCOMParser::IndividualAttribute::e_Idno:
      tag = " IDNO";
      break;
    case GEDCOMParser::IndividualAttribute::e_Nati:
      tag = " NATI";
      break;
    case GEDCOMParser::IndividualAttribute::e_Nchi:
      tag = " NCHI";
      break;
    case GEDCOMParser::IndividualAttribute::e_Nmr:
      tag = " NMR";
      break;
    case GEDCOMParser::IndividualAttribute::e_Occu:
      tag = " OCCU";
      break;
    case GEDCOMParser::IndividualAttribute::e_Prop:
      tag = " PROP";
      break;
    case GEDCOMParser::IndividualAttribute::e_Reli:
      tag = " RELI";
      break;
    case GEDCOMParser::IndividualAttribute::e_Resi:
      tag = " RESI";
      break;
    case GEDCOMParser::IndividualAttribute::e_Ssn:
      tag = " SSN";
      break;
    case GEDCOMParser::IndividualAttribute::e_Titl:
      tag = " TITL";
      break;
    default:
      {
	tag = "";
	std::
	  cerr << _("Found IndividualAttribute #") << object->getType () <<
	  _("Warning - GEDCOM writing : Attribute unknown (tagging as EVEN)")
	  << std::endl;
	break;
      }
    }
  if (tag != "")
    {
      _writer->WriteString (toString < int >(_ref_level) + tag,
			    object->getValue ());
      if (!object->getEvent ().Null ())
	{
	  _ref_level++;
	  VisitEvent (object->getEvent ().getPtr ());
	  _ref_level--;
	}
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitIndividualEvent (GEDCOMParser::
							     IndividualEvent *
							     object)
{
  std::string tag = "";
  bool
    famc_tag =
    false;
  bool
    adop_tag =
    false;
  switch (object->getRawType ())
    {
    case GEDCOMParser::IndividualEvent::e_Birt:
      tag = " BIRT";
      famc_tag = true;
      break;
    case GEDCOMParser::IndividualEvent::e_Chr:
      tag = " CHR";
      famc_tag = true;
      break;
    case GEDCOMParser::IndividualEvent::e_Deat:
      tag = " DEAT";
      break;
    case GEDCOMParser::IndividualEvent::e_Buri:
      tag = " BURI";
      break;
    case GEDCOMParser::IndividualEvent::e_Crem:
      tag = " CREM";
      break;
    case GEDCOMParser::IndividualEvent::e_Adop:
      tag = " ADOP";
      famc_tag = true;
      adop_tag = true;
      break;
    case GEDCOMParser::IndividualEvent::e_Bapm:
      tag = " BAPM";
      break;
    case GEDCOMParser::IndividualEvent::e_Barm:
      tag = " BARM";
      break;
    case GEDCOMParser::IndividualEvent::e_Basm:
      tag = " BASM";
      break;
    case GEDCOMParser::IndividualEvent::e_Bles:
      tag = " BLES";
      break;
    case GEDCOMParser::IndividualEvent::e_Chra:
      tag = " CHRA";
      break;
    case GEDCOMParser::IndividualEvent::e_Conf:
      tag = " CONF";
      break;
    case GEDCOMParser::IndividualEvent::e_Fcom:
      tag = " FCOM";
      break;
    case GEDCOMParser::IndividualEvent::e_Ordn:
      tag = " ORDN";
      break;
    case GEDCOMParser::IndividualEvent::e_Natu:
      tag = " NATU";
      break;    
    case GEDCOMParser::IndividualEvent::e_Emig:
      tag = " EMIG";
      break;
    case GEDCOMParser::IndividualEvent::e_Immi:
      tag = " IMMI";
      break;
    case GEDCOMParser::IndividualEvent::e_Cens:
      tag = " CENS";
      break;
    case GEDCOMParser::IndividualEvent::e_Prob:
      tag = " PROB";
      break;
    case GEDCOMParser::IndividualEvent::e_Will:
      tag = " WILL";
      break;
    case GEDCOMParser::IndividualEvent::e_Grad:
      tag = " GRAD";
      break;
    case GEDCOMParser::IndividualEvent::e_Reti:
      tag = " RETI";
      break;
    case GEDCOMParser::IndividualEvent::e_Even:
      tag = " EVEN";
      break;
    default:
      {
	tag = " EVEN";
	std::
	  cerr << _("Found IndividualEvent #") << object->getType () << " "<<
	  _("Warning - GEDCOM writing : Event unknown (tagging as EVEN)") <<
	  std::endl;
	break;
      }
    }
  if (!object->getEvent ().Null ())
    {
      _writer->WriteString (toString < int >(_ref_level) + tag, " ");
      _ref_level++;
      VisitEvent (object->getEvent ().getPtr ());
      _ref_level--;
    }

  if (famc_tag)
    {
      _writer->WriteString (toString < int >(_ref_level + 1) + " FAMC",
			    object->getFamcXref ());
      if (adop_tag)
	{
	  _writer->WriteString (toString < int >(_ref_level + 2) + " ADOP",
				object->getAdop ());

	}
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitIndividualRecord (GEDCOMParser::
							      IndividualRecord
							      * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"INDI");
  _writer->WriteString (toString < int >(_ref_level + 1) + " RESN",
			object->getResn ());
  _ref_level++;
  GEDCOMParser::PersonalNames_t personal_names = object->getPersonalNames ();
  VisitPersonalNames (&personal_names);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " SEX",
			object->getSex ());
  _ref_level++;
  GEDCOMParser::IndividualEvents_t individual_events =
    object->getIndividualEvents ();
  VisitIndividualEvents (&individual_events);
  GEDCOMParser::IndividualAttributes_t individual_attributes =
    object->getIndividualAttributes ();
  VisitIndividualAttributes (&individual_attributes);
  GEDCOMParser::LdsIndividualOrdinances_t lds_individual_ordinances =
    object->getLdsIndividualOrdinances ();
  VisitLdsIndividualOrdinances (&lds_individual_ordinances);
  GEDCOMParser::ChildToFamilyLinks_t child_to_family_links =
    object->getChildToFamilyLinks ();
  VisitChildToFamilyLinks (&child_to_family_links);
  GEDCOMParser::SpouseToFamilyLinks_t spouse_to_family_links =
    object->getSpouseToFamilyLinks ();
  VisitSpouseToFamilyLinks (&spouse_to_family_links);
  _ref_level--;
  GEDCOMParser::SubmXrefs_t subm_xrefs = object->getSubmXrefs ();
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " SUBM",
			 subm_xrefs);
  _ref_level++;
  GEDCOMParser::Associations_t associations = object->getAssociations ();
  VisitAssociations (&associations);
  _ref_level--;
  GEDCOMParser::AliaXrefs_t alia_xrefs = object->getAliaXrefs ();
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " ALIA",
			 alia_xrefs);
  GEDCOMParser::AnciXrefs_t anci_xrefs = object->getAnciXrefs ();
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " ANCI",
			 anci_xrefs);
  GEDCOMParser::DesiXrefs_t desi_xrefs = object->getDesiXrefs ();
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " DESI",
			 desi_xrefs);
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
    object->getMultimediaLinks ();
  VisitMultimediaLinks (&mm_links);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RFN",
			object->getRfn ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " AFN",
			object->getAfn ());
  _ref_level++;
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitLdsIndividualOrdinance (GEDCOMParser::LdsIndividualOrdinance * object)
{
  std::string tag = "";
  bool
    famc_tag =
    false;
  switch (object->getType ())
    {
    case GEDCOMParser::LdsIndividualOrdinance::e_Bapl:
      tag = " BAPL";
      break;
    case GEDCOMParser::LdsIndividualOrdinance::e_Conl:
      tag = " CONL";
      break;
    case GEDCOMParser::LdsIndividualOrdinance::e_Endl:
      tag = " ENDL";
      break;
    case GEDCOMParser::LdsIndividualOrdinance::e_Slgc:
      tag = " SLGC";
      famc_tag = true;
      break;
    default:
      {
	tag = "";
	std::
	  cerr << _("Found LdsIndividualOrdinance #") << object->getType () <<
	  _("Warning - GEDCOM writing : Attribute unknown (tagging as EVEN)")
	  << std::endl;
	break;
      }
    }
  if (tag != "")
    {
      _writer->WriteString (toString < int >(_ref_level) + tag, " ");
      _writer->WriteString (toString < int >(_ref_level + 1) + " STAT",
			    object->getStat ());
      _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
			  object->getDateObject ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " TEMP",
			    object->getTempleCode ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " PLAC",
			    object->getPlace ());
      if (famc_tag)
	{
	  _writer->WriteString (toString < int >(_ref_level + 1) + " FAMC",
				object->getFamcXref ());
	}
      _ref_level++;
      GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
	object->getSourceCitations ();
      VisitSourceCitations (&source_citations);
      GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
	object->getNoteStructures ();
      VisitNoteStructures (&note_structures);
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitLdsSpouseSealing (GEDCOMParser::
							      LdsSpouseSealing
							      * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " SLGS", " ");
  _writer->WriteString (toString < int >(_ref_level + 1) + " STAT",
			object->getStatus ());
  _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
		      object->getDateObject ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " TEMP",
			object->getTempleCode ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " PLAC",
			object->getPlace ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitLineageLinkageGedcom (GEDCOMParser::LineageLinkageGedcom * object)
{
  _ref_level = 0;
  if (!object->getHeader ().Null ())
    {
      object->getHeader ().getPtr ()->Accept (this);
    }
  if (!object->getSubmission ().Null ())
    {
      object->getSubmission ().getPtr ()->Accept (this);
    }
  GEDCOMParser::FamilyRecords_t family_records = object->getFamilyRecords ();
  for (GEDCOMParser::FamilyRecords_t::iterator iter = family_records.begin ();
       iter != family_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::IndividualRecords_t individual_records =
    object->getIndividualRecords ();
  for (GEDCOMParser::IndividualRecords_t::iterator iter =
       individual_records.begin (); iter != individual_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::MultimediaRecords_t mm_records =
    object->getMultimediaRecords ();
  for (GEDCOMParser::MultimediaRecords_t::iterator iter = mm_records.begin ();
       iter != mm_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::NoteRecords_t note_records = object->getNoteRecords ();
  for (GEDCOMParser::NoteRecords_t::iterator iter = note_records.begin ();
       iter != note_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::RepositoryRecords_t repository_records =
    object->getRepositoryRecords ();
  for (GEDCOMParser::RepositoryRecords_t::iterator iter =
       repository_records.begin (); iter != repository_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::SourceRecords_t source_records = object->getSourceRecords ();
  for (GEDCOMParser::SourceRecords_t::iterator iter = source_records.begin ();
       iter != source_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  GEDCOMParser::SubmitterRecords_t submitter_records =
    object->getSubmitterRecords ();
  for (GEDCOMParser::SubmitterRecords_t::iterator iter =
       submitter_records.begin (); iter != submitter_records.end (); iter++)
    {
      iter->second->Accept (this);
    }
  _writer->WriteString ("0 TRLR", " ");
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitMultimediaLink (GEDCOMParser::
							    MultimediaLink *
							    object)
{
  if (object->getObjXref () != "")
    {
      _writer->WriteString (toString < int >(_ref_level) + " OBJE",
			    object->getObjXref ());
    }
  else
    {
      _writer->WriteString (toString < int >(_ref_level) + " OBJE", " ");
      _writer->WriteString (toString < int >(_ref_level + 1) + " FORM",
			    object->getFormat ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " TITL",
			    object->getTitle ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " FILE",
			    object->getMMFileRef ());
      _ref_level++;
      GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
	object->getNoteStructures ();
      VisitNoteStructures (&note_structures);
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitMultimediaRecord (GEDCOMParser::
							      MultimediaRecord
							      * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"OBJE");
  _writer->WriteString (toString < int >(_ref_level + 1) + " FORM",
			object->getForm ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " TITL",
			object->getTitl ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  _writer->WriteText (toString < int >(_ref_level + 1) + " BLOB",
		      object->getBlob (), _ref_level + 1);
  _writer->WriteString (toString < int >(_ref_level + 1) + " OBJE",
			object->getObjeXref ());
  _ref_level++;
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitNoteRecord (GEDCOMParser::
							NoteRecord * object)
{
  // FIXME : here the SubmitterText may be a long string !
  _writer->WriteText (toString < int >(_ref_level) + " " + object->getId () + " NOTE",
			object->getSubmitterText (), _ref_level);
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitNoteStructure (GEDCOMParser::
							   NoteStructure *
							   object)
{
  if (object->getNoteXref () != "")
    {
      _writer->WriteString (toString < int >(_ref_level) + " NOTE",
			    object->getNoteXref ());

    }
  else
    {
      _writer->WriteText (toString < int >(_ref_level) + " NOTE",
			  object->getSubmitterText (), _ref_level);
    }
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitPersonalName (GEDCOMParser::
							  PersonalName *
							  object)
{
  _writer->WriteString (toString < int >(_ref_level) + " NAME",
			object->getRawName ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " NPFX",
			object->getNpfx ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " GIVN",
			object->getGivn ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " NICK",
			object->getNick ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " SPFX",
			object->getSpfx ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " SURN",
			object->getSurn ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " NSFX",
			object->getNsfx ());
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitPlace (GEDCOMParser::Place *
						   object)
{
  _writer->WriteString (toString < int >(_ref_level) + " PLAC",
			object->getPlaceValue ());
  //    std::string const tmp = object->getPlaceHierarchy ();
  std::string const
    tmp =
    "";				//FIXME
  _writer->WriteString (toString < int >(_ref_level) + " FORM", tmp);
  _ref_level++;
  GEDCOMParser::AttachmentManager::SourceCitations_t source_citations =
    object->getSourceCitations ();
  VisitSourceCitations (&source_citations);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitRepositoryRecord (GEDCOMParser::
							      RepositoryRecord
							      * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"REPO");
  _writer->WriteString (toString < int >(_ref_level + 1) + " NAME",
			object->getName ());

  _ref_level++;
  if (!object->getAddress ().Null ())
    {
      object->getAddress ().getPtr ()->Accept (this);
    }
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceCitation (GEDCOMParser::
							    SourceCitation *
							    object)
{
  if (object->getSourceXref () != "")
    {
      _writer->WriteString (toString < int >(_ref_level) + " SOUR",
			    object->getSourceXref ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " PAGE",
			    object->getPage ());
      _writer->WriteString (toString < int >(_ref_level + 1) + " EVEN",
			    object->getEvent ());
      _writer->WriteString (toString < int >(_ref_level + 2) + " ROLE",
			    object->getEventRole ());
      if ((object->getDataDate () != "") || (object->getDataText () != ""))
	{
	  _writer->WriteString (toString < int >(_ref_level + 1) + " DATA",
				"",
				true);
	  _writer->WriteDate (toString < int >(_ref_level + 2) + " DATE",
				object->getDataDateObject ());
	  _writer->WriteText (toString < int >(_ref_level + 2) + " TEXT",
			      object->getDataText (), _ref_level + 2);
	}
      _writer->WriteString (toString < int >(_ref_level + 1) + " QUAY",
			    object->getQuay ());
      _ref_level++;
      GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
	object->getMultimediaLinks ();
      VisitMultimediaLinks (&mm_links);
      GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
	object->getNoteStructures ();
      VisitNoteStructures (&note_structures);
      _ref_level--;
    }
  else
    {
      _writer->WriteText (toString < int >(_ref_level) + " SOUR",
			  object->getDescription (), _ref_level);
      _writer->WriteText (toString < int >(_ref_level + 1) + " TEXT",
			  object->getText (), _ref_level + 1);
      _ref_level++;
      GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
	object->getNoteStructures ();
      VisitNoteStructures (&note_structures);
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceRecord (GEDCOMParser::
							  SourceRecord *
							  object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"SOUR");
  if (!object->getData ().Null ())
    {
      _ref_level++;
      object->getData ().getPtr ()->Accept (this);
      _ref_level--;
    }
  _writer->WriteText (toString < int >(_ref_level + 1) + " AUTH",
		      object->getAuth (), _ref_level + 1);
  _writer->WriteText (toString < int >(_ref_level + 1) + " TITL",
		      object->getTitle (), _ref_level + 1);
  _writer->WriteString (toString < int >(_ref_level + 1) + " ABBR",
			object->getAbbr ());
  _writer->WriteText (toString < int >(_ref_level + 1) + " PUBL",
		      object->getPubl (), _ref_level + 1);
  _writer->WriteText (toString < int >(_ref_level + 1) + " TEXT",
		      object->getText (), _ref_level + 1);

  if (!object->getSourceRepositoryCitation ().Null ())
    {
      _ref_level++;
      VisitSourceRepositoryCitation (object->getSourceRepositoryCitation ().
				     getPtr ());
      _ref_level--;
    }
  _ref_level++;
  GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
    object->getMultimediaLinks ();
  VisitMultimediaLinks (&mm_links);
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  GEDCOMParser::UserReferences_t user_references =
    object->getUserReferences ();
  VisitUserReferences (&user_references);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceRecordData (GEDCOMParser::
							      SourceRecord::
							      Data * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " DATA", " ");
  _ref_level++;
  GEDCOMParser::SourceRecord::Data::Events_t events = object->getEvents ();
  VisitSourceRecordDataEvents (&events);
  _ref_level--;
  _writer->WriteString (toString < int >(_ref_level) + " AGNC",
			object->getAgnc ());
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitSourceRecordDataEvent (GEDCOMParser::SourceRecord::Data::Event * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " EVEN",
			object->getEvent ());
  _writer->WriteDate (toString < int >(_ref_level + 1) + " DATE",
		      object->getDateObject ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " PLAC",
			object->getPlace ());
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::

VisitSourceRepositoryCitation (GEDCOMParser::SourceRepositoryCitation *
			       object)
{
  _writer->WriteString (toString < int >(_ref_level) + " REPO",
			object->getRepoXref ());

  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  GEDCOMParser::SourceRepositoryCitation::
    SourceCallNumbers_t source_call_numbers = object->getSourceCallNumbers ();
  VisitSourceCallNumbers (&source_call_numbers);
  _ref_level--;
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::

VisitSourceRepositoryCitationSourceCallNumber (GEDCOMParser::
					       SourceRepositoryCitation::
					       SourceCallNumber * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " CALN",
			object->getCallNumber ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " MEDI",
			object->getMedi ());
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSpouseToFamilyLink (GEDCOMParser::
								SpouseToFamilyLink
								* object)
{
  _writer->WriteString (toString < int >(_ref_level) + " FAMS",
			object->getFamXref ());

  _ref_level++;
  GEDCOMParser::AttachmentManager::NoteStructures_t note_structures =
    object->getNoteStructures ();
  VisitNoteStructures (&note_structures);
  _ref_level--;

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSubmission (GEDCOMParser::
							Submission * object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"SUBN");
  _writer->WriteString (toString < int >(_ref_level + 1) + " SUBM",
			object->getSubmXref ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " FAMF",
			object->getFamf ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " TEMP",
			object->getTemp ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " ANCE",
			object->getAnce ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " DESC",
			object->getDesc ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " ORDI",
			object->getOrdi ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSubmitterRecord (GEDCOMParser::
							     SubmitterRecord *
							     object)
{
  _writer->WriteString (toString < int >(_ref_level) + " " + object->getId (),
			"SUBM");
  _writer->WriteString (toString < int >(_ref_level + 1) + " NAME",
			object->getName ());
  if (!object->getAddress ().Null ())
    {
      _ref_level++;
      object->getAddress ().getPtr ()->Accept (this);
      _ref_level--;
    }
  _ref_level++;
  GEDCOMParser::AttachmentManager::MultimediaLinks_t mm_links =
    object->getMultimediaLinks ();
  VisitMultimediaLinks (&mm_links);
  _ref_level--;
  _writer->WriteStrings (toString < int >(_ref_level + 1) + " LANG",
			 object->getLanguagePreferences ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " RFN",
			object->getRfn ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " RIN",
			object->getRin ());
  if (!object->getChangeDate ().Null ())
    {
      _ref_level++;
      VisitChangeDate (object->getChangeDate ().getPtr ());
      _ref_level--;
    }

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitUserReference (GEDCOMParser::
							   UserReference *
							   object)
{
  _writer->WriteString (toString < int >(_ref_level) + " REFN",
			object->getNumber ());
  _writer->WriteString (toString < int >(_ref_level + 1) + " TYPE",
			object->getType ());

  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitFamilyRecords (GEDCOMParser::
							   FamilyRecords_t *
							   m)
{
  for (GEDCOMParser::FamilyRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitIndividualRecords (GEDCOMParser::
							       IndividualRecords_t
							       * m)
{
  for (GEDCOMParser::IndividualRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitMultimediaRecords (GEDCOMParser::
							       MultimediaRecords_t
							       * m)
{
  for (GEDCOMParser::MultimediaRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitNoteRecords (GEDCOMParser::
							 NoteRecords_t * m)
{
  for (GEDCOMParser::NoteRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitRepositoryRecords (GEDCOMParser::
							       RepositoryRecords_t
							       * m)
{
  for (GEDCOMParser::RepositoryRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceRecords (GEDCOMParser::
							   SourceRecords_t *
							   m)
{
  for (GEDCOMParser::SourceRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitSourceRecordDataEvents (GEDCOMParser::SourceRecord::Data::Events_t * v)
{
  for (GEDCOMParser::SourceRecord::Data::Events_t::iterator iter =
       v->begin (); iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSubmitterRecords (GEDCOMParser::
							      SubmitterRecords_t
							      * m)
{
  for (GEDCOMParser::SubmitterRecords_t::iterator iter = m->begin ();
       iter != m->end (); iter++)
    {
      iter->second->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitAssociations (GEDCOMParser::
							  Associations_t * v)
{
  for (GEDCOMParser::Associations_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitLdsSpouseSealings (GEDCOMParser::
							       LdsSpouseSealings_t
							       * v)
{
  for (GEDCOMParser::LdsSpouseSealings_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitMultimediaLinks (GEDCOMParser::
							     AttachmentManager::
							     MultimediaLinks_t
							     * v)
{
  for (GEDCOMParser::AttachmentManager::MultimediaLinks_t::iterator iter =
       v->begin (); iter != v->end (); iter++)
    {
      (*iter)->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitNoteStructures (GEDCOMParser::
							    AttachmentManager::
							    NoteStructures_t *
							    v)
{
  for (GEDCOMParser::AttachmentManager::NoteStructures_t::iterator iter =
       v->begin (); iter != v->end (); iter++)
    {
      (*iter)->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitPersonalNames (GEDCOMParser::
							   PersonalNames_t *
							   v)
{
  for (GEDCOMParser::PersonalNames_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceCitations (GEDCOMParser::
							     AttachmentManager::
							     SourceCitations_t
							     * v)
{
  for (GEDCOMParser::AttachmentManager::SourceCitations_t::iterator iter =
       v->begin (); iter != v->end (); iter++)
    {
      (*iter)->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitEvents (GEDCOMParser::
						    SourceRecord::Data::
						    Events_t * v)
{
  for (GEDCOMParser::SourceRecord::Data::Events_t::iterator iter =
       v->begin (); iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitSourceCallNumbers (GEDCOMParser::
							       SourceRepositoryCitation::
							       SourceCallNumbers_t
							       * v)
{
  for (GEDCOMParser::SourceRepositoryCitation::SourceCallNumbers_t::
       iterator iter = v->begin (); iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitSpouseToFamilyLinks (GEDCOMParser::SpouseToFamilyLinks_t * v)
{
  for (GEDCOMParser::SpouseToFamilyLinks_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitUserReferences (GEDCOMParser::
							    UserReferences_t *
							    v)
{
  for (GEDCOMParser::UserReferences_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitChildToFamilyLinks (GEDCOMParser::
								ChildToFamilyLinks_t
								* v)
{
  for (GEDCOMParser::ChildToFamilyLinks_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitFamilyEvents (GEDCOMParser::
							  FamilyEvents_t * v)
{
  for (GEDCOMParser::FamilyEvents_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::VisitIndividualEvents (GEDCOMParser::
							      IndividualEvents_t
							      * v)
{
  for (GEDCOMParser::IndividualEvents_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitIndividualAttributes (GEDCOMParser::IndividualAttributes_t * v)
{
  for (GEDCOMParser::IndividualAttributes_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}

void
GEDCOMParser::GEDCOMSerializerVisitor::
VisitLdsIndividualOrdinances (GEDCOMParser::LdsIndividualOrdinances_t * v)
{
  for (GEDCOMParser::LdsIndividualOrdinances_t::iterator iter = v->begin ();
       iter != v->end (); iter++)
    {
      (*iter).getPtr ()->Accept (this);
    }
  return;
}
