/* 
   Copyright 2002 Cyril Picard

   This file is part of the GEDCOMParser library 
   (developed within the Genealogy Free Software Tools project).

   The GEDCOMParser library is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   The GEDCOMParser library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with the GEDCOMParser library ; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _GEDCOMPARSER_INDIVIDUAL_RECORD_HH_
#define _GEDCOMPARSER_INDIVIDUAL_RECORD_HH_

#include <map>
#include <string>
#include <vector>

#include "GEDCOMParser/PersonalName.hh"
#include "GEDCOMParser/IndividualEvent.hh"
#include "GEDCOMParser/IndividualAttribute.hh"
#include "GEDCOMParser/LdsIndividualOrdinance.hh"
#include "GEDCOMParser/ChildToFamilyLink.hh"
#include "GEDCOMParser/SpouseToFamilyLink.hh"
#include "GEDCOMParser/Association.hh"
#include "GEDCOMParser/AttachmentManager.hh"
#include "GEDCOMParser/AutomatedRecordIdManager.hh"
#include "GEDCOMParser/ChangelogManager.hh"
#include "GEDCOMParser/UserReferenceManager.hh"
#include "GEDCOMParser/Visitor/GEDCOMElement.hh"

///
namespace GEDCOMParser
{
  class GEDCOMVisitor;
  ///@name typedefs
  //@{
  /// @doc handling of many submitter references
  typedef std::vector < std::string > SubmXrefs_t;
  /// @doc handling of many alias references
  typedef std::vector < std::string > AliaXrefs_t;
  /// @doc handling of many anci references
  typedef std::vector < std::string > AnciXrefs_t;
  /// @doc handling of many desi references
  typedef std::vector < std::string > DesiXrefs_t;
  //@}

  /// @memo Implements the GEDCOM 5.5 INDIVIDUAL_RECORD structure 
  class IndividualRecord:public GEDCOMElement,
    ///
    public AttachmentManager,
    public AutomatedRecordIdManager,
    public ChangelogManager, public UserReferenceManager
  {
  public:
    IndividualRecord (std::string const &id = "");
      virtual ~ IndividualRecord (void);
    static std::string const MALE;
    static std::string const FEMALE;

    ///@name Accessors (set) 
    //@{      
    void setId (std::string const &);
    ///
    void setResn (std::string const &);
    ///
    PersonalName *addPersonalName (PersonalName * const name = 0);
    ///
    void setSex (std::string const &);
    ///
    IndividualEvent *addIndividualEvent (IndividualEvent * const event = 0);
    ///
    IndividualAttribute *addIndividualAttribute (IndividualAttribute *
						 const attr = 0);
    ///
      GEDCOMParser::LdsIndividualOrdinance *
      addLdsIndividualOrdinance (LdsIndividualOrdinance * const data = 0);
    ///
      GEDCOMParser::ChildToFamilyLink *
      addChildToFamilyLink (ChildToFamilyLink * const data = 0);
    ///
    void RemoveChildToFamilyLink (SmartPtr < ChildToFamilyLink > const &data);
    ///
      GEDCOMParser::SpouseToFamilyLink *
      addSpouseToFamilyLink (SpouseToFamilyLink * const data = 0);
    ///
    void RemoveSpouseToFamilyLink (SmartPtr < SpouseToFamilyLink >
				   const &data);
    ///
    void addSubmXref (std::string const &);
    ///
      GEDCOMParser::Association * addAssociation (Association * const data =
						  0);
    ///
    void addAliaXref (std::string const &);
    ///
    void addAnciXref (std::string const &);
    ///
    void addDesiXref (std::string const &);
    ///
    void setRfn (std::string const &);
    ///
    void setAfn (std::string const &);
    ///
    //@}
    ///@name Accessors (get)
    //@{    
    virtual std::string const &getId (void) const;
    ///
    virtual std::string const &getResn (void) const;
    ///
    PersonalNames_t const &getPersonalNames (void) const;
    ///
    virtual std::string const &getSex (void) const;
    ///
    IndividualEvents_t const &getIndividualEvents (void) const;
    ///
    IndividualAttributes_t const &getIndividualAttributes (void) const;
    ///
    LdsIndividualOrdinances_t const &getLdsIndividualOrdinances (void) const;
    ///
    ChildToFamilyLinks_t const &getChildToFamilyLinks (void) const;
    ///
    SpouseToFamilyLinks_t const &getSpouseToFamilyLinks (void) const;
    ///
    SubmXrefs_t const &getSubmXrefs (void) const;
    ///
    Associations_t const &getAssociations (void) const;
    ///
    AliaXrefs_t const &getAliaXrefs (void) const;
    ///
    AnciXrefs_t const &getAnciXrefs (void) const;
    ///
    DesiXrefs_t const &getDesiXrefs (void) const;
    ///
    virtual std::string const &getRfn (void) const;
    ///
    virtual std::string const &getAfn (void) const;
    ///
    //@}

    /// operator == used to find an individual by its id
    friend bool operator== (IndividualRecord const &ind,
			    std::string const &id)
    {
      return (ind._id == id);
    };
    virtual void Accept (GEDCOMParser::GEDCOMVisitor *);

  protected:
      std::string _id;
      std::string _resn;
    PersonalNames_t _personal_names;
      std::string _sex;
    IndividualEvents_t _individual_events;
    IndividualAttributes_t _individual_attributes;
    LdsIndividualOrdinances_t _lds_individual_ordinances;
    ChildToFamilyLinks_t _child_to_family_links;
    SpouseToFamilyLinks_t _spouse_to_family_links;
    SubmXrefs_t _subm_xrefs;
    Associations_t _associations;
    AliaXrefs_t _alia_xrefs;
    AnciXrefs_t _anci_xrefs;
    DesiXrefs_t _desi_xrefs;
      std::string _rfn;
      std::string _afn;
  };

  ///@name typedefs
  //@{
  /// @doc handling of many individus (element)
  typedef std::pair < std::string,
    SmartPtr < IndividualRecord > >IndividualRecords_elem_t;
  /// @doc handling of many individus (association)
  typedef std::map < std::string,
    SmartPtr < IndividualRecord > >IndividualRecords_t;
  //@}
};
#endif
