/* 
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _GEDCOMPARSER_ADDRESS_HH_
#define _GEDCOMPARSER_ADDRESS_HH_

#include "GEDCOMParser/Visitor/GEDCOMElement.hh"

#include <string>

//@Include: ../../doc/reference/GEDCOMPARSER.desc
namespace GEDCOMParser
{
  class GEDCOMVisitor;
  /** The class GEDCOMParser::Address implements the GEDCOM 5 5 ADDRESS_STRUCTURE structure.

      {\bf Warning:} The integrity between multiples lines of Addr and other values (Adr1, Adr2, ...) is not assured by the class.
      
      {\bf GEDCOM 5.5 definition:} 
      \begin{verbatim}
      ADDRESS_STRUCTURE: =
      n  ADDR <ADDRESS_LINE>  {0:1}
      +1 CONT <ADDRESS_LINE>  {0:M}
      +1 ADR1 <ADDRESS_LINE1>  {0:1}
      +1 ADR2 <ADDRESS_LINE2>  {0:1}
      +1 CITY <ADDRESS_CITY>  {0:1}
      +1 STAE <ADDRESS_STATE>  {0:1}
      +1 POST <ADDRESS_POSTAL_CODE>  {0:1}
      +1 CTRY <ADDRESS_COUNTRY>  {0:1}
      n  PHON <PHONE_NUMBER>  {0:3}
      \end{verbatim}
      {\it  The address structure should be formed as it would appear on a mailing label using the ADDR and ADDR.CONT lines. These lines are required if an ADDRess is present. Optionally, additional structure is provided for systems that have structured their addresses for indexing and sorting. }

      @see ADDRESS_STRUCTURE
  */

  class Address:public GEDCOMElement
  {
  protected:
    std::string _addr;
    std::string _adr1;
    std::string _adr2;
    std::string _city;
    std::string _state;
    std::string _post;
    std::string _country;
    std::string _phone;
  public:
    /// Constructor
    Address (void)
    {
    };
    /// Destructor
      virtual ~Address (void)
    {
    };
    /** Fill the #ADDRESS_LINE# value. 
	This method can be invoked multiple times.
	The resulting address is the concatenation of the multiples parameters.

	{\bf Warning:} this method is an extension to the GEDCOM 5.5 format

	@see ADDRESS_LINE
     */
    void concAddr (std::string const &);
    /** Fill the #ADDRESS_LINE# value.
	This method can be invoked multiple times.
	The resulting address is the concatenation of the multiples parameters,
	separated by a newline character.
	
	@see ADDRESS_LINE
    */
    void contAddr (std::string const &);
    /** Set the #ADDRESS_LINE1# value
	
        @see ADDRESS_LINE1
     */
    void setAdr1 (std::string const &);
    /** Set the #ADDRESS_LINE2# value
	
        @see ADDRESS_LINE2
     */
    void setAdr2 (std::string const &);
    /** Set the #ADDRESS_CITY# value

        @see ADDRESS_CITY
    */
    void setCity (std::string const &);
    /** set the #ADDRESS_STATE# value
	
      @see ADDRESS_STATE
     */
    void setState (std::string const &);
    /** set the #ADDRESS_POSTAL_CODE# value

	@see ADDRESS_POSTAL_CODE
     */
    void setPost (std::string const &);
    /** set the #ADDRESS_COUNTRY# value

	@see ADDRESS_COUNTRY
    */
    void setCountry (std::string const &);
    /** set the #PHONE_NUMBER# value
	
	@see PHONE_NUMBER
     */
    void setPhone (std::string const &);
    /** get the #ADDRESS_LINE# value

        @see ADDRESS_LINE
     */
    virtual std::string const &getAddr (void) const;
    /** get the #ADDRESS_LINE1# value
	
        @see ADDRESS_LINE1
    */
    virtual std::string const &getAdr1 (void) const;
    /** get the #ADDRESS_LINE2# value
	
        @see ADDRESS_LINE2
    */
    virtual std::string const &getAdr2 (void) const;
    /** get the #ADDRESS_CITY# value
	
    @see ADDRESS_CITY
    */
    virtual std::string const &getCity (void) const;
    /** get the #ADDRESS_STATE# value
	
    @see ADDRESS_STATE
    */
    virtual std::string const &getState (void) const;
    /** get the #ADDRESS_POSTAL_CODE# value
	
    @see ADDRESS_POSTAL_CODE
    */
    virtual std::string const &getPost (void) const;
    /** get the #ADDRESS_COUNTRY# value
	
    @see ADDRESS_COUNTRY
    */
    virtual std::string const &getCountry (void) const;
    /** get the #PHONE_NUMBER# value
	
    @see PHONE_NUMBER
    */
    virtual std::string const &getPhone (void) const;
    //@Include: ../../doc/reference/ACCEPT.desc    
    virtual void Accept (GEDCOMParser::GEDCOMVisitor *);
    /** @name ADDRESS_STRUCTURE
	@doc ADDRESS_STRUCTURE values
     */
    //@{
    /** @name ADDRESS_LINE
	@doc Address information that, when combined with NAME and CONTinuation lines, meets requirements for sending communications through the mail.
    */
    /** @name ADDRESS_LINE1
	@doc The first line of the address used for indexing. This corresponds to the ADDRESS_LINE value of the ADDR line in the address structure.
    */
    /** @name ADDRESS_LINE2
	@doc The second line of the address used for indexing. This corresponds to the ADDRESS_LINE value of the first CONT line subordinate to the ADDR tag in the address structure.
    */
    /** @name ADDRESS_CITY
	@doc The name of the city used in the address. Isolated for sorting or indexing.
    */
    /** @name ADDRESS_STATE
	@doc The name of the state used in the address. Isolated for sorting or indexing.
    */
    /** @name ADDRESS_POSTAL_CODE
	@doc The ZIP or postal code used by the various localities in handling of mail. Isolated for sorting or indexing.
    */
    /** @name ADDRESS_COUNTRY
	@doc The name of the country that pertains to the associated address. Isolated by some systems for sorting or indexing. Used in most cases to facilitate automatic sorting of mail.
    */
    /** @name PHONE_NUMBER
	@doc  A phone number.
    */
    //@}
    
  };
};

#endif
