/** 
    Copyright 2003 Cyril Picard

    This file is part of the GEDCOMViewer tool 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMViewer tool is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMViewer tool is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMViewer tool ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

**/

#include "BakeryGateway/ViewFamilyRecord.hh"
#include "widgets/ButtonIndividualFactory.hh"
#include "widgets/LabelIndividualFactory.hh"
#include "BakeryGateway/ViewIndividualRecordTargetFactory.hh"
#include "BakeryGateway/ViewIndividualRecord.hh"
#include "BakeryGateway/View.hh"
#include "BakeryGateway/PredicateMenuItemsFactory.hh"
#include "BakeryGateway/ViewEvent.hh"
#include "GEDCOMParser/Predicates/GenericPredicateIndividualRecords.hh"
#include "GEDCOMParser/Predicates/PredicateIndividualRecords.hh"
#include "GEDCOMParser/Predicates/IsMale.hh"
#include "GEDCOMParser/Predicates/HasChild.hh"
#include "GEDCOMParser/Predicates/NameEqualsTo.hh"
#include "GEDCOMParser/Predicates/HasParent.hh"
#include "GEDCOMParser/Predicates/True.hh"
#include "GEDCOMHelper/IndividualRecordHelper.hh"
#include "GEDCOMHelper/FamilyRecordHelper.hh"

BakeryGateway::ViewFamilyRecord::ViewFamilyRecord(MMIFamilyRecord * mmi) : 
  _mmi(mmi),
  _default_target_view(0),
  _static_individual_id(),
  _dynamic_widget_individual_factory(0),
  _static_widget_individual_factory(0),
  _father_predicates(0),
  _mother_predicates(0),
  _child_predicates(0)
{
  _dynamic_widget_individual_factory = new ButtonIndividualFactory();
  _static_widget_individual_factory = new LabelIndividualFactory();
  return;
}

BakeryGateway::ViewFamilyRecord::~ViewFamilyRecord(void)
{
  delete _dynamic_widget_individual_factory; _dynamic_widget_individual_factory = 0;
  delete _static_widget_individual_factory; _static_widget_individual_factory = 0;
  delete _father_predicates; _father_predicates = 0;
  delete _mother_predicates; _mother_predicates = 0;
  delete _child_predicates; _child_predicates = 0;
  return;
}

void BakeryGateway::ViewFamilyRecord::load_from_document() 
{
  if (SubscribeesAreLoadable())
    {
      delete_subscribed_views();
      _mmi->Clear();
      if (_parent_entity != 0)
	{
	  std::string const &fam_id = _parent_entity->getFamXref();
	  GEDCOMParser::FamilyRecords_t families = get_document()->getFamilyRecords();
	  GEDCOMParser::FamilyRecords_t::const_iterator family_iter = families.find(fam_id);
	  if (family_iter != families.end())
	    {
	      GEDCOMParser::FamilyRecord * const family_record = (*family_iter).second.getPtr();
	      WidgetIndividual * widget_individual = 0;
	      GEDCOMParser::IndividualRecords_t const & individuals = get_document()->getIndividualRecords();
	      GEDCOMParser::IndividualRecords_t::const_iterator individual_iter = individuals.find(_static_individual_id);
	      GEDCOMParser::IndividualRecord const * base_individual_ptr = 0;
	      if (individual_iter != individuals.end())
		{
		  base_individual_ptr = (*individual_iter).second.getPtr();
		}
	      
	      std::string individual_id = family_record->getHusb();
	      WidgetIndividualFactory * factory = 0;
	      factory = get_factory_widget(individual_id);
	      widget_individual = _mmi->addHusband(factory);
	      manage_individual(individuals, individual_id, widget_individual);    
	      PredicateMenuItemsFactory::IndividualRecordPredicates_t * predicates = get_father_predicates(base_individual_ptr);
	      manage_menu_items(individuals, widget_individual, predicates);
	      widget_individual->unlinked().connect(slot(this, &ViewFamilyRecord::husband_unlinked));
	      widget_individual->newed().connect(slot(this, &ViewFamilyRecord::husband_newed));
	      widget_individual->changed().connect(slot(this, &ViewFamilyRecord::husband_changed));
	      individual_id = family_record->getWife();
	      factory = get_factory_widget(individual_id);
	      widget_individual = _mmi->addWife(factory);
	      manage_individual(individuals, individual_id, widget_individual);
	      predicates = get_mother_predicates();
	      manage_menu_items(individuals, widget_individual, predicates);
	      widget_individual->unlinked().connect(slot(this, &ViewFamilyRecord::wife_unlinked));
	      widget_individual->newed().connect(slot(this, &ViewFamilyRecord::wife_newed));
	      widget_individual->changed().connect(slot(this, &ViewFamilyRecord::wife_changed));
	      individual_id = family_record->getHusb();
	      base_individual_ptr = 0;
	      if (individual_id != "")
		{
		  individual_iter = individuals.find(individual_id);
		  if (individual_iter != individuals.end())
		    {
		      base_individual_ptr = (*individual_iter).second.getPtr();
		    }
		}
	      predicates = get_child_predicates(base_individual_ptr);
	      GEDCOMParser::ChildXrefs_t childs = family_record->getChilXrefs();
	      int child_index = 0;
	      for (GEDCOMParser::ChildXrefs_t::const_iterator child_iter = childs.begin(); child_iter != childs.end(); child_iter++)
		{
		  individual_id = *child_iter;
		  factory = get_factory_widget(individual_id);
		  widget_individual = _mmi->addChild(factory);
		  manage_individual(individuals, individual_id, widget_individual);
		  manage_menu_items(individuals, widget_individual, predicates);
		  widget_individual->unlinked().connect(bind(slot(this, &ViewFamilyRecord::child_unlinked),child_index));
		  widget_individual->newed().connect(slot(this, &ViewFamilyRecord::child_newed));
		  widget_individual->changed().connect(slot(this, &ViewFamilyRecord::child_changed));
		  child_index++;
		}
	      // add of a widget to be able to add a new child
	      widget_individual = _mmi->addChild(_dynamic_widget_individual_factory);
	      manage_menu_items(individuals, widget_individual, predicates);
	      widget_individual->unlinked().connect(bind(slot(this, &ViewFamilyRecord::child_unlinked),child_index));
	      widget_individual->newed().connect(slot(this, &ViewFamilyRecord::child_newed));
	      widget_individual->changed().connect(slot(this, &ViewFamilyRecord::child_changed));
	      // management of event
	      MMIEvent * mmi_event = _mmi->addUnionEvent();
	      mmi_event->setEventLabel(_("Union"));
	      BakeryGateway::ViewUnionEvent * event_view = new BakeryGateway::ViewUnionEvent(mmi_event);
	      subscribe_view(event_view);
	      event_view->set_parent_entity(family_record);
	    }
	}
    }
  return;
}

void BakeryGateway::ViewFamilyRecord::setStaticIndividualId(std::string const &individual_id)
{
  _static_individual_id = individual_id;
  return;
}

void BakeryGateway::ViewFamilyRecord::setDefaultTargetView(BakeryGateway::ViewIndividualRecord * target_view)
{
  _default_target_view = target_view;
  if (_default_target_view != 0)
    {
      BakeryGateway::ViewIndividualRecordTargetFactory * target_factory = new BakeryGateway::ViewIndividualRecordTargetFactory();
      target_factory->setDefaultTarget(_default_target_view);
      setTargetFactory(target_factory);
    }
  return;
}

WidgetIndividualFactory * BakeryGateway::ViewFamilyRecord::get_factory_widget(std::string const &individual_id)
{
  WidgetIndividualFactory * res = 0;
  if (individual_id == _static_individual_id)
    {
      res = _static_widget_individual_factory;
    }
  else
    {
      res = _dynamic_widget_individual_factory;
    }
  return res;
}

void BakeryGateway::ViewFamilyRecord::manage_individual(GEDCOMParser::IndividualRecords_t const &individuals, std::string const &individual_id, WidgetIndividual * widget)
{
  widget->setValue(get_name_individual(individuals, individual_id), individual_id);
  widget->navigation_asked().connect(SigC::slot(this, &BakeryGateway::ViewFamilyRecord::manage_navigation));
  return;
}

PredicateMenuItemsFactory::IndividualRecordPredicates_t * const BakeryGateway::ViewFamilyRecord::get_father_predicates(GEDCOMParser::IndividualRecord const * const individual)
{
  if (_father_predicates == 0)
    {
      _father_predicates = new PredicateMenuItemsFactory::IndividualRecordPredicates_t();
      GEDCOMParser::LineageLinkageGedcom const * const lineage = get_document();
      SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords > predicate;
      if (individual != 0)
	{
	  std::string const name = individual->getPersonalNames()[0]->getSurn();
	  predicate = GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::compose2(std::logical_and<bool>(), std::compose1(std::logical_not<bool>(), std::bind1st(GEDCOMParser::Predicates::HasChild(),lineage)), std::compose2(std::logical_and<bool>(), GEDCOMParser::Predicates::IsMale(), std::bind1st(GEDCOMParser::Predicates::NameEqualsTo(), name))));
	  _father_predicates->push_back(predicate);
	  
	  predicate = SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords >(GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::compose2(std::logical_and<bool>(),GEDCOMParser::Predicates::IsMale(),std::bind1st(GEDCOMParser::Predicates::NameEqualsTo(), name))));
	  _father_predicates->push_back(predicate);
	}
      predicate = SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords >(new GEDCOMParser::Predicates::IsMale());
      _father_predicates->push_back(predicate);
    }
  return _father_predicates;
}

PredicateMenuItemsFactory::IndividualRecordPredicates_t * const BakeryGateway::ViewFamilyRecord::get_mother_predicates(void)
{
  if (_mother_predicates == 0)
    {
      _mother_predicates = new PredicateMenuItemsFactory::IndividualRecordPredicates_t();
      GEDCOMParser::LineageLinkageGedcom const * const lineage = get_document();
      SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords > predicate(GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::compose2(std::logical_and<bool>(),std::compose1(std::logical_not<bool>(),std::bind1st(GEDCOMParser::Predicates::HasChild(),lineage)),std::compose1(std::logical_not<bool>(),GEDCOMParser::Predicates::IsMale()))));
      _mother_predicates->push_back(predicate);
      predicate = SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords >(GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::compose1(std::logical_not < bool > (), GEDCOMParser::Predicates::IsMale())));
      _mother_predicates->push_back(predicate);
    }
  return _mother_predicates;
}

PredicateMenuItemsFactory::IndividualRecordPredicates_t * const BakeryGateway::ViewFamilyRecord::get_child_predicates(GEDCOMParser::IndividualRecord const * const individual)
  // returns firstly the individual records with the same name as the parameter, without parents
  // then the individual records with the same name as the parameter
  // then all the individual records
{
  if (_child_predicates == 0)
    {
      _child_predicates = new PredicateMenuItemsFactory::IndividualRecordPredicates_t();
      GEDCOMParser::LineageLinkageGedcom const * const lineage = get_document();
      SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords > predicate;
      if (individual != 0)
	{
	  std::string const name = individual->getPersonalNames()[0]->getSurn();
	  predicate = GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::compose2(std::logical_and<bool>(), std::compose1(std::logical_not<bool>(), std::bind1st(GEDCOMParser::Predicates::HasParent(),lineage)), std::bind1st(GEDCOMParser::Predicates::NameEqualsTo(), name)));
	  _child_predicates->push_back(predicate);
	  
	  predicate = SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords >(GEDCOMParser::Predicates::getGenericPredicateIndividualRecords(std::bind1st(GEDCOMParser::Predicates::NameEqualsTo(), name)));
	  _child_predicates->push_back(predicate);
	}
      predicate = SmartPtr < GEDCOMParser::Predicates::PredicateIndividualRecords >(new GEDCOMParser::Predicates::True());
      _child_predicates->push_back(predicate);
    }
  return _child_predicates;
}


std::string BakeryGateway::ViewFamilyRecord::get_name_individual(GEDCOMParser::IndividualRecords_t const &individuals, std::string const &individual_id) const
{
  std::string individual_name;
  GEDCOMParser::IndividualRecords_t::const_iterator iter = individuals.find(individual_id);
  if (iter != individuals.end())
    {
      if ((*iter).second->getPersonalNames().size() > 0)
	{
	  if (!((*iter).second->getPersonalNames()[0]).Null())
	    {
	      individual_name = (*iter).second->getPersonalNames()[0]->getName();
	    }
	}
    }
  return individual_name;
}

WidgetIndividual::NavigationReturnHandler_t BakeryGateway::ViewFamilyRecord::manage_navigation(std::string target, bool new_win)
{
  Navigate(target, new_win);
  return 1;
}

void BakeryGateway::ViewFamilyRecord::manage_menu_items(GEDCOMParser::IndividualRecords_t const &individuals, WidgetIndividual * const widget, PredicateMenuItemsFactory::IndividualRecordPredicates_t * const predicates)
{
  PredicateMenuItemsFactory * predicate_menu_factory = new PredicateMenuItemsFactory();
  predicate_menu_factory->setIndividualRecords(individuals);
  predicate_menu_factory->setPredicates(predicates);
  widget->setLinkMenuItemsFactory(predicate_menu_factory);
  return;
}

GEDCOMParser::FamilyRecord * const BakeryGateway::ViewFamilyRecord::get_parent_family_record(void) const
{
  GEDCOMParser::FamilyRecord * res = 0;

  std::string const &fam_id = _parent_entity->getFamXref();
  GEDCOMParser::FamilyRecords_t const &families = get_document()->getFamilyRecords();
  GEDCOMParser::FamilyRecords_t::const_iterator family_record_iter = families.find(fam_id);
  if (family_record_iter != families.end())
    {
      res = family_record_iter->second.getPtr();
    }
  return res;
}

WidgetIndividual::ChangeReturnHandler_t BakeryGateway::ViewFamilyRecord::husband_changed(std::string new_id)
{
  spouse_changed(new_id);
  return 1;
}

WidgetIndividual::ChangeReturnHandler_t BakeryGateway::ViewFamilyRecord::wife_changed(std::string new_id)
{
  spouse_changed(new_id);
  return 1;
}

WidgetIndividual::ChangeReturnHandler_t BakeryGateway::ViewFamilyRecord::spouse_changed(std::string const &new_id)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      GEDCOMParser::IndividualRecords_t const &individual_records = get_document()->getIndividualRecords();
      GEDCOMParser::IndividualRecords_t::const_iterator individual_iter = individual_records.find(new_id);
      GEDCOMHelper::IndividualRecordHelper helper(individual_iter->second.getPtr(), get_document());
      helper.setSpouseInFamily(family_record);
    }
  get_document()->set_modified(true);
  return 0;
}

WidgetIndividual::ChangeReturnHandler_t BakeryGateway::ViewFamilyRecord::child_changed(std::string new_id)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      GEDCOMParser::IndividualRecords_t const &individual_records = get_document()->getIndividualRecords();
      GEDCOMParser::IndividualRecords_t::const_iterator individual_iter = individual_records.find(new_id);
      GEDCOMHelper::IndividualRecordHelper helper(individual_iter->second.getPtr(), get_document());
      helper.setChildInFamily(family_record);
    }
  get_document()->set_modified(true);
  return 0;
}

WidgetIndividual::NewReturnHandler_t BakeryGateway::ViewFamilyRecord::husband_newed(void)
{
  spouse_newed(GEDCOMParser::IndividualRecord::MALE);
  return 0;
}

WidgetIndividual::NewReturnHandler_t BakeryGateway::ViewFamilyRecord::wife_newed(void)
{
  spouse_newed(GEDCOMParser::IndividualRecord::FEMALE);
  return 0;
}

WidgetIndividual::NewReturnHandler_t BakeryGateway::ViewFamilyRecord::spouse_newed(std::string const &sex)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      GEDCOMParser::IndividualRecord * spouse = get_document()->addIndividualRecord();
      spouse->setSex(sex);
      GEDCOMHelper::IndividualRecordHelper helper(spouse, get_document());
      helper.setPersonalName("", "NEW INDIVIDU");
      helper.setSpouseInFamily(family_record);
      get_document()->set_modified(true);
    }
  return 0;
}
  
WidgetIndividual::NewReturnHandler_t BakeryGateway::ViewFamilyRecord::child_newed(void)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      GEDCOMParser::IndividualRecord * child = get_document()->addIndividualRecord();
      GEDCOMHelper::IndividualRecordHelper helper(child, get_document());
      helper.setPersonalName("", "NEW INDIVIDU");
      helper.setChildInFamily(family_record);
      get_document()->set_modified(true);
    }
  return 0;
}

WidgetIndividual::UnlinkReturnHandler_t BakeryGateway::ViewFamilyRecord::husband_unlinked(void)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      std::string const &individu_id = family_record->getHusb();
      spouse_unlinked(individu_id, family_record);
    }
  return 0;
}

WidgetIndividual::UnlinkReturnHandler_t BakeryGateway::ViewFamilyRecord::spouse_unlinked(std::string const &spouse_id, GEDCOMParser::FamilyRecord * const family_record)
{
  _is_loadable = true;
  GEDCOMParser::IndividualRecords_t const &individual_records = get_document()->getIndividualRecords();
  GEDCOMParser::IndividualRecords_t::const_iterator individual_iter = individual_records.find(spouse_id);
  GEDCOMHelper::IndividualRecordHelper helper(individual_iter->second.getPtr(), get_document());
  helper.RemoveSpouseFromFamily(family_record);
  get_document()->set_modified(true);
  return 0;
}

WidgetIndividual::UnlinkReturnHandler_t BakeryGateway::ViewFamilyRecord::wife_unlinked(void)
{
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      std::string const &individu_id = family_record->getWife();
      spouse_unlinked(individu_id, family_record);
    }
  return 0;
}

WidgetIndividual::UnlinkReturnHandler_t BakeryGateway::ViewFamilyRecord::child_unlinked(int child_index)
{
  _is_loadable = true;
  GEDCOMParser::FamilyRecord * const family_record = get_parent_family_record();
  if (family_record != 0)
    {
      std::string const &child_id = family_record->getChilXrefs()[child_index];
      GEDCOMParser::IndividualRecords_t const &individual_records = get_document()->getIndividualRecords();
      GEDCOMParser::IndividualRecords_t::const_iterator child_iter = individual_records.find(child_id);
  GEDCOMHelper::IndividualRecordHelper helper(child_iter->second.getPtr(), get_document());
      helper.RemoveChildFromFamily(family_record);
      get_document()->set_modified(true);
    }
  return 0;
}

