# -*- coding: utf-8 -*-
# Copyright (C) 2008, 2009 Leonardo Gastón De Luca leo[at]kde.org.ar
# Copyright (C) 2010 Osama Khalid osamak[at]gnu.org
# Copyright (C) 2010, 2011, 2012  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Based on youogg's plugins.py file from SVN revision 50.


"""
GetMediumURL plugin for YouTube videos.
"""


from __future__ import unicode_literals

import re

from urlreader.propcache import cachedproperty

from getmediumurl.compat import parse_qs, unquote
from getmediumurl.format import Format
from getmediumurl.htmlplugin import HTMLMedium, HTMLPlugin


__all__ = ("YouTube",)


class YouTubeMedium(HTMLMedium):

    """Data of a video on YouTube."""

    raw_url_format = "http://www.youtube.com/watch?v=%s&skipcontrinter=1"
    url_format = "http://www.youtube.com/watch?v=%s"
    website_name = "YouTube"

    @cachedproperty
    def _flashvars(self):
        """A dictionary of variables used by the site's Flash player."""
        data = self.get_content_substring(start='flashvars=\\"', end='\"')
        return parse_qs(unquote(data))

    @property
    def thumbnail(self):
        """Thumbnail image URL."""
        return "http://i3.ytimg.com/vi/%s/default.jpg" % self.mediumid

    @property
    def language(self):
        """Video language."""
        assert self
        # Not sure how this could be obtained.
        return ""

    @cachedproperty
    def author_name(self):
        """Return submitter's user name."""
        return self.page.xpath("//p[@id='watch-uploader-info']"
                               "/a/text()")[0]

    @cachedproperty
    def author_url(self):
        """Return submitter's profile URL."""
        return self.make_absolute(self.page.xpath("//p[@id='watch-uploader-"
                                                  "info']/a/@href")[0])

    def __iter__(self):
        """Get data about formats of the video."""
        fmt_url_map = unquote(self.get_content_substring(\
                start="url_encoded_fmt_stream_map=", end="\\u0026amp;"))
        fmt_list = self._flashvars["fmt_list"][0]
        fmts = {}
        for fmt in fmt_list.split(","):
            parts = fmt.split("/")
            width, height = (int(x) for x in parts[1].split("x"))
            fmts[parts[0]] = (width, height)
        old_codes = set()
        for fmt in fmt_url_map.split(","):
            fmt = parse_qs(fmt)
            code = fmt["itag"][0]
            url = fmt["url"][0]
            mime_type = fmt["type"][0]
            if code in old_codes:
                continue
            old_codes.add(code)
            yield Format(url, mime_type, *fmts[code])


class YouTube(HTMLPlugin):

    """Plugin to get YouTube video URLs."""

    match_re = re.compile(r'^https?://(?:www\.|.{2}\.)?youtube\.com/' \
                              '(?:watch(?:_popup)?\?v=|v/|embed/)' \
                              '(?P<mediumid>.{11})')
    raw_url_format = "http://www.youtube.com/watch?v=%s&skipcontrinter=1"
    high_quality = True
    medium_class = YouTubeMedium
