# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test functions used by `getmediumurl.plugin.Plugin.prepare_match_re`.
"""


import unittest

from getmediumurl.utils import escape_regex, check_single_percent_s


__all__ = ("EscapeRegexTestCase", "CheckSinglePercentSTestCase")


class EscapeRegexTestCase(unittest.TestCase):

    """Test `getmediumurl.utils.escape_regex`."""

    def test_some_regex(self):
        """Test escape_regex"""
        for string, regex in (
            ("plain text", "plain text"),
            (r"\verb|plain text| in \LaTeX",
             r"\\verb\|plain text\| in \\LaTeX"),
            ("(group)?", r"\(group\)\?"),
            ("example.com", r"example\.com"),
            ("$$typing equations cost$$", r"\$\$typing equations cost\$\$"),
            ):
            self.assertEqual(escape_regex(string), regex)


class CheckSinglePercentSTestCase(unittest.TestCase):

    """Test `getmediumurl.utils.check_single_percent_s`."""

    def test_other_types(self):
        """Test that non-strings don't have %s"""
        for obj in (
            (lambda x: lambda y: x),
            (0, 1),
            None,
            True,
            False,
            set(),
            ("a", "text", "with", "%s"),
            ):
            self.assert_(not check_single_percent_s(obj))

    def test_empty(self):
        """Test that empty string doesn't have %s"""
        self.assert_(not check_single_percent_s(""))

    def test_nonempty(self):
        """Test some strings without %s"""
        for string in (
            "a string",
            "Colourless green ideas furiously sleep.",
            "fix f = f . fix $ f",
            ):
            self.assert_(not check_single_percent_s(string))

    def test_final_percent(self):
        """Test strings with final % but not %%"""
        for string in (
            "100%",
            "unformatted %",
            "%%%",
            "10%s0%",
            "%sunformatted %",
            "%s%%%",
            ):
            self.assert_(not check_single_percent_s(string))

    def test_final_double_percent(self):
        """Test strings with final %%"""
        for string in (
            "%s%%%%%%%%%%%%%%%%%%%%%%%%",
            "percent is %sformatted by %%",
            "%%%s%%",
            ):
            self.assert_(check_single_percent_s(string))

    def test_double_percent_s(self):
        """Test strings with multiple %s"""
        for string in (
            "%s%%%%%%%%%%%%%%%%%%%%%%%%%s",
            "%spercent is %sformatted by %%",
            "%%%s%s%%",
            ):
            self.assert_(not check_single_percent_s(string))

    def test_other_percent_chars(self):
        """Test strings with other characters after %"""
        for string in (
            "%s%%%%%%%%%%%%%%%%%%%%%%%%%p",
            "percent is %s%formatted by %%",
            "%%%s%%%x",
            ):
            self.assert_(not check_single_percent_s(string))
