# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Plugin for sites using ``xmoov-flv-player``."""


from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.format import SimpleFormat


__all__ = ("XmoovFLVPlayer",)


def _ascii(string):
    """Convert an ASCII `string` to a ``bytes`` object."""
    try:
        return bytes(string, "ascii")
    except (NameError, TypeError):  # Python 2
        return string


class XmoovFLVMedium(HTMLMedium):

    """A video using ``xmoov-flv-player``."""

    url_format = "%s"

    @property
    def website(self):
        """Website name."""
        return self.url.split(_ascii("/"))[2]

    @property
    def title(self):
        """Video title."""
        title = self.get_content_substring('"video_title","', '");')
        if title is not None:
            return title.replace(_ascii("+"), _ascii(" "))
        else:
            return super(XmoovFLVMedium, self).title

    def __iter__(self):
        """Iterate formats."""
        url = self.get_content_substring('"file","', '");') \
            or self.get_content_substring('"file", "', '");') \
            or self.get_content_substring('"url", "', '.flv')
        yield SimpleFormat(url=url)

    @property
    def thumbnail(self):
        """Video thumbnail URL."""
        return self.get_content_substring('thumbnail","', '");') \
            or super(XmoovFLVMedium, self).thumbnail


class XmoovFLVPlayer(HTMLPlugin):

    """Plugin for sites using ``xmoov-flv-player``."""

    url_format = "%s"
    medium_class = XmoovFLVMedium

    @classmethod
    def match(cls, url, matcher):
        """Match URL to the plugin."""
        try:
            content = matcher.urlreader(url).content
        except ValueError:
            return None
        if not hasattr(content, "find"):
            return None
        if content.find(_ascii("xmoov-flv-player")) == -1:
            return None
        else:
            return cls(url, matcher)
