# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Plugin for media in local files.
"""


import os.path
import base64
import random
from urlparse import urlparse
from subprocess import Popen, PIPE

from getmediumurl.cache import cachedproperty
from getmediumurl.plugin import Plugin
from getmediumurl.medium import Medium
from getmediumurl.format import SimpleFormat
from getmediumurl.getsubstring import get_substring


__all__ = ("File",)


class FileMedium(Medium):

    """Medium in a file."""

    url_format = "%s"

    @cachedproperty
    def _ffprobe_output(self):
        """Output from the ``ffprobe`` command."""
        try:
            process = Popen(("ffprobe", "-show_format", "-show_streams",
                             self.mediumid), stdout=PIPE, stderr=PIPE)
        except OSError:
            return None
        return process.communicate()[0]

    @property
    def url(self):
        """File URL."""
        return "file://" + self.mediumid

    @cachedproperty
    def title(self):
        """Return medium title."""
        title = get_substring(self._ffprobe_output,
                              start="TAG:title=", end="\n",
                              include_start=False, include_end=False)
        if title is None:
            title = get_substring(self._ffprobe_output,
                                  start="TAG:TITLE=", end="\n",
                                  include_start=False, include_end=False)
        return title

    @cachedproperty
    def thumbnail(self):
        """Medium thumbnail URL."""
        duration = get_substring(self._ffprobe_output,
                                 start="duration=", end="\n",
                                 include_start=False, include_end=False)
        frame = random.randrange(int(duration))
        # TODO: other size?
        try:
            process = Popen(("ffmpeg", "-i", self.mediumid,
                             "-s", "320x240",
                             "-f", "image2pipe",
                             "-vframes", "1",
                             "-ss", str(frame)),
                            stdout=PIPE, stderr=PIPE)
        except OSError:
            return None
        stdout = process.communicate()[0]
        return "data:image/jpeg;base64,%s" % base64.b64encode(stdout)

    def __iter__(self):
        """Iterate formats."""
        yield SimpleFormat(url=self.url)


class File(Plugin):

    """Plugin for media in local files.

    Files must be specified by ``file:`` URLs or local paths.
    Absolute path is used as `mediumid`.

    """

    def __iter__(self):
        """Iterate media."""
        yield FileMedium(self)

    @classmethod
    def match(cls, url, matcher):
        """Match `url` to plugin."""
        # Make path from URL.
        url = urlparse(url, "file")
        if url.scheme != "file":
            return None
        else:
            mediumid = url.path
        if not os.path.exists(mediumid):
            return None
        return cls(os.path.abspath(mediumid), matcher)
