# -*- coding: utf-8 -*-
# Copyright (C) 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
A plugin to find media linked to by an Atom feed.
"""


from getmediumurl.plugin import Plugin
from getmediumurl.xmlhtml import read_xml


__all__ = ("AtomFeed",)


class AtomFeed(Plugin):

    """A plugin getting media linked to by an Atom feed."""

    def __iter__(self):
        """Iterate media."""
        reader = self.matcher.urlreader(self.mediumid)
        doc = read_xml(reader.content)
        nses = {"a": "http://www.w3.org/2005/Atom",
                "m": "http://search.yahoo.com/mrss/"}
        for element in doc.xpath("a:entry", namespaces=nses):
            url = element.xpath("a:link[@rel='alternate' "
                                "and @type='text/html']/@href",
                                namespaces=nses)[0]
            plugin = self.matcher.match(url)
            if plugin is None:
                continue
            for medium in plugin:
                medium.title = element.xpath("a:title[@type='text']/text()",
                                             namespaces=nses)[0]
                try:
                    medium.description = element.xpath("a:content"
                                                       "[@type='text']"
                                                       "/text()",
                                                       namespaces=nses)[0]
                except IndexError:
                    medium.description = None
                medium.author_name = element.xpath("a:author/a:name/text()",
                                                   namespaces=nses)[0]
                medium.author_url = element.xpath("a:author/a:uri/text()",
                                                  namespaces=nses)[0]
                thumbnails = element.xpath("m:group/m:thumbnail",
                                           namespaces=nses)
                # Choose last thumbnail of the same time as the first
                # one.
                if thumbnails:
                    time = thumbnails[0].get("time")
                    chosen = thumbnails[0]
                    for tmb in thumbnails:
                        if tmb.get("time") == time:
                            chosen = tmb
                    try:
                        medium.thumbnail = chosen.get("url")
                    except AttributeError:
                        # Some plugins don't use cacheproperty for
                        # thumbnails.  Assume that they don't fetch
                        # any page for it.
                        assert True
                yield medium

    @classmethod
    def match(cls, url, matcher):
        """Match every Atom feed."""
        try:
            reader = matcher.urlreader(url)
        except ValueError:
            return None
        if not reader.content_type.startswith("application/atom+xml"):
            return None
        return cls(url, matcher)
