# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Useful functions not related to the task of getting medium URL.
"""


import logging
import warnings
import urllib
import urlparse

from getmediumurl.compat import parse_qs, NullHandler


__all__ = ("deprecated", "LOGGER", "escape_regex", "check_single_percent_s",
           "parse_urlpart")


def deprecated(message):
    """Warn on use of deprecated function."""
    warnings.warn(message, DeprecationWarning, stacklevel=3)


LOGGER = logging.getLogger("getmediumurl")
"""
A logger of the `logging` package.
"""


LOGGER.addHandler(NullHandler())


_REGEX_SPECIAL = frozenset(r".^$*+?{}\[]|()")
"""
A `frozenset` of characters having special meaning in regexes.
"""


def escape_regex(string):
    """Escape characters in `string` with special meaning in regexes."""
    return "".join(("\\" if char in _REGEX_SPECIAL else "") + char
                   for char in string)


def check_single_percent_s(string):
    """Check if `string` is formatted by single ``%s``.

    If `string` is a string containing exactly single ``%s``, with all
    other percent characters replaced by ``%%``, then `True` is
    returned.  Otherwise `False` is returned.
    """
    if not isinstance(string, basestring):
        return False
    i = 0
    length = len(string)
    was_percent_s = False
    while i < length:
        i = string.find("%", i)
        if i == -1:
            break
        elif i == length - 1:
            return False
        elif string[i + 1] == "%":
            i += 2
        elif string[i + 1] == "s":
            if not was_percent_s:
                was_percent_s = True
                i += 2
            else:
                return False
        else:
            return False
    return was_percent_s


def parse_urlpart(url, part):
    """Parse `part` of an `url` into a dictionary."""
    return parse_qs(urllib.unquote(getattr(urlparse.urlparse(url), part)))
