# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Shared plugin code.
"""


import re

from getmediumurl.utils import escape_regex, check_single_percent_s
from getmediumurl.clsutils import overrides
from getmediumurl.format import PluginFormat


__all__ = ("Plugin",)


class Plugin(object):

    """
    Common base class of GetMediumURL plugins.

    The `match_re` and `url_format` class attributes should be
    overridden in subclasses.  To be useful, plugins must override
    also the `get_file_url` method.
    """

    match_re = None
    """
    Regular expression matched on URL to determine if the plugin can
    be used for this URL, with `mediumid` group passed to the
    constructor.
    """

    url_format = None
    """
    A string where ``%s`` will be replace by `mediumid` to make the
    URL matched by `match_re`.
    """

    website_name = None
    """
    A string returned by default `website` if not `None`.

    By default the class name is used.
    """

    high_quality = False
    """
    An attribute set to `True` or `False` to use default `has_high_quality`.
    """

    def __init__(self, mediumid, urlreader):
        """Make a plugin object for use with specific medium.

        :Parameters:
          mediumid
            a plugin-specific string identifying the specific medium
          urlreader
            a subclass of `getmediumurl.reader.URLReader` used by the
            plugin
        """
        self.mediumid = mediumid
        """
        A string of plugin-specific format, with the plugin class it
        can uniquely determine medium.
        """
        self.urlreader = urlreader
        """
        A subclass of `getmediumurl.reader.URLReader`.
        """

    @classmethod
    def get_plugin_name(cls):
        """Return plugin class name."""
        return cls.__name__

    @property
    def website(self):
        """Name of the website containing medium.

        Unless overridden by a subclass, it is the value of
        `website_name` attribute, or (if `None`) the name of the
        plugin class.
        """
        if self.website_name is not None:
            return self.website_name
        return unicode(self.__class__.__name__)

    @property
    def title(self):
        """Medium's title, or `None` if untitled."""
        assert self
        return None

    @property
    def url(self):
        """URL which was matched to the plugin, or equivalent.

        Default implementation uses the `url_format` class attribute.
        """
        return self.url_format % self.mediumid

    def formats(self):
        """Return an iterable of medium formats.

        Each element yielded is an instance of
        `getmediumurl.format.Format` and represents a format in which
        a medium file can be downloaded.

        Default implementation uses the `get_file_url` method to
        obtain the medium, without specifying metadata available by
        other format implementations.
        """
        # TODO: warn if this method is not overridden when the plugin
        # supports high quality media.
        return (PluginFormat(self),)

    def get_file_url(self, high_quality=False):
        """Return URL which can be used to download the medium file.

        :Parameters:

          high_quality
            If it is true, then the highest available quality will be
            tried.  Otherwise the "standard" quality URL will be
            returned.

        May return `None` if the URL cannot be determined.

        The default implementation uses the `formats` method and
        returns the URL of the format with lowest image area.
        """
        try:
            if high_quality:
                return max(self.formats()).url
            else:
                return min(self.formats()).url
        except ValueError:  # min or max of an empty sequence
            return None

    @property
    def thumbnail(self):
        """URL of a thumbnail image or `None` if not found."""
        assert self
        return None

    @property
    def description(self):
        """Medium description or `None` if unknown."""
        assert self
        return None

    @property
    def license(self):
        """Medium license URL or `None` if unknown."""
        assert self
        return None

    @property
    def language(self):
        """Medium language code or an empty string if unknown."""
        assert self
        return u""

    @property
    def author_name(self):
        """Name or user identification of medium's author."""
        assert self
        return None

    @property
    def author_url(self):
        """URL of medium's author."""
        assert self
        return None

    @classmethod
    def has_high_quality(cls):
        """Determine if the plugin finds different high quality media URLs."""
        return cls.high_quality

    @classmethod
    def disabled(cls):
        """Iterate over strings describing why this plugin is not usable.

        The strings state e.g. that a required module is not found, or
        the plugin does not have necessary code.

        Plugins which can be used must return no such strings.

        This implementation checks if fields needed by other methods
        are set, unless these methods are overridden, and if
        `formats` or `get_file_url` is replaced.
        """
        if not overrides("url", cls, Plugin) \
                and cls.url_format is None:
            yield "get_url not overridden or url_format not changed"
        if not overrides("match", cls, Plugin) \
                and cls.match_re is None:
            if cls.url_format is None:
                yield "match not overridden, match_re " \
                    "and url_format not specified"
            elif not cls._prepare_match_re():
                yield "match not overridden, match_re " \
                    "not specified and could not determine it " \
                    "from url_format"
        if not overrides("formats", cls, Plugin) \
                and not overrides("get_file_url", cls, Plugin):
            yield "formats and get_file_url not overridden"

    @classmethod
    def match(cls, url, urlreader):
        """Return plugin's object handling this URL, or `None`.

        This implementation uses a regular expression object in class
        attribute `match_re` with `mediumid` group passed to the
        `__init__` method.

        The `urlreader` argument is the same as in `__init__`.
        """
        if cls.match_re is None:
            if not cls._prepare_match_re():
                raise ValueError("Tried matching plugin with match_re of None")
        match = cls.match_re.match(url)
        if not match:
            return None
        return cls(match.group("mediumid"), urlreader)

    @classmethod
    def _prepare_match_re(cls):
        """Set `match_re` to value obtained from `url_format`.

        If `url_format` is a string containing exactly single ``%s``,
        with all other percent characters replaced by ``%%``, then it
        will make it a regex matching whole line, with ``mediumid``
        named group in place of ``%s``.

        Returns `True` on success, `False` otherwise.
        """
        if not check_single_percent_s(cls.url_format):
            return False
        cls.match_re = re.compile(u"^%s$"
                                  % escape_regex(cls.url_format)
                                  .replace(u"%s", u"(?P<mediumid>.+)"))
        return True
