# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test `getmediumurl.reader` and related modules.
"""


from shutil import rmtree
from tempfile import mkdtemp
import unittest

from getmediumurl.reader import URLReader, ReaderError, NotFoundError
from getmediumurl.readercache import dict_cache, directory_cache


__all__ = ("DictCacheTestCase", "DirectoryCacheTestCase")


class OneUseURLReader(URLReader):

    """`URLReader` subclass returning document's content only once."""

    @property
    def content(self):
        """Return the URL on first call, ``None`` on later calls."""
        value = self._url
        self._url = None
        return value


_COUNTER = 0


class FailingURLReader(URLReader):

    """An `URLReader` failing for every document."""

    def __init__(self, url):
        """Raise `ReaderError` and increase `_COUNTER`."""
        global _COUNTER
        _COUNTER += 1
        super(FailingURLReader, self).__init__(url)
        raise NotFoundError("document not found")


class DictCacheTestCase(unittest.TestCase):

    """Test `dict_cache`."""

    def test_cached_document(self):
        """Test that a document can be cached"""
        reader = dict_cache(OneUseURLReader)
        for url in ("http://example.org/", "http://other.example.com"):
            self.assertEquals(reader(url).content, url)
            self.assert_(reader(url).content is None)
        for url in ("http://example.org/", "http://other.example.com"):
            self.assert_(reader(url).content is None)

    def test_single_element_cache(self):
        """Test that cache can limit its size"""
        reader = dict_cache(OneUseURLReader, max_size=1)
        self.assertEquals(reader("http://example.org/").content,
                          "http://example.org/")
        self.assert_(reader("http://example.org/").content is None)
        self.assertEquals(reader("http://example.com/").content,
                          "http://example.com/")
        self.assert_(reader("http://example.com/").content is None)
        self.assertEquals(reader("http://example.org/").content,
                          "http://example.org/")

    def test_cached_fail(self):
        """Test that a non-existent document can be cached"""
        global _COUNTER
        reader = dict_cache(FailingURLReader)
        for url in ("http://example.org/", "http://other.example.com"):
            _COUNTER = 0
            self.assertRaises(ReaderError, reader, url)
            self.assertEquals(_COUNTER, 1)
            self.assertRaises(ReaderError, reader, url)
            self.assertEquals(_COUNTER, 1)

    def test_invalid_max_size(self):
        """Test that cache size must be an integer"""
        dict_cache(None, max_size=None)
        dict_cache(None, max_size=2)
        for size in "large", "42", 3.1415926, -1, 0:
            self.assertRaises(ValueError, dict_cache, None, max_size=size)


class DirectoryCacheTestCase(unittest.TestCase):

    """Test `directory_cache`."""

    def setUp(self):
        """Prepare."""
        self.directory = mkdtemp()

    def tearDown(self):
        """Remove temporary files."""
        rmtree(self.directory)

    def test_cached_document(self):
        """Test that a document can be cached"""
        reader = directory_cache(OneUseURLReader, directory=self.directory)
        for url in ("http://example.org/", "http://other.example.com"):
            self.assert_(reader(url).content is None)
            self.assertEquals(reader(url).content, url.encode("ascii"))
        for url in ("http://example.org/", "http://other.example.com"):
            self.assertEquals(reader(url).content, url.encode("ascii"))

    def test_cached_fail(self):
        """Test that a non-existent document can be cached"""
        global _COUNTER
        reader = directory_cache(FailingURLReader, directory=self.directory)
        for url in ("http://example.org/", "http://other.example.com"):
            _COUNTER = 0
            self.assertRaises(ReaderError, reader, url)
            self.assertEquals(_COUNTER, 1)
            self.assertRaises(ReaderError, reader, url)
            self.assertEquals(_COUNTER, 1)
