# -*- coding: utf-8 -*-
# Copyright (C) 2008, 2009 Leonardo Gastón De Luca leo[at]kde.org.ar
# Copyright (C) 2010 Osama Khalid osamak[at]gnu.org
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Based on youogg's plugins.py file from SVN revision 50.


"""
GetMediumURL plugin for YouTube videos.
"""


import re
import urllib

from getmediumurl.cache import cachedproperty
from getmediumurl.htmlplugin import HTMLPlugin
from getmediumurl.compat import parse_qs


__all__ = ("YouTube",)


class YouTube(HTMLPlugin):

    """Plugin to get YouTube video URLs."""

    match_re = re.compile(r'^http://(?:www\.|.{2}\.)?youtube\.com/' \
                              '(?:watch(?:_popup)?\?v=|v/)(?P<mediumid>.{11})')
    raw_url_format = u"http://www.youtube.com/watch?v=%s&skipcontrinter=1"
    url_format = u"http://www.youtube.com/watch?v=%s"
    website_name = u"YouTube"
    high_quality = True

    @cachedproperty
    def _flashvars(self):
        """A dictionary of variables used by the site's Flash player."""
        data = self.get_content_substring(start=r'flashvars=\"', end='\"')
        return parse_qs(urllib.unquote(data.decode("ascii")))

    # Needed for tests, to avoid HTTP request.
    @cachedproperty
    def url(self):
        """URL which was matched to the plugin, or equivalent."""
        return self.url_format % self.mediumid

    @property
    def thumbnail(self):
        """Thumbnail image URL."""
        return u"http://i3.ytimg.com/vi/%s/default.jpg" % self.mediumid

    @property
    def language(self):
        """Video language."""
        assert self
        # Not sure how this could be obtained.
        return u""

    @cachedproperty
    def author_name(self):
        """Return submitter's user name."""
        for element in self.page.getiterator("div"):
            if element.get("id") == "watch-headline-user-info":
                return element.findtext("a/strong")

    @cachedproperty
    def author_url(self):
        """Return submitter's profile URL."""
        for element in self.page.getiterator("div"):
            if element.get("id") == "watch-headline-user-info":
                return self.make_absolute(element.find("a").get("href"))

    def get_file_url(self, high_quality=False):
        """Get URL of downloadable video."""
        quality = u""
        if high_quality:
            # Get formats in which the video is available.
            fmt_list = self._flashvars["fmt_list"][0]
            formats = [desc.split(u"/")[0] for desc in fmt_list.split(u",")]
            for fmt in u"22", u"35", u"18":
                if fmt in formats:
                    quality = u"&fmt=" + fmt
                    break
        token = self._flashvars["t"][0]
        return u"http://www.youtube.com/get_video?video_id=" \
            u"%(mediumid)s&t=%(token)s%(quality)s&el=" \
            u"detailpage&asv=2" % \
            {u"mediumid": self.mediumid, u"token": token,
             u"quality": quality}
