clear pde;
gf_workspace('clear all');
% import the mesh
m=gf_mesh('import','gid','tripod.GiD.msh');
mfu=gf_mesh_fem(m,3);     % mesh-fem supporting a 3D-vector field
mfd=gf_mesh_fem(m,1);     % scalar mesh_fem

% we choose a P2 fem for the main unknown
gf_mesh_fem_set(mfu,'fem',gf_fem('FEM_PK(3,2)'), gf_integ('IM_TETRAHEDRON(5)'));
% the material is homogeneous, hence we use a P0 fem for the data
gf_mesh_fem_set(mfd,'fem',gf_fem('FEM_PK(3,0)'), gf_integ('IM_TETRAHEDRON(1)'));
% display some informations about the mesh
disp(sprintf('nbcvs=%d, nbpts=%d, nbdof=%d',gf_mesh_get(m,'nbcvs'),...
             gf_mesh_get(m,'nbpts'),gf_mesh_fem_get(mfu,'nbdof')));
P=gf_mesh_get(m,'pts'); % get list of mesh points coordinates
pidtop=find(abs(P(2,:)-13)<1e-6); % find those on top of the object
pidbot=find(abs(P(2,:)+10)<1e-6); % find those on the bottom
% build the list of faces from the list of points
ftop=gf_mesh_get(m,'faces from pid',pidtop); 
fbot=gf_mesh_get(m,'faces from pid',pidbot);
% assign boundary numbers
for mf=[mfu,mfd]
  gf_mesh_fem_set(mf,'boundary',1,ftop);
  gf_mesh_fem_set(mf,'boundary',2,fbot);
end;
% setup the pde object
pde.type = 'linear elasticity';
E = 1e3; Nu = 0.3;
pde.mf_u=mfu;
pde.mf_d=mfd;
pde.lambda={ E*Nu/((1+Nu)*(1-2*Nu)) }; % Lam coefficients
pde.mu={ E/(2*(1+Nu)) };
pde.F = {0,0,0};
% negative force on the Y axis, applied on boundary 1 (top of the object)
pde.bound{1}.type = 'Neumann';
pde.bound{1}.G    = {0,-1,0}; 
% the bottom of the object is fixed on the ground
pde.bound{2}.type = 'Dirichlet';
pde.bound{2}.R  = {0,0,0};
disp('running solve... can take some minutes and needs ~120MB of memory (reduce the degree of the fem if it is too much)');

% solve it (the linear system is a sparse 16000x16000 matrix).
t0=cputime; [U,pde]=gf_solve(pde); disp(sprintf('solve done in %.2f sec', cputime-t0));

% now that we have the solution, we want to compute the von mises stress
% first, we need to get the derivate of the solution:
mfdu=gf_mesh_fem(m,1);
% the P2 fem is not derivable across elements, hence we use a discontinuous
% fem for the derivative of U.
gf_mesh_fem_set(mfdu,'fem',gf_fem('FEM_PK_DISCONTINUOUS(3,1)'), ...
		gf_integ('IM_TETRAHEDRON(5)'));
% on output size(DU)=[3,3,nbdof(mfdu)]
DU=gf_compute(mfu,U,'gradient',mfdu);

% from the derivative, we compute the von mises stress
VM=zeros(1,gf_mesh_fem_get(mfdu,'nbdof'));
N=gf_mesh_get(m,'dim');
for i=1:size(DU,3),
  t=DU(:,:,i);
  E=(t+t')/2;
  VM(i) = sum(E(:).^2) - (1./N)*sum(diag(E))^2;
end;
VM = 4*pde.lambda{1}^2*VM;

disp('plotting ... can also take some minutes!');

% we plot the von mises on the deformed object, in superposition with the initial mesh.
gf_plot(mfdu,VM,'mesh','on', 'cvlst',gf_mesh_get(mfdu,'outer faces'),...
	'deformation',U,'deformation_mf',mfu);
caxis([0 10]); colorbar; view(180,-50); camlight;

r=[0.7 .7 .7]; l = r(end,:); s=63; s1=20; s2=25; s3=48;s4=55; for i=1:s, c1 = max(min((i-s1)/(s2-s1),1),0);c2 = max(min((i-s3)/(s4-s3),1),0); r(end+1,:)=(1-c2)*((1-c1)*l + c1*[1 0 0]) + c2*[1 .8 .2]; end; colormap(r); colorbar;

sm=gf_mesh_get(m,'char');
smfu=gf_mesh_fem_get(mfu,'char');
smfdu=gf_mesh_fem_get(mfdu,'char');
save tripod U VM DU r smfu smfdu sm

