/* -*- c++ -*- (enables emacs c++ mode)                                    */
/* *********************************************************************** */
/*                                                                         */
/* Library : Matlab Interface (matlabint)                                  */
/* File    : matlabint.h : global definitions for Matlab interface.        */
/*     									   */
/*                                                                         */
/* Date : September 26, 2001.                                              */
/* Author : Yves Renard, Yves.Renard@gmm.insa-tlse.fr                      */
/*                                                                         */
/* *********************************************************************** */
/*                                                                         */
/* Copyright (C) 2001  Yves Renard.                                        */
/*                                                                         */
/* This file is a part of GETFEM++                                         */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; version 2 of the License.                 */
/*                                                                         */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                         */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software Foundation, */
/* Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.         */
/*                                                                         */
/* *********************************************************************** */


#ifndef MATLABINT_H__
#define MATLABINT_H__

#include <matlabint_std.h>
#include <set>
//#include <getfem_config.h>
//#include <getfem_mesh.h>
#include <getfem_mesh_fem.h>
#include <gmm.h>
#include <gfi_array.h>

namespace getfem {
  class mesh_slice;
  class getfem_mesh;
  class mesh_fem;
  class mat_elem_type;
  typedef const mat_elem_type* pmat_elem_type; 
}

namespace matlabint
{
  /* exception-throwing version of the allocation functions of gfi_array.h */
  gfi_array* checked_gfi_array_create(int ndim, int *dims, gfi_type_id type);
  gfi_array* checked_gfi_array_create_1(int M, gfi_type_id type);
  gfi_array* checked_gfi_array_create_2(int M, int N, gfi_type_id type);
  gfi_array* checked_gfi_array_from_string(const char*s);
  gfi_array* checked_gfi_create_sparse(int m, int n, int nzmax);

  typedef bgeot::dim_type dim_type;
  typedef bgeot::scalar_type scalar_type; 

  typedef gmm::row_matrix<gmm::wsvector<scalar_type> >  gf_sparse_by_row;
  typedef gmm::col_matrix<gmm::wsvector<scalar_type> >  gf_sparse_by_col;
  typedef gmm::csc_matrix_ref<const double *, const unsigned int *, const unsigned int *> 
  gf_sparse_matlab_const_ref;

  class matlabint_mesh;
  class matlabint_mesh_fem;
  class matlabint_mesh_slice;

  /* fake vector class for matlab arrays (necessary for template functions) */
  class mlab_vect {
    unsigned sz;
    double *data;
    unsigned m,n,p; /* original number of rows & columns of matlab array 
		       p is a pseudo third dimension
		    */
  public:
    typedef double* iterator;
    typedef const double* const_iterator;

    double& operator[](unsigned i) { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    const double& operator[](unsigned i) const { if (i >= sz) THROW_INTERNAL_ERROR; return data[i]; }
    double& operator()(unsigned i, unsigned j, unsigned k=0) { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR; 
    return data[i+j*m+k*m*n]; }
    const double& operator()(unsigned i, unsigned j, unsigned k=0) const { if (i+j*m+k*m*n >= sz) THROW_INTERNAL_ERROR;return data[i+j*m+k*m*n]; }

    mlab_vect(): sz(0),data(0),m(0),n(0),p(0) {}
    void assign(const gfi_array *mx) { 
      if (gfi_array_get_class(mx) != GFI_DOUBLE) THROW_INTERNAL_ERROR;
      data = gfi_double_get_data(mx); sz = gfi_array_nb_of_elements(mx);
      int ndim = gfi_array_get_ndim(mx);
      const int *dim = gfi_array_get_dim(mx);
      if (ndim>0) m = dim[0]; else m = 0;
      if (ndim>1) n = dim[1]; else n = 1;
      p = 1;
      for (int i=2; i < ndim; ++i) p*=dim[i];
      if (n*m*p != sz) THROW_INTERNAL_ERROR;
    }
    mlab_vect(const gfi_array *mx) { 
      assign(mx);
    }
    mlab_vect(double *v, int sz_) { data = v; sz = sz_; m=sz_; n=1; p = sz ? sz/(m*n) : 0; }
    unsigned size() const { return sz; }
    unsigned getm() const { return m; }
    unsigned getn() const { return n; }
    unsigned getp() const { return p; }

    void reshape(unsigned n_, unsigned m_, unsigned p_=1) {
      if (sz != n_*m_*p_) THROW_INTERNAL_ERROR;
      n = n_; m = m_; p = p_;
    }

    getfem::base_node col_to_bn(unsigned j, unsigned k=0) const {
      getfem::base_node P(m);
      for (unsigned i=0; i < m; i++) P[i] = operator()(i,j,k);
      return P;
    }
    getfem::base_matrix row_col_to_bm(unsigned k=0) const {
      getfem::base_matrix M(m,n);
      for (unsigned i=0; i < m; i++) 
	for (unsigned j=0; j < n; j++) M(i,j) = operator()(i,j,k);
      return M;
    }
    bool in_range(double vmin, double vmax);

    iterator begin() { return data; }
    iterator end() { return data+sz; }
    const_iterator begin() const { return data; }
    const_iterator end() const { return data+sz; }

    /* copies the matlab vector into a new VECT object */
    template<class VECT> VECT to_vector() const {
      VECT v(sz);
      std::copy(begin(), end(), v.begin());
      return v;
    }
  };
}

/* intermde: interface gmm pour les mlab_vect */
namespace gmm {
  template<> struct linalg_traits<matlabint::mlab_vect> {
    typedef matlabint::mlab_vect this_type;
    typedef linalg_modifiable is_reference; /* is a writable reference */
    typedef abstract_vector linalg_type;
    typedef double value_type;
    typedef value_type origin_type;
    typedef double& reference;
    typedef this_type::iterator iterator;
    typedef this_type::const_iterator const_iterator;
    typedef gmm::abstract_dense storage_type;
    static size_type size(const this_type &v) { return v.size(); }
    static iterator begin(this_type &v) { return v.begin(); }
    static const_iterator begin(const this_type &v) { return v.begin(); }
    static iterator end(this_type &v) { return v.end(); }
    static const_iterator end(const this_type &v) { return v.end(); }
    static const origin_type* origin(const this_type &v) { return &v[0]; }
    static origin_type* origin(this_type &v) { return &v[0]; }
    static void clear(origin_type* o, const iterator &it, const iterator &ite)
    { std::fill(it, ite, value_type(0)); }
    static void do_clear(this_type &v) { std::fill(v.begin(), v.end(), 0.); }
    static value_type access(const origin_type *, const const_iterator &it,
			     const const_iterator &, size_type i)
    { return it[i]; }
    static reference access(origin_type *, const iterator &it,
			    const iterator &, size_type i)
    { return it[i]; }  
  };

  template <> struct temporary_dense_vector<matlabint::mlab_vect> {
    typedef bgeot::vsvector<double> vector_type;
  };
} /* fin de l'intermde */


namespace matlabint {

  /* input argument of the gf_* mex-files */
  class mexarg_in {
    void check_int_values(int vmin=INT_MIN, int vmax=INT_MAX);
    double to_scalar_(bool isint);
  public:
    const gfi_array *arg;
    int argnum;

    mexarg_in(const gfi_array *arg_, int num_) { arg = arg_; argnum = num_; }
    bool                        is_string() { return (gfi_array_get_class(arg) == GFI_CHAR); }
    bool                        is_cell() { return (gfi_array_get_class(arg) == GFI_CELL); }
    bool                        is_object_id(id_type *pid=0, id_type *pcid=0);
    bool                        is_mesh();
    bool                        is_mesh_fem();
    bool                        is_mesh_slice();
    int                         to_integer(int min_val=INT_MIN, int max_val=INT_MAX);
    double                      to_scalar(double min_val=-1e300, double max_val=1e300);
    std::string                 to_string();
    id_type                     to_object_id(id_type *pid=0, id_type *pcid=0);
    bgeot::base_poly *          to_poly();
    getfem::mesh_fem *          to_mesh_fem();
    const getfem::getfem_mesh * to_const_mesh();
    const getfem::getfem_mesh * to_const_mesh(id_type& id);
    matlabint_mesh *            to_matlabint_mesh();
    matlabint_mesh_fem *        to_matlabint_mesh_fem();
    getfem::getfem_mesh *       to_mesh();
    matlabint_mesh_slice *      to_matlabint_mesh_slice();
    getfem::mesh_slice *        to_mesh_slice();
    getfem::pintegration_method to_fem_interpolation();
    getfem::pfem                to_fem();
    getfem::pmat_elem_type      to_mat_elem_type();
    bgeot::pgeometric_trans     to_pgt();
    bgeot::pconvex_structure    to_convex_structure();

    /* do not perform any check on the number of dimensions of the matlab array */
    mlab_vect                   to_scalar_vector();

    /* expect the argument to be a row or column vector of given dimension */
    mlab_vect                   to_scalar_vector(int expected_dim);

    /* expect the argument to be a matrix (or possibly a 3D array)
       if any of the arguments has a value of -1, the corresponding dimension
       is not checked
     */
    mlab_vect                   to_scalar_vector(int expected_n, int expected_m, int expected_k=1);
    mlab_vect                   to_int_vector();
    mlab_vect                   to_int_vector(int expected_dim);
    mlab_vect                   to_int_vector(int expected_n, int expected_m, int expected_k=1);

    dal::bit_vector             to_bit_vector(const dal::bit_vector *subsetof = NULL, int shiftvals=-1);
    getfem::base_node           to_base_node() { return to_base_node(-1); }
    getfem::base_node           to_base_node(int expected_dim);
    void                        to_sparse(gf_sparse_matlab_const_ref& M);
  };

  /* output argument of the gf_* mex-files */
  class mexarg_out {
  public:
    gfi_array *& arg;
    int argnum;
    mexarg_out(gfi_array * &arg_, int num_) : arg(arg_), argnum(num_) {}

    void from_object_id(id_type id, id_type class_id);
    void from_object_id(std::vector<id_type> ids, id_type class_id);
    void from_integer(int i);
    void from_scalar(double v);
    void from_string(const char *s);
    template<class STR_CONT> void from_string_container(const STR_CONT& s);
    void from_bit_vector(const dal::bit_vector& bv, int shift=+1);
    void from_sparse(const gf_sparse_by_row& M, double threshold=1e-12);
    void from_sparse(const gf_sparse_by_col& M, double threshold=1e-12);
    void from_tensor(const getfem::base_tensor& t);
    mlab_vect create_vector(unsigned dim);
    mlab_vect create_vector(unsigned n, unsigned m); 
    mlab_vect create_vector(unsigned n, unsigned m, unsigned p); 
    template<class VECT> void from_vector(VECT& v) {
      create_vector(v.size());
      std::copy(v.begin(), v.end(), gfi_double_get_data(arg));
    }
    template<class VEC_CONT> void from_vector_container(const VEC_CONT& vv);
  };

  template<class STR_CONT> void 
  mexarg_out::from_string_container(const STR_CONT& s)
  {
    arg = checked_gfi_array_create_2(s.size(), 1, GFI_CELL);
    gfi_array **c = gfi_cell_get_data(arg);
    typename STR_CONT::const_iterator it;
    size_type cnt = 0;
    for (it = s.begin(); it != s.end(); ++it, ++cnt) {
      c[cnt] = checked_gfi_array_from_string((*it).c_str());
    }
  }

  /* output a matrix from a container of vectors (of same dimensions) */
  template<class VEC_CONT> void 
  mexarg_out::from_vector_container(const VEC_CONT& vv)
  {
    size_type n = vv.size();
    size_type m = (n == 0) ? 0 : vv[0].size();
    mlab_vect w  = create_vector(m,n);
    for (size_type j=0; j < n; ++j)
      for (size_type i=0; i < m; ++i)
	w(i,j) = vv[j][i];
  }

  gfi_array *create_object_id(int nid, id_type *ids, id_type cid);
  inline gfi_array *create_object_id(id_type id, id_type cid) {
    return create_object_id(1, &id, cid);
  }

  /* handles the list of input arguments */
  class mexargs_in {
    const gfi_array **in;
    dal::bit_vector idx;
    int nb_arg;
    bool use_cell;
    mexargs_in(const mexargs_in& );
    mexargs_in& operator=(const mexargs_in& );
  public:
    mexargs_in(int n, const gfi_array *p[], bool use_cell);
    ~mexargs_in();
    void check() const { if (idx.card() == 0) THROW_INTERNAL_ERROR; }
    const gfi_array *pop_gfi_array(size_type decal=0, int *out_idx = NULL) {
      size_type i = idx.first_true(); 
      check();
      if (decal >= idx.card()) THROW_INTERNAL_ERROR;
      while (decal>0) { i++; check(); if (idx.is_in(i)) decal--; }
      idx.sup(i); 
      if (out_idx) *out_idx = i;
      return in[i];
    }
    mexarg_in pop(size_type decal=0) { 
      int i;
      const gfi_array *m = pop_gfi_array(decal, &i);
     return mexarg_in(m,i+1); 
    }
    void restore(size_type i) { idx.add(i); }
    mexarg_in front() const { check(); return mexarg_in(in[idx.first_true()],idx.first_true()); }
    int narg() const { return nb_arg; }
    int remaining() const { return idx.card(); }
    void get_array(const gfi_array **& m) {
      if (remaining()) {
	m = new const gfi_array *[remaining()];
	for (size_type i=0; remaining(); i++) {
	  m[i] = pop_gfi_array();
	}
      } else m = NULL;
    }
    void pop_all() { idx.clear(); }
  };

  /* handles the list of output arguments */
  class mexargs_out {
    gfi_array **out;
    int nb_arg;
    int idx;

    mexargs_out(const mexargs_out& );
    mexargs_out& operator=(const mexargs_out& );
  public:
    mexargs_out(int nb_arg_, gfi_array *p[]);
    ~mexargs_out() {}
    void check() const { if ((idx >= nb_arg || idx < 0) && !(idx==0)) THROW_INTERNAL_ERROR; }
    mexarg_out pop() { check(); idx++; return mexarg_out(out[idx-1], idx); }
    mexarg_out front() const { check(); return mexarg_out(out[idx], idx+1); }
    int narg() const { return nb_arg; }
    int remaining() const { return std::max(nb_arg,1) - idx; }
    gfi_array **get_array() const { return out+idx; }
    void pop_all() { idx = nb_arg; }
    void return_packed_obj_ids(const std::vector<id_type>& ids, id_type class_id);
  };


  bool cmd_strmatch(const std::string& a, const char *s);
  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, 
		 int min_argin=0, int max_argin=-1);
  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_out& out,
		 int min_argout=0, int max_argout=-1);
  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, const mexargs_out& out,
		 int min_argin=0, int max_argin=-1,
		 int min_argout=0, int max_argout=-1);
  
  inline void bad_cmd(std::string& cmd) { 
    THROW_BADARG("Bad command name: " << cmd); }

  void check_cv_fem(const getfem::mesh_fem& mf, size_type cv);
  
  double get_NaN();
}  /* end of namespace matlabint.                                          */

#endif /* MATLABINT_H__                                                    */
