#include <matlabint_misc.h>
#include <matlabint_workspace.h>
#include <matlabint_mesh.h>
#include <matlabint_mesh_fem.h>
#include <matlabint_mesh_slice.h>
#include <memory> // auto_ptr for g++ 2.95

using namespace matlabint;

namespace getfem {
  class mesh_slice_streamline : public mesh_slice {
    scalar_type EPS;
  public:
    mesh_slice_streamline(mesh_slice_cv_dof_data_base *mfU, std::vector<base_node>& seeds, bool forward, bool backward) : 
      mesh_slice(mfU->pmf->linked_mesh()), EPS(1e-10) {
      const getfem_mesh &ml = mfU->pmf->linked_mesh();
      bgeot::geotrans_inv gti;
      std::vector<base_node> ref_pts(seeds.size());
      std::vector<size_type> indpts(seeds.size());
      gti.add_points(seeds);
      std::vector<size_type> cv2lst(ml.convex_index().last()+1, size_type(-1));
      for (dal::bv_visitor cv(ml.convex_index()); !cv.finished(); ++cv) {
        size_type nbpt;
        if ((nbpt=gti.points_in_convex(ml.convex(cv), ml.trans_of_convex(cv), ref_pts, indpts))) {
          //infomsg() << nbpt << " seeds found in convex " << cv << ": " << indpts << endl;
          for (size_type i=0; i < nbpt; ++i) {
            if (forward)
              extract_streamline(mfU, cv2lst, cv, seeds[indpts[i]], ref_pts[i], +1);
            if (backward)
              extract_streamline(mfU, cv2lst, cv, seeds[indpts[i]], ref_pts[i], -1);
          }
        }
      }
    }
  private:
    base_node interpolate(pfem pf, const base_node& x, const base_matrix& G, bgeot::pgeometric_trans pgt, const base_vector& coeff) {
      base_node y(x.size());
      base_vector val(pf->target_dim());
      base_vector c(pf->nb_dof());
      size_type qmult = x.size()/val.size();
      assert(coeff.size() == c.size()*qmult);
      base_node::iterator yit = y.begin();
      for (size_type q = 0; q < qmult; ++q) {
        dal::copy_n(dal::reg_spaced_iterator(coeff.begin()+q, qmult), pf->nb_dof(), c.begin());
        /*        for (size_type j=0; j < pf->nb_dof(); ++j) {
          c[j] = coeff[j*qmult+q];
          }*/
        val.fill(0.);
        pf->interpolation(x,G,pgt,c,val);
	yit = std::copy(val.begin(), val.end(), yit);
      }
      return y;
    }
    /*
      computes next node with runge-kutta.
       return 0 if next node is on a face of the convex,
       +1 if it falls outside and -1 if it is inside
    */
    int do_runge_kutta(bgeot::geotrans_inv_convex& gti,
                       const base_matrix& G, pfem pf, bgeot::pgeometric_trans pgt, const base_vector& coeff, 
                       const base_node& P0, const base_node& refP0, scalar_type h, base_node& P1, base_node& refP1) {
      base_node k1 = interpolate(pf,refP0, G, pgt, coeff); k1 *= h/2;
      //cerr << "do_runge_kutta: P0=" << P0 << ", refP0=" << refP0 << ", coeff=" << coeff << ", h=" << h << ", k1=" << k1 << endl;
      P1 = P0+k1;
      gti.invert(P1, refP1);
      scalar_type in1 = pgt->convex_ref()->is_in(refP1);

      //cerr << ", P1=" << P1 << ", refP1=" << refP1 << ", in1=" << in1 << endl;
      if (dal::abs(in1) < EPS) return 0;
      else if (in1 > 0) return +1;
      else {
        base_node k2 = interpolate(pf,refP1, G, pgt, coeff); k2 *= h;
        P1 = P0+k2;
        gti.invert(P1, refP1);
        in1 = pgt->convex_ref()->is_in(refP1);
        //cerr << "do_runge_kutta2: P1=" << P1 << ", refP1=" << refP1 << ", in1=" << in1 << endl;
        if (dal::abs(in1) < EPS) return 0;
        else if (in1 > 0) return +1;
      }
      return -1;
    }

    size_type find_convex_of_point(const getfem_mesh& ml, size_type cv, const base_node& P, base_node& refP, bgeot::geotrans_inv_convex& gti) {      
      /* find on which face is the point (approximately) */
      dim_type f = dim_type(-1);
      scalar_type best_f = 1e10;
      size_type cnt = 0;
      for (size_type i=0; i < ml.structure_of_convex(cv)->nb_faces(); ++i) {
        scalar_type v = ml.trans_of_convex(cv)->convex_ref()->is_in_face(i,refP);
        cnt++;
        if (v < best_f || cnt == 0) { best_f = v; f = dim_type(i); }
      }

      /* look for the other convex sharing this face */
      bgeot::mesh_face_convex_ind_ct clst = neighbour_of_convex(ml, cv, f);
      size_type best = size_type(-1); scalar_type best_v = 1e10;
      cnt = 0;
      for (bgeot::mesh_face_convex_ind_ct::const_iterator it = clst.begin();
           it != clst.end(); ++it) {
        if (*it != cv && ml.structure_of_convex(*it)->dim() == ml.dim()) {
          ++cnt;
          gti.init(ml.convex(*it), ml.trans_of_convex(*it));
          gti.invert(P, refP);
          scalar_type v = ml.trans_of_convex(*it)->convex_ref()->is_in(refP);
          if (v < best_v || cnt == 0) { best_v = v; best = *it; }
        }
      }
      if (cnt == 0) return size_type(-1); else return best;
    }
    
    void extract_streamline(mesh_slice_cv_dof_data_base *mfU, std::vector<size_type> cv2lst, size_type cv, 
                            const base_node& seed, const base_node& seed_ref, double dir) {
      mesh_slice::cs_nodes_ct snodes;
      mesh_slice::cs_simplexes_ct ssimplexes;
      bool change_convex = true, first=true;
      dim_type sdim = mfU->pmf->linked_mesh().dim();
      base_node P0(seed), refP0(seed_ref), P1, refP1;
      scalar_type h = 0;
      pfem pf = 0;
      bgeot::pgeometric_trans pgt = 0;
      base_vector coeff; 
      base_matrix G;
      bgeot::geotrans_inv_convex gti(mfU->pmf->linked_mesh().convex(cv), mfU->pmf->linked_mesh().trans_of_convex(cv));
      bool store_convex_and_stop = false;
      if (sdim != mfU->pmf->get_qdim()) 
        DAL_THROW(dal::failure_error, 
                  "won't compute streamline of a field whose dimension is not equal to mesh dimension");
      size_type cnt = 0;
      do {
        if (change_convex) {
          //cerr << "changement de convexe, ancien=" << cv << endl;
          /* init convex-dependant data */
          if (!first) {
            
            cv = find_convex_of_point(mfU->pmf->linked_mesh(), cv, P0, refP0, gti);
            //cerr << "nouveau convexe: " << cv << endl;
            if (cv == size_type(-1)) {
              //cerr << "fin au convex " << cv << endl;
              break;
            }
          }
          first = false;
          pf = mfU->pmf->fem_of_element(cv);
          pgt = mfU->pmf->linked_mesh().trans_of_convex(cv);
          mfU->copy(cv, coeff); if (dir<0) coeff = -coeff;
          /* get convex size estimation */
          h = 1e10;
          for (size_type i=0; i < pgt->nb_points(); ++i) {
            for (size_type j=0; j < pgt->nb_points(); ++j) {
              if (j!=i) {
                h = std::min(h, bgeot::vect_dist2_sqr(mfU->pmf->linked_mesh().points_of_convex(cv)[i], 
                                                      mfU->pmf->linked_mesh().points_of_convex(cv)[j]));
              }
            }
          }
          h = sqrt(h);
          scalar_type z=gmm::vect_norminf(coeff);
          if (z > EPS) 
            h = h/(10*z);
          else h=0; /* on va s'arreter */

          snodes.resize(1); ssimplexes.resize(0);
          snodes[0].faces.reset(); snodes[0].pt = P0; snodes[0].pt_ref = refP0;
          change_convex = false;
        }
        int rk = do_runge_kutta(gti, G, pf, pgt, coeff, P0, refP0, h, P1, refP1);

        if (bgeot::vect_dist2_sqr(P0,P1) < 1e-16) { store_convex_and_stop = true; } // we don't move anymore

        /* basic dichotomy on h if the step is to big */
        if (rk > 0) {
          //cerr << "->>> on va sortir du convexe, debut dichotomie\n";
          change_convex = true;
          scalar_type h0 = 0, h1 = 1;
          while (h1 - h0 > 1e-7*h) { 
            rk = do_runge_kutta(gti, G, pf, pgt, coeff, P0, refP0, (h0+h1)/2, P1, refP1);
            if (rk == 0) break;
            else if (rk > 0) h1 = (h0 + h1)/2;
            else h0 = (h0 + h1)/2;
          }
          //cerr << "fin dichotomie, h0=" << h0 << ", h1=" << h1 << ", rk=" << rk << endl;
        }
        
        //cerr << "AJOUT: cv = " << cv << ": P0=" << P0 << " - P1=" << P1 << endl;

        /* add the segment */
        snodes.push_back(slice_node(P1,refP1)); 
	snodes.back().faces.reset();
        ssimplexes.push_back(slice_simplex(2)); 
	ssimplexes.back().inodes[0] = snodes.size()-2; 
	ssimplexes.back().inodes[1] = snodes.size()-1;

        if (++cnt > 3000) { infomsg() << "too much iterations for streamline extraction\n"; store_convex_and_stop = true; }

        if (change_convex || store_convex_and_stop) {
          /* store streamline of previous convex */
          dal::bit_vector splx_in; splx_in.add(0, ssimplexes.size());
          cv2lst[cv] = set_convex(cv2lst[cv], cv, pgt->convex_ref(), snodes, ssimplexes, pgt->convex_ref()->structure()->nb_faces(), splx_in);
        }
        P0 = P1; refP0 = refP1;
      } while (!store_convex_and_stop);
    }
  };
}

static getfem::slicer* 
build_slicers(const getfem::getfem_mesh& m, dal::ptr_collection<getfem::slicer> & slicers, 
              const gfi_array *arg) {
  if (gfi_array_get_class(arg) != GFI_CELL) {
    DAL_THROW(matlabint_bad_arg, "slices must be described as imbricated cell arrays");
  }
  mexargs_in in(1, &arg, true); 
  std::string cmd = in.pop().to_string();
  if (check_cmd(cmd, "planar", in, 3, 3)) {
    int orient = in.pop().to_integer(-1,1);
    getfem::base_node x0 = in.pop().to_base_node();
    slicers.push_back(new getfem::slicer_half_space(x0, in.pop().to_base_node(), orient));
  } else if (check_cmd(cmd, "ball", in, 3, 3)) {
    int orient = in.pop().to_integer(-1,1);
    getfem::base_node x0 = in.pop().to_base_node();
    slicers.push_back(new getfem::slicer_sphere(x0, in.pop().to_scalar(1e-5), orient));
  } else if (check_cmd(cmd, "cylinder", in, 4, 4)) {
    int orient = in.pop().to_integer(-1,1);
    getfem::base_node x0 = in.pop().to_base_node();
    getfem::base_node x1 = in.pop().to_base_node();
    slicers.push_back(new getfem::slicer_cylinder(x0, x1, in.pop().to_scalar(1e-5), orient));
  } else if (check_cmd(cmd, "isovalues", in, 4, 4)) {
    int orient = in.pop().to_integer(-1,1);
    const getfem::mesh_fem &mf = *in.pop().to_mesh_fem();
    mlab_vect U = in.pop().to_scalar_vector(1, mf.nb_dof());
    slicers.push_back(new getfem::slicer_isovalues(getfem::mesh_slice_cv_dof_data<mlab_vect>(mf,U),
						   in.pop().to_scalar(), orient));
  } else if (check_cmd(cmd, "boundary", in, 0, 1)) {
    getfem::slicer *s1 = 0;
    if (in.remaining()) {
      s1 = build_slicers(m, slicers, in.pop().arg);
    } else {
      slicers.push_back(new getfem::slicer_none());
      s1 = slicers.back();
    }
    convex_face_ct cvflst;
    getfem::outer_faces_of_mesh(m, m.convex_index(), cvflst);
    slicers.push_back(new getfem::slicer_boundary(m,s1,cvflst));
  } else if (check_cmd(cmd, "none", in, 0, 0)) {
    slicers.push_back(new getfem::slicer_none());
  } else if (check_cmd(cmd, "diff", in, 2, 2)) {
    getfem::slicer *s1 = build_slicers(m, slicers, in.pop().arg);
    getfem::slicer *s2 = build_slicers(m, slicers, in.pop().arg);
    slicers.push_back(new getfem::slicer_complementary(s2));
    slicers.push_back(new getfem::slicer_intersect(s1, slicers.back()));
  } else if (check_cmd(cmd, "comp", in, 1, 1)) {
    getfem::slicer *s = build_slicers(m, slicers, in.pop().arg);
    slicers.push_back(new getfem::slicer_complementary(s));
  } else if (check_cmd(cmd, "union", in, 1, -1)) {
    getfem::slicer *s1 = build_slicers(m, slicers, in.pop().arg);
    while (in.remaining()) {
      getfem::slicer *s2 = build_slicers(m, slicers, in.pop().arg);
      slicers.push_back(new getfem::slicer_union(s1,s2));
      s1 = slicers.back();
    }
  } else if (check_cmd(cmd, "intersection", in, 1, -1)) {
    getfem::slicer *s1 = build_slicers(m, slicers, in.pop().arg);
    while (in.remaining()) {
      getfem::slicer *s2 = build_slicers(m, slicers, in.pop().arg);
      slicers.push_back(new getfem::slicer_intersect(s1,s2));
    }
  } else bad_cmd(cmd);  
  return slicers.back();
}


/*MLABCOM

  FUNCTION sl = gf_slice(sliceop, mesh M, int REFINE [, CVFLST])
  FUNCTION sl = gf_slice(sliceop, mesh_fem MF, vec U, int REFINE [, CVFLST])
  FUNCTION sl = gf_slice(sliceop, slice SL)
  FUNCTION sl = gf_slice('streamlines', mesh_fem MF, vec U, mat SEEDS)
  FUNCTION sl = gf_slice('points', mesh M, mat PTS)

  Creation of a mesh slice. Mesh slices are very similar to a
  P1-discontinuous mesh_fem on which interpolation is very fast. The
  slice is built from a mesh object, and a description of the slicing
  operation, for example, 
  
  sl = gf_slice({'planar',+1,[0;0],[1;0]}, m, 5);

  cuts the original mesh with the half space {y>0}. Each convex of the
  original mesh m is simplexified (for example a quadrangle is
  splitted into 2 triangles), and each simplex is refined 5 times.

  Slicing operations can be:
   - cutting with a plane, a sphere or a cylinder
   - intersection or union of slices
   - isovalues surfaces/volumes
   - "points", "streamlines" (see below)
  
  If the first argument is a mesh_fem mf instead of a mesh, and if it
  is followed by a field U (with size(U,1) == gf_mesh_fem_get(mf,U)),
  then the deformation U will be applied to the mesh before the
  slicing operation.

  The first argument can also be a slice.

  Slicing operations:
  Always specifiy them between braces (i.e. in a cell array).  The
  first argument is the name of the operation, followed the slicing
  options.


  * {'none'}

  Does not cut the mesh.

  * {'planar', orient, p, n} 

  Planar cut. p and n define a half-space, p being a point belong to
  the boundary of the half-space, and n being its normal. If orient is
  equal to -1 (resp. 0, +1), then the slicing operation will cut the
  mesh with the "interior" (resp. "boundary", "exterior") of the
  half-space.

  * {'ball', orient, c, r}
  
  Cut with a ball of center c and radius r.

  * {'cylinder', orient, p1, p2, r}

  Cut with a cylinder whose axis is the line (p1,p2) and whose radius
  is r.

  * {'isovalues',orient, mesh_fem MF, vec U, scalar V}

  Cut using the isosurface of the field U (defined on the mesh_fem
  MF). The result is the set {x such that U(x) <= V} or {x such that
  U(x) == V} or {x such that U(x) <= V} depending on the value of
  ORIENT.

  * {'boundary'[, SLICEOP]}
  
  Return the boundary of the result of SLICEOP, where SLICEOP is any
  slicing operation. If SLICEOP is not specified, then the whole mesh
  is considered (i.e. it is equivalent to {'boundary',{'none'}}).

  * {'union', SLICEOP1, SLICEOP2}
  * {'intersection', SLICEOP1, SLICEOP2}
  * {'comp', SLICEOP}
  * {'diff', SLICEOP1, SLICEOP2}

  Boolean operations: returns the union,intersection,complementary or
  difference of slicing operations.


  EXAMPLE:
  
  sl = gf_slice({intersection',{'planar',+1,[0;0;0],[0;0;1]},...
                {'isovalues',-1,mf2,U2}},mf,U,5);


  SPECIAL SLICES:

  There are also some special calls to gf_slice:

  * gf_slice('streamlines',mf, U, mat SEEDS)

  computes streamlines of the (vector) field U, with seed points given
  by the columns of SEEDS.

  * gf_slice('points', m, mat PTS)

  returns the "slice" composed of points given by the columns of PTS
  (useful for interpolation on a given set of sparse points, see
  gf_compute(mf,U,'interpolate on',sl).

  $Id: gf_slice.C,v 1.15 2004/01/19 11:11:56 pommier Exp $
MLABCOM*/

void gf_slice(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg()  <  2) THROW_BADARG("Wrong number of input arguments");
  if (out.narg() != 1) THROW_BADARG("Wrong number of output arguments");

  matlabint_mesh *mm = 0;
  std::auto_ptr<getfem::mesh_slice> ms;

  /* "normal" slices */
  if (in.front().is_cell()) {
    /* build slicers */
    const gfi_array *arg = in.pop().arg;

    /* check the source argument (mesh/mesh_fem or slice) */
    std::auto_ptr<getfem::mesh_slice_cv_dof_data<mlab_vect> > mfdef;
    matlabint_mesh_slice *msl = 0;
    if (in.front().is_mesh_fem() && in.remaining()  >=  3) {
      mm = object_to_mesh(workspace().object(in.front().to_matlabint_mesh_fem()->linked_mesh_id()));
      getfem::mesh_fem& mf = *in.pop().to_mesh_fem();
      mlab_vect Udef = in.pop().to_scalar_vector(-1, mf.nb_dof());
      if (!(mf.get_qdim() == mm->mesh().dim() && Udef.getm() == 1) &&
	  !(mf.get_qdim() == 1 && Udef.getm() == mm->mesh().dim())) {
	THROW_BADARG("either the mesh_fem must have a Qdim=" << int(mm->mesh().dim()) << 
		     ", either the data must have " << int(mm->mesh().dim()) << " rows");
      }
      mfdef.reset(new getfem::mesh_slice_cv_dof_data<mlab_vect>(mf,Udef));      
    } else if (in.front().is_mesh_slice()) {
      msl = in.pop().to_matlabint_mesh_slice();
      mm = object_to_mesh(workspace().object(msl->linked_mesh_id()));
    } else {
      id_type id; in.pop().to_const_mesh(id); mm = object_to_mesh(workspace().object(id));
    }

    dal::ptr_collection<getfem::slicer> slicers;
    getfem::slicer * s = build_slicers(mm->mesh(), slicers, arg);  

    /* build the slice */
    ms.reset(new getfem::mesh_slice(mm->mesh()));
    if (msl == 0) {
      if (in.remaining() == 0) THROW_BADARG("Not enough input arguments");
      size_type nrefine = in.pop().to_integer(1,1000);
      std::vector<convex_face> cvf;
      if (in.remaining()) {
	mlab_vect v = in.pop().to_int_vector(-1, -1);
	build_convex_face_lst(mm->mesh(), cvf, &v);
      } else build_convex_face_lst(mm->mesh(), cvf, 0);
      ms->build(s, nrefine, cvf, mfdef.get());
    } else {
      ms->build_from_slice(msl->mesh_slice(), s);
    }
    if (in.remaining()) THROW_BADARG("too much input arguments");
  } else if (in.front().is_string()) {
    /* "special" slices */
    std::string cmd = in.pop().to_string();
    if (check_cmd(cmd, "streamlines", in, 3, 3)) {
      getfem::mesh_fem *mf = in.front().to_mesh_fem();
      id_type id; in.pop().to_const_mesh(id); mm = object_to_mesh(workspace().object(id));
      mlab_vect U = in.pop().to_scalar_vector(1,mf->nb_dof());
      mlab_vect v = in.pop().to_scalar_vector(mm->mesh().dim(), -1);      
      std::vector<getfem::base_node> seeds(v.getn());
      for (size_type j=0; j < v.getn(); ++j)
        seeds[j] = v.col_to_bn(j);
      getfem::mesh_slice_cv_dof_data<mlab_vect> mfU(*mf,U);
      ms.reset(new getfem::mesh_slice_streamline(&mfU, seeds, true, true));
    } else if (check_cmd(cmd, "points", in, 2, 2)) {
      id_type id; in.pop().to_const_mesh(id); mm = object_to_mesh(workspace().object(id));
      ms.reset(new getfem::mesh_slice(mm->mesh()));
      mlab_vect w = in.pop().to_scalar_vector(mm->mesh().dim(), -1);
      std::vector<getfem::base_node> N(w.getn());
      for (size_type i=0; i < w.getn(); ++i) N[i] = w.col_to_bn(i);
      getfem::slicer_none s;
      ms->build_from_points(N, &s);
    } else bad_cmd(cmd);
  } else THROW_BADARG("a slicer specification (i.e. cell array) or a string "
		      "was expecting as the first argument");
  if (mm == 0 || ms.get() == 0) THROW_INTERNAL_ERROR;
  matlabint_mesh_slice *mms = new matlabint_mesh_slice(*mm, ms.release());
  out.pop().from_object_id(workspace().push_object(mms), SLICE_CLASS_ID);
  workspace().set_dependance(mms, mm);
}

