#include <matlabint_misc.h>
#include <matlabint_mesh.h>

using namespace matlabint;

static void check_empty_mesh(const getfem::getfem_mesh *pmesh)
{
  if (pmesh->dim() == bgeot::dim_type(-1) || pmesh->dim() == 0) {
    THROW_ERROR( "mesh object has an invalid dimension");
  }
}

/*MLABCOM
  FUNCTION [x] = gf_mesh_set(mesh M, operation [, args])

    General function for modification of a mesh object.

    * IDX = gf_mesh_set(M, 'add point', PT)
    Inserts new points in the mesh. PT should be an [n x m] matrix ,
    where n is the mesh dimension, and m is the number of points that
    will be added to the mesh. On output, IDX contains the indices 
    of these new points.

    Remark: if some points are already part of the mesh, they won't
    be inserted again, be IDX will contains the previously assigned
    indices of the points.

    * gf_mesh_set(M, 'del point', IDX)
    Removes one or more points from the mesh. IDX should contain the 
    point indexes, such as the one returned by the 'add point' command.

    * IDX = gf_mesh_set(M, 'add convex', geotrans CVTR, mat CVPTS)
    Adds a new convex of structure CVTR (obtained with gf_geotrans),
    and whose points are given by the columns of CVPTS. On return, IDX
    contains the convex number.  CVPTS might be a three dimensional
    array in order to insert more than one convex (or a two
    dimensional array correctly shaped).

    * gf_mesh_set(M, 'del convex', IDX)
    Removes one or more convexes from the mesh. IDX should contain the
    convexes indexes, such as the ones returned bu the 'add convex'
    command.

    * gf_mesh_set(M, 'translate', vec V)
    Translates each point of the mesh from V.

    * gf_mesh_set(M, 'transform', mat T)
    Applies the matrix T to each point of the mesh.

    * gf_mesh_set(M, 'merge', MESH M2) 
    Merges the mesh M2 in the mesh M (overlapping points won't be
    duplicated). If M2 is a mesh_fem object, its linked mesh will be
    used.

    * gf_mesh_set(M, 'optimize structure')
    Reset point and convex numbering. After optimisation, the points (resp.
    convexes) will be consecutively from 1 to gf_mesh_get(M,'max pid')
    (resp. gf_mesh_get(M,'max cvid')).

  MLABCOM*/
void gf_mesh_set(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }

  getfem::getfem_mesh *pmesh = in.pop().to_mesh();
  std::string cmd            = in.pop().to_string();
  if (check_cmd(cmd, "add point", in, out, 1, 1, 0, 1)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(), -1);
    mlab_vect w = out.pop().create_vector(1, v.getn());
    for (size_type j=0; j < v.getn(); j++) {
      w[j] = (double) pmesh->add_point(v.col_to_bn(j)) + 1;
    }
  } else if (check_cmd(cmd, "del point", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_int_vector();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = (int)v[j]-1;
      if (pmesh->point_is_valid(id)) {
	THROW_ERROR( "Can't remove point " << id+1 
		     << ": a convex is still attached to it.");
      }
      pmesh->sup_point(id);
    }
  } else if (check_cmd(cmd, "add convex", in, out, 2, 2, 0, 1)) {
    check_empty_mesh(pmesh);
    bgeot::pgeometric_trans pgt = in.pop().to_pgt();
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(), pgt->nb_points(), -1);
    mlab_vect w = out.pop().create_vector(1, v.getp());

    std::vector<matlabint::id_type> qp(pgt->nb_points());
    /* loop over convexes */
    for (size_type k=0; k < v.getp(); k++) {
      /* loop over convex points */
      for (size_type j=0; j < v.getn(); j++) {
	qp[j] = pmesh->add_point(v.col_to_bn(j,k));
      }
      id_type cv_id = pmesh->add_convex(pgt, qp.begin());
      w[k] = (double)(cv_id+1);
    }
  } else if (check_cmd(cmd, "del convex", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_int_vector();
    
    for (size_type j=0; j < v.size(); j++) {
      id_type id = (int)v[j]-1;
      if (pmesh->convex_index().is_in(id)) {
	pmesh->sup_convex(id);
      } else {
	THROW_ERROR("can't delete convex " << id+1 << ", it is not part of the mesh");
      }
    }
  } else if (check_cmd(cmd, "translate", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(),1);
    pmesh->translation(v.col_to_bn(0));
  } else if (check_cmd(cmd, "transform", in, out, 1, 1, 0, 0)) {
    check_empty_mesh(pmesh);
    mlab_vect v = in.pop().to_scalar_vector(pmesh->dim(),pmesh->dim());
    pmesh->transformation(v.row_col_to_bm());
  } else if (check_cmd(cmd, "merge", in, out, 1, 1, 0, 0)) {
    const getfem::getfem_mesh *pmesh2 = in.pop().to_const_mesh();
    for (dal::bv_visitor cv(pmesh2->convex_index()); !cv.finished(); ++cv)
      pmesh->add_convex_by_points(pmesh2->trans_of_convex(cv), pmesh2->points_of_convex(cv).begin());
  } else if (check_cmd(cmd, "optimize structure", in, out, 0, 0, 0, 0)) {
    pmesh->optimize_structure();
  } else bad_cmd(cmd);
}
