#include <matlabint_misc.h>
#include <matlabint_matelemtype.h>
#include <matlabint_pfem.h>

using namespace matlabint;

/*MLABCOM

  FUNCTION ELTM = gf_eltm(elt_matrix_type, args..)

  Generates a descriptor for an elementary matrix type.

  * ELTM = gf_eltm('base', fem FEM)
  Integration of shape functions on elements, using the fem FEM.

  * ELTM = gf_eltm('grad', fem FEM)
  Integration of gradient of shape functions on elements, using the fem FEM.

  * ELTM = gf_eltm('hessian', fem FEM)
  Integration of hessian of shape functions on elements, using the fem FEM.

  * ELTM = gf_eltm('product', eltm A, eltm B)
  Integration of the tensorial product of elementary matrices A and B.
  
  In order to obtain a numerical value of theses matrices, see gf_mesh_fem_get(mf, 'eltm').

MLABCOM*/


void gf_eltm(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 1) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  std::string cmd                  = in.pop().to_string();
  
  if (check_cmd(cmd, "base", in, out, 1, 1, 0, 1)) {  
    getfem::pfem pf                  = in.pop().to_fem();
    out.pop().from_object_id(matlabint::ind_matelemtype(getfem::mat_elem_base(pf)), 
			     ELTM_CLASS_ID);
  } else if (check_cmd(cmd, "grad", in, out, 1, 1, 0, 1)) {  
    getfem::pfem pf                  = in.pop().to_fem();
    out.pop().from_object_id(matlabint::ind_matelemtype(getfem::mat_elem_grad(pf)),
			     ELTM_CLASS_ID);
  } else if (check_cmd(cmd, "hessian", in, out, 1, 1, 0, 1)) {  
    getfem::pfem pf                  = in.pop().to_fem();
    out.pop().from_object_id(matlabint::ind_matelemtype(getfem::mat_elem_hessian(pf)),
			     ELTM_CLASS_ID);
  } else if (check_cmd(cmd, "product", in, out, 2, 2, 0, 1)) {  
    getfem::pmat_elem_type  m1 = in.pop().to_mat_elem_type();
    getfem::pmat_elem_type  m2 = in.pop().to_mat_elem_type();
    out.pop().from_object_id(matlabint::ind_matelemtype(getfem::mat_elem_product(m1,m2)),
			     ELTM_CLASS_ID);
  } else bad_cmd(cmd);
}

