#include <gmm.h>
#include <getfem_assembling.h>
#include <matlabint_misc.h>

using namespace matlabint;
namespace matlabint {
  struct mlab_vect_with_mxArray : public mlab_vect {
    gfi_array *mx;    
    mlab_vect_with_mxArray(const bgeot::tensor_ranges& r) {
      size_type sz = 1; for (size_type i=0; i < r.size(); ++i) sz *= r[i];
      if (sz == 0) ASM_THROW_TENSOR_ERROR("can't create a vector of size " << r);
      std::vector<int> tab(r.size());
      std::copy(r.begin(), r.end(), tab.begin());
      mx = checked_gfi_array_create(r.size(), &(tab.begin()[0]),GFI_DOUBLE);
      assign(mx);
    }
  };
}
namespace gmm {
  template<> struct linalg_traits<matlabint::mlab_vect_with_mxArray> {
    typedef matlabint::mlab_vect_with_mxArray this_type;
    typedef linalg_false is_reference;
    typedef abstract_vector linalg_type;
    typedef double value_type;
    typedef value_type origin_type;
    typedef double& reference;
    typedef this_type::iterator iterator;
    typedef this_type::const_iterator const_iterator;
    typedef abstract_dense storage_type;
    static size_type size(const this_type &v) { return v.size(); }
    static iterator begin(this_type &v) { return v.begin(); }
    static const_iterator begin(const this_type &v) { return v.begin(); }
    static iterator end(this_type &v) { return v.end(); }
    static const_iterator end(const this_type &v) { return v.end(); }
    static const origin_type* origin(const this_type &v) { return &v[0]; }
    static origin_type* origin(this_type &v) { return &v[0]; }
    static void clear(origin_type* o, const iterator &it, const iterator &ite)
    { std::fill(it, ite, value_type(0)); }
    static void do_clear(this_type &v) { std::fill(v.begin(), v.end(), 0.); }
    static value_type access(const origin_type *, const const_iterator &it,
			     const const_iterator &, size_type i)
    { return it[i]; }
    static reference access(origin_type *, const iterator &it,
			    const iterator &, size_type i)
    { return it[i]; }  
  };
}

namespace getfem {
  template<> class vec_factory<mlab_vect_with_mxArray> : 
    public base_vec_factory, private std::deque<asm_vec<matlabint::mlab_vect_with_mxArray> > {
  public:
    base_asm_vec* create_vec(const tensor_ranges& r) {
      asm_vec<matlabint::mlab_vect_with_mxArray> v(new matlabint::mlab_vect_with_mxArray(r));
      push_back(v); return &this->back();
    }
    ~vec_factory() { 
      for (size_type i=0; i < this->size(); ++i) {
	delete (*this)[i].vec(); // but it does not deallocate the mxArray !! that's fine
      }
    }
  };
}

static void
do_generic_assembly(mexargs_in& in, mexargs_out& out, bool on_boundary)
{
  size_type boundary_num = size_type(-1);
  dal::bit_vector cvlst; 
  bool with_cvlst = false;
  if (!on_boundary) {
    if (in.remaining() && !in.front().is_string()) {
      cvlst = in.pop().to_bit_vector();
      with_cvlst = true;
    }
  } else boundary_num = in.pop().to_integer();

  std::string s = in.pop().to_string();
  getfem::generic_assembly assem(s);
  /* stores the mesh_fem identifiers */
  while (in.remaining() && in.front().is_mesh_fem()) {
    assem.push_mf(*in.pop().to_mesh_fem());
  }
  if (assem.mf().size() == 0) THROW_INTERNAL_ERROR; /* should not be possible */

  /* store the data vectors */
  std::deque<mlab_vect> vtab;
  while (in.remaining()) {
    vtab.push_back(in.pop().to_scalar_vector());
  }
  /* DON't do that in the loop above, since push_back may invalidate every pointer on vtab[i] */
  for (size_type i=0; i < vtab.size(); ++i) assem.push_data(vtab[i]);

  /* set the kind of matrix/vector to build */
  getfem::mat_factory<gf_sparse_by_row> mat_fact; 
  getfem::vec_factory<mlab_vect_with_mxArray> vec_fact; 
  assem.set_mat_factory(&mat_fact);
  assem.set_vec_factory(&vec_fact);
  
  if (!on_boundary) {
    assem.volumic_assembly(with_cvlst ? cvlst : assem.mf()[0]->linked_mesh().convex_index());
  } else {
    assem.boundary_assembly(boundary_num);
  }
  // get the matrix back
  for (size_type i=0; out.remaining() && i < assem.mat().size(); ++i) {
    if (assem.mat()[i] != 0) {
      //      out.pop().from_sparse(*dynamic_cast<getfem::asm_mat<gf_sparse_by_row>*>(assem.mat()[i])->mat());
      getfem::base_asm_mat *BM = assem.mat()[i];
      getfem::asm_mat<gf_sparse_by_row> * M = 
	static_cast<getfem::asm_mat<gf_sparse_by_row>*>(BM);
      out.pop().from_sparse(*M->mat());
    }
  }
    
  if (out.remaining()) {
    for (size_type i=0; out.remaining() && i < assem.vec().size(); ++i) {
      if (assem.vec()[i] != 0) {
	//out.pop().from_vector(*dynamic_cast<getfem::asm_vec<bgeot::vsvector<getfem::scalar_type> >*>(assem.vec()[i])->vec());
	getfem::base_asm_vec *BV = assem.vec()[i];
	getfem::asm_vec<mlab_vect_with_mxArray> *V = 
	  static_cast<getfem::asm_vec<mlab_vect_with_mxArray> *>(BV);
	mexarg_out mo = out.pop(); mo.arg = V->vec()->mx;
      }
    }
  }
}
/*MLABCOM
  
  FUNCTION gf_asm(operation[, arg])

  General assembly function.

  * F = gf_asm('volumic source', mf_u, mf_d, Fd)
  Assembles a volumic source term, on the mesh_fem MF_U, using the 
  data vector FD defined on the data mesh_fem MF_D.

  * F = gf_asm('boundary source',boundary_num, mesh_fem mf_u, mesh_fem mf_d, vec g)
  Assembles a Neumann boundary condition. 
  G should be a [QDIM x N] matrix, where n is the number of degree of freedom
  of mf_d, and QDIM is dimension of the unknown u (that is set when creating the 
  mesh_fem).

  * M = gf_asm('mass matrix', mf1[, mf2])

  * M = gf_asm('laplacian', mf_u, mf_d, A)
  assembles elementary matrices for $\int a(x)(grad(u).grad(v))$

  * M = gf_asm('linear elasticity', mf_u, mf_d, lambda_d, mu_d)
  assembles elementary matrices for linear elasticity (and partially stokes)
  
  * K,B = gf_asm('stokes', mesh_fem mf_u, mesh_fem mf_p, mesh_fem mf_data, vec visc)

  Assembles elementary matrices for the pressure term in stokes mixed elements
  formulation: output: B is a sparse matrix corresponding to $\int p.div v$ and
  K is the linear elasticity stiffness matrix with lambda=0 and 2*mu=visc.

  * [HH,RR] = gf_asm('dirichlet', boundary_num, mesh_fem mf_u, 
                       mesh_fem mf_d, mat H, vec R[, threshold])

  Assembles Dirichlet conditions of type 'h.u = r' where h is a square
  matrix (of any rank) whose size is equal to gf_mesh_fem_get(mf_u,
  'qdim'). This matrix is stored in 'H', one column per dof in
  'mf_d', each column containing the values of the matrix 'h' stored
  in fortran order: H(:,j) = [h11(x_j) h21(x_j) h12(x_j) h22(x_j)] if u 
  is a 2D vector field.
  Of course, if the unknown is a scalar field, you just have to set
  H = ones(1, gf_mesh_fem_get(mf_d, 'nbdof')).

  This is basically the same than calling gf_asm('boundary qu term')
  for H and calling gf_asm('neumann') for R, except that this
  function tries to produce a 'better' (more diagonal) constraints
  matrix (when possible).

  * [N,U0]=gf_asm('dirichlet nullspace', H, R)

  Solves the dirichlet conditions HU=R, returning a solution U0 which
  has a minimum L2-norm. The sparse matrix N contains an orthogonal
  basis of the kernel of the constraints matrix H (hence, the PDE
  linear system should be solved on this subspace):
  the initial problem
     KU = B with constraints HU=R
  is replaced by
     (N'*K*N)*UU = N'*B
     U = N*UU + U0


  * M = gf_asm('boundary qu term',boundary_num, mesh_fem mf_u, 
                 mesh_fem mf_d,mat q)

  Q should be be a [(Qdim*Qdim) x N] matrix, where n is the number
  of degree of freedom of mf_d, and QDIM is dimension of the unkown u
  (that is set when creating the mesh_fem). Q is stored in the same
  order than the H matrix of gf_asm('dirichlet').

  * [Q,G,H,R,F]=gf_asm('pdetool boundary conditions',mf_u, mf_d, b, e[, f_expr])

  Assembles pdetool boundary conditions: B is the boundary matrix
  exported by the pdetool, and E is the edges array. 'F_EXPR' is an
  optionnal expression (or vector) for the volumic term. On return
  Q,G,H,R,F contain the assembled boundary conditions (Q and H are
  matrices), similar to the ones returned by the function ASSEMB
  from PDETOOL.

  * [...] = gf_asm('volumic' [,CVLST], expr, mesh_fems, data...)
  * [...] = gf_asm('boundary', bnum, expr, mesh_fems, data...)

  Generic assembly procedure for volumic and boundary assembly. The expression
  'expr' is evaluated over the mesh_fems listed in the arguments (with
  optional data) and assigned to the output arguments. For details about the
  syntax of assembly expressions, please refer to the getfem user manual (or
  look at the file getfem_assembling.h in the getfem++ sources).

  For example, the L2 norm of a field can be computed with gf_compute(mf,U,'L2
  norm') or with:  
  gf_asm('volumic','u=data(#1); V()+=u(i).u(j).comp(Base(#1).Base(#1))(i,j)',mf,U)

  The Laplacian stiffness matrix can be evaluated with gf_asm('Laplacian',mf, A) or equivalently with:
gf_asm('volumic',['a=data(#2); ',)6-A
       'M(#1,#1)+=sym(comp(Grad(#1).Grad(#1).Base(#2))(:,i,:,i,j).a(j))'], mf, A);

  

  MLABCOM*/
/*MLABEXT
  if (nargin>=1 & strcmpi(varargin{1},'pdetool boundary conditions')),
    [varargout{1:nargout}]=gf_asm_pdetoolbc(varargin{[1 3:nargin]}); return;
  end;
  MLABEXT*/
void gf_asm(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 1) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  std::string cmd = in.pop().to_string();

  if (check_cmd(cmd, "mass matrix", in, out, 1, 2, 0, 1)) {
    getfem::mesh_fem *mf_u1 = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_u2 = in.remaining() ? in.pop().to_mesh_fem() : mf_u1;
    
    gf_sparse_by_row M(mf_u1->nb_dof(), mf_u2->nb_dof());
    getfem::asm_mass_matrix(M, *mf_u1, *mf_u2);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "laplacian", in, out, 3, 3,0, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect A            = in.pop().to_scalar_vector(mf_d->nb_dof());
    gf_sparse_by_row M(mf_u->nb_dof(), mf_u->nb_dof());
    getfem::asm_stiffness_matrix_for_laplacian(M, *mf_u, *mf_d, A);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "linear elasticity", in, out, 4, 4, 0, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect lambda       = in.pop().to_scalar_vector(mf_d->nb_dof());
    mlab_vect mu           = in.pop().to_scalar_vector(mf_d->nb_dof());
    gf_sparse_by_row M(mf_u->nb_dof(), mf_u->nb_dof());
    getfem::asm_stiffness_matrix_for_linear_elasticity(M, *mf_u, *mf_d, lambda, mu);
    out.pop().from_sparse(M);
  } else if (check_cmd(cmd, "stokes", in, out, 4, 4, 0, 2)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_p = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect        vec_d = in.pop().to_scalar_vector(mf_d->nb_dof());
    gf_sparse_by_row  K(mf_u->nb_dof(), mf_u->nb_dof());
    gf_sparse_by_row  B(mf_u->nb_dof(), mf_p->nb_dof());
    getfem::asm_stokes(K, B, *mf_u, *mf_p, *mf_d, vec_d);
    out.pop().from_sparse(K);
    out.pop().from_sparse(B);
  } else if (check_cmd(cmd, "volumic source", in, out, 3, 3, 1, 1)) {
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    mlab_vect Fd           = in.pop().to_scalar_vector(mf_u->get_qdim(), mf_d->nb_dof());
    mlab_vect Fu           = out.pop().create_vector(1, mf_u->nb_dof());
    getfem::asm_source_term(Fu, *mf_u, *mf_d, Fd);
  } else if (check_cmd(cmd, "boundary source", in, out, 4, 4, 0, 1)) {
    int boundary_num       = in.pop().to_integer();
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    unsigned q_dim = mf_u->get_qdim();
    mlab_vect g            = in.pop().to_scalar_vector(q_dim, mf_d->nb_dof());
    mlab_vect F            = out.pop().create_vector(mf_u->nb_dof(),1);
    getfem::asm_source_term(F, *mf_u, *mf_d, g, boundary_num);
  } else if (check_cmd(cmd, "dirichlet", in, out, 5, 6, 2, 2)) {
    int boundary_num       = in.pop().to_integer();
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();
    unsigned q_dim = mf_u->get_qdim(); 
    unsigned q_dim2 = q_dim*q_dim;
    mlab_vect h            = in.pop().to_scalar_vector(q_dim2, mf_d->nb_dof());
    mlab_vect r            = in.pop().to_scalar_vector(q_dim, mf_d->nb_dof());
    double threshold = 1e-8;
    if (in.remaining()) {
      threshold = in.pop().to_scalar();
      if (threshold < 0 || threshold > 1e10) THROW_BADARG("wrong threshold\n");
    }

    gf_sparse_by_row H(mf_u->nb_dof(), mf_u->nb_dof());
    mexarg_out out_H       = out.pop();
    mlab_vect R            = out.pop().create_vector(mf_u->nb_dof(),1);
    
    getfem::asm_dirichlet_constraints(H, R, *mf_u, *mf_d, h, r, boundary_num);
    
    out_H.from_sparse(H,threshold);
  } else if (check_cmd(cmd, "dirichlet nullspace", in, out, 2, 2, 2, 2)) {
    gf_sparse_matlab_const_ref H;  in.pop().to_sparse(H);
    mlab_vect R            = in.pop().to_scalar_vector();
    
    gf_sparse_by_col      NS(gmm::mat_ncols(H), gmm::mat_nrows(H));
    bgeot::vsvector<double>   Ud(H.ncols());

    size_type nl = getfem::Dirichlet_nullspace(H, NS, 
					       R.to_vector<bgeot::vsvector<double> >(), Ud);
    gmm::resize(NS,gmm::mat_nrows(NS),nl); /* remove unused columns */
    //NS.resize(nl);
    out.pop().from_sparse(NS);
    out.pop().from_vector(Ud);
  } else if (check_cmd(cmd, "boundary qu term", in, out, 4, 4, 0, 1)) {
    int boundary_num       = in.pop().to_integer();
    getfem::mesh_fem *mf_u = in.pop().to_mesh_fem();
    getfem::mesh_fem *mf_d = in.pop().to_mesh_fem();

    unsigned q_dim = mf_u->get_qdim();
    unsigned q_dim2 = q_dim*q_dim;
    mlab_vect q            = in.pop().to_scalar_vector(q_dim2, 
						       mf_d->nb_dof());
    gf_sparse_by_row Q(mf_u->nb_dof(), mf_u->nb_dof());
    getfem::asm_qu_term(Q, *mf_u, *mf_d, q, boundary_num);
    out.pop().from_sparse(Q);
  } else if (check_cmd(cmd, "volumic", in, out, 2, -1, 0, -1)) {
    do_generic_assembly(in, out, false);
  } else if (check_cmd(cmd, "boundary", in, out, 3, -1, 0, -1)) {
    do_generic_assembly(in, out, true);
  } else bad_cmd(cmd);
}
