#include <map>
#include <matlabint.h>
#include <matlabint_misc.h>
#include <matlabint_workspace.h>
#include <matlabint_poly.h>
#include <matlabint_mesh.h>
#include <matlabint_mesh_slice.h>
#include <matlabint_mesh_fem.h>
#include <matlabint_matelemtype.h>
#include <matlabint_matelem.h>
#include <matlabint_pfem.h>
#include <matlabint_pfi.h>
#include <matlabint_pgt.h>
#include <matlabint_convex_structure.h>

#include <matlabint_misc.h>
//#ifdef MAINTAINER_MODE
# ifdef HAVE_CSIGNAL
#  include <csignal>
# else 
#  include <signal.h>
# endif
//#endif
#include <exception> // NE PAS METTRE CE FICHIER EN PREMIER !!!! ou bien ennuis avec dec cxx 6.3 garantis

namespace matlabint {

  /* associate the class ID found in the matlab structures referencing
     getfem object to a class name which coincides with the class name
     given by matlab to the structure (hum..) */
  const char *name_of_getfemint_class_id(unsigned cid) {
    static const char *cname[GETFEMINT_NB_CLASS] = {
      "gfMesh", "gfMeshFem", "gfGeoTrans", 
      "gfFem", "gfInteg","gfEltm","gfCvStruct","gfPoly", "gfSlice"
    };

    if (cid >= GETFEMINT_NB_CLASS) return "not_a_getfem_class";
    else return cname[cid];
  }

  static std::string dim_of_gfi_array(const gfi_array *t) {
    std::stringstream ss;
    for (size_type i=0; i < static_cast<size_type>(gfi_array_get_ndim(t)); ++i) {
      if (i) ss << "x"; ss << gfi_array_get_dim(t)[i];
    }
    return ss.str();
  }

  double
  mexarg_in::_to_scalar(bool isint)
  {
    double dv;
    if (gfi_array_nb_of_elements(arg) != 1) {
      THROW_BADARG("Argument " << argnum << 
		   "has dimensions " << dim_of_gfi_array(arg) <<
		   " but a [1x1] " << std::string(isint ? "integer" : "scalar") << 
		   " was expected");
    }
    switch (gfi_array_get_class(arg)) {
    case GFI_DOUBLE: {
      dv = *gfi_double_get_data(arg);
    } break;
    case GFI_INT32: {
      dv = (double)( *((dal::int32_type*)gfi_int32_get_data(arg)));
    } break;
    case GFI_UINT32: {
      dv = (double)( *((dal::int32_type*)gfi_uint32_get_data(arg)));
    } break;
    default: {
      THROW_BADARG("Argument " << argnum << " of class " << gfi_array_get_class_name(arg) << 
                   " is not an scalar value");
    } break;
    }
    return dv;
  }

  double
  mexarg_in::to_scalar(double minval, double maxval)
  {
    double dv = _to_scalar(false);
    if (dv < minval || dv > maxval) {
      THROW_BADARG("Argument " << argnum << 
		" is out of bounds : " << dv << " not in [" << 
		minval << "..." << maxval << "]");
    }
    return dv;
  }

  int
  mexarg_in::to_integer(int minval, int maxval)
  {
    double dv = _to_scalar(true);
    if (dv != floor(dv)) {
      THROW_BADARG("Argument " << argnum << 
		" is not an integer value");
    }
    if (dv < minval || dv > maxval) {
      THROW_BADARG("Argument " << argnum << 
		" is out of bounds : " << dv << " not in [" << 
		minval << "..." << maxval << "]");
    }
    return (int)dv;
  }



  bool
  mexarg_in::is_object_id(id_type *pid, id_type *pcid) 
  {    
    if (gfi_array_get_class(arg) == GFI_OBJID && 
        gfi_array_nb_of_elements(arg) == 1) {
      if (pid)  *pid  = gfi_objid_get_data(arg)->id;
      if (pcid)  *pcid  = gfi_objid_get_data(arg)->cid;
      return true;
    }
    return false;
  }

  bool
  mexarg_in::is_mesh() 
  {
    id_type id, cid;
    if (is_object_id(&id, &cid) && cid == MESH_CLASS_ID) {
      getfem_object *o = workspace().object(id, "MESH");
      return (object_is_mesh(o));
    } else return false;
  }

  bool
  mexarg_in::is_mesh_fem() 
  {
    id_type id, cid;
    if (is_object_id(&id, &cid) && cid == MESHFEM_CLASS_ID) {
      getfem_object *o = workspace().object(id, "MESH FEM");
      return (object_is_mesh_fem(o));
    } else return false;
  }

  bool
  mexarg_in::is_mesh_slice() 
  {
    id_type id, cid;
    if (is_object_id(&id, &cid) && cid == SLICE_CLASS_ID) {
      getfem_object *o = workspace().object(id, "MESH SLICE");
      return (object_is_mesh_slice(o));
    } else return false;
  }

  std::string 
  mexarg_in::to_string()
  {
    /* string => row vector. */
    if (!is_string()) 
      THROW_BADARG("Argument " << argnum << " must be a string.");
    return std::string(gfi_char_get_data(arg), gfi_array_nb_of_elements(arg));
  }

  id_type
  mexarg_in::to_object_id(id_type *pid, id_type *pcid) 
  {
    id_type id,cid;
    if (!is_object_id(&id, &cid)) {
      THROW_BADARG("wrong type for argument " << argnum <<
		   ": expecting a getfem object, got a " << gfi_array_get_class_name(arg));
    }
    if (pid) *pid = id;
    if (pcid) *pcid = cid;
    return id;
  }


  /*
    check if the argument is a valid handle to an intset
    and returns it
  */
  /*
  dal::bit_vector * mexarg_in::to_intset()
  {
    getfem_object *o = workspace().object(to_object_id());
    if (!object_is_intset(o)) {
      THROW_BADARG("Argument " << argnum << " should be an intset descriptor");
    }
    return &object_to_intset(o)->intset();
  }
  */


  /*
    check if the argument is a valid handle to a base_poly
    and returns it
  */
  bgeot::base_poly *
  mexarg_in::to_poly()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != POLY_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a polynom descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return &object_to_poly(o)->poly();
  }


  /*
    check if the argument is a valid handle to a mesh_fem,
    and returns it
  */
  matlabint_mesh_fem *
  mexarg_in::to_matlabint_mesh_fem()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESHFEM_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh_fem descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return object_to_mesh_fem(o);
  }
  getfem::mesh_fem *
  mexarg_in::to_mesh_fem() {
    return &to_matlabint_mesh_fem()->mesh_fem();
  }

  /*
    check if the argument is a valid mesh handle
    if a mesh_fem handle is given, its associated
    mesh is used
  */
  const getfem::getfem_mesh *
  mexarg_in::to_const_mesh() { id_type mid; return to_const_mesh(mid); }

  const getfem::getfem_mesh *
  mexarg_in::to_const_mesh(id_type& mid)
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESHFEM_CLASS_ID && cid != MESH_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh or mesh_fem descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    const getfem::getfem_mesh *mesh = NULL;
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    if (object_is_mesh(o)) {
      mid = id;
      mesh = &object_to_mesh(o)->mesh();
    } else if (object_is_mesh_fem(o)) {
      mid = object_to_mesh_fem(o)->linked_mesh_id();
      mesh = &object_to_mesh_fem(o)->mesh_fem().linked_mesh();
    } else THROW_INTERNAL_ERROR;
    return mesh;
  }


  /*
    check if the argument is a valid mesh handle
    (the returned arg is not const, so we can't use the mesh from mesh_fem objects)
  */
  matlabint_mesh *
  mexarg_in::to_matlabint_mesh()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != MESH_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return object_to_mesh(o);
  }

  getfem::getfem_mesh *
  mexarg_in::to_mesh() {
    return &to_matlabint_mesh()->mesh();
  }


  matlabint_mesh_slice *
  mexarg_in::to_matlabint_mesh_slice()
  {
    id_type id, cid;
    to_object_id(&id,&cid);
    if (cid != SLICE_CLASS_ID) {
      THROW_BADARG("argument " << argnum << " should be a mesh slice descriptor, its class is " << name_of_getfemint_class_id(cid));
    }
    getfem_object *o = workspace().object(id,name_of_getfemint_class_id(cid));
    return object_to_mesh_slice(o);
  }

  getfem::mesh_slice *
  mexarg_in::to_mesh_slice() {
    return &to_matlabint_mesh_slice()->mesh_slice();
  }
  
  getfem::pintegration_method
  mexarg_in::to_fem_interpolation()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != INTEG_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be an integration method descriptor");
    if (!exists_pfi(id)) {
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid integration method handle");
    }
    return addr_pfi(id);
  }

  getfem::pmat_elem_type
  mexarg_in::to_mat_elem_type()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != ELTM_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be a elementary matrix descriptor.");
    if (!exists_matelemtype(id))
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid elementary matrix handle");
    return addr_matelemtype(id);
  }

  getfem::pfem
  mexarg_in::to_fem()
  {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != FEM_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " should be a fem descriptor");
    
    if (!exists_pfem(id))
      THROW_BADARG("Argument " << argnum << 
		   " is not a valid fem handle");
    return addr_pfem(id);
  }


  bgeot::pgeometric_trans
  mexarg_in::to_pgt() {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != GEOTRANS_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " is not a geometric transformation handle");
    if (!matlabint::exists_pgt(id))
      THROW_BADARG("Argument " << argnum << 
		   " refers to a geometric transformation that does not exists");
    return matlabint::addr_pgt(id);
  }

  bgeot::pconvex_structure
  mexarg_in::to_convex_structure() {
    id_type id,cid;
    to_object_id(&id,&cid);
    if (cid != CVSTRUCT_CLASS_ID)
      THROW_BADARG("Argument " << argnum << 
		   " is not a convex structure handle");
    if (!matlabint::exists_convex_structure(id))
      THROW_BADARG("Argument " << argnum << 
		   " refers to a convex structure that does not exists");
    return matlabint::addr_convex_structure(id);
  }

  mlab_vect
  mexarg_in::to_scalar_vector() {
    if(!(gfi_array_get_class(arg) == GFI_DOUBLE)) {
      THROW_BADARG("Argument " << argnum << 
		   " should be a DOUBLE REAL data array");
    }
    return mlab_vect(arg);
  }

  /* check that the supplied argument IS a vector,
     with the right number of elements */
  mlab_vect
  mexarg_in::to_scalar_vector(int expected_dim) {
    mlab_vect v = to_scalar_vector();
    if (v.getn() != 1 && v.getm() != 1 && v.size() != 0) {
      THROW_BADARG("Argument " << argnum << 
		   " should be a vector, not a matrix");
    }
    if (expected_dim != -1 && (int)v.size() != expected_dim) {
      THROW_BADARG("Argument " << argnum << 
		   " has wrong dimensions: expected " << expected_dim << 
		   ", found " << v.size());
    }
    return v;
  }

  /* check that the supplied array has a good number of rows
     and/or a good number of columns (and or a good number of third dimension) */
  mlab_vect
  mexarg_in::to_scalar_vector(int expected_m, int expected_n, int expected_p) {
    mlab_vect v = to_scalar_vector();
    if (expected_m != -1 && (int)v.getm() != expected_m)
      THROW_BADARG("Argument " << argnum << 
		   " has a wrong number of rows (" << v.getm() <<
		   ") , " << expected_m << " rows were expected");
    if (expected_n != -1 && (int)v.getn() != expected_n)
      THROW_BADARG("Argument " << argnum << 
		   " has a wrong number of columns (" << v.getn() <<
		   ") , " << expected_n << " columns were expected");

    if (expected_p != -1 && (int)v.getp() != expected_p)
      THROW_BADARG("Argument " << argnum << 
		   " was expected to be a three-dimensional array, with " << 
		   expected_p << " elements in its third dimension (got " << 
		   v.getp() << ")");

    return v;
    /*
    if (m == 1 || n == 1) {
      inout_dim = 1; len = (m == 1) ? n : m;
      if (len > nbdof && nbdof && (len % nbdof == 0)) {
	inout_dim = len / nbdof;
	len = nbdof;
      }
    } else {
      inout_dim = m;
      len = n;
    }

    if (len != nbdof) {
      THROW_BADARG("Argument " << argnum << 
      " has a bad size (" << len << 
      ") , it should be " << nbdof);
    }

    if (dim != inout_dim && dim != 0) {
    THROW_BADARG("Argument " << argnum << 
		"is supposed to have " << dim << " rows");
    }
    return mlab_vect(arg);*/
  }

  void
  mexarg_in::check_int_values(int vmin, int vmax) {
    mlab_vect v = to_scalar_vector();
    for (unsigned i=0; i < v.size(); i++) {
      if (floor(v[i]) != v[i]) {
	THROW_BADARG("Argument " << argnum << 
		     " should be a DOUBLE REAL data array containing only "
		     "INTEGER values --- at index " << i+1 << 
		     " the scalar value " << v[i] << " was found");
      }
      if (v[i] < vmin || v[i] > vmax) {
	THROW_BADARG("Argument " << argnum << 
		     " has an out-of-bound value at index " << i+1 << 
		     " : " << v[i] << " is not in the range [" << vmin << 
		     ":" << vmax << "]");
      }
    }
  }


  mlab_vect
  mexarg_in::to_int_vector() {
    check_int_values();
    return to_scalar_vector();
  }

  mlab_vect
  mexarg_in::to_int_vector(int expected_dim) {
    check_int_values();
    return to_scalar_vector(expected_dim);    
  }

  mlab_vect
  mexarg_in::to_int_vector(int expected_m, int expected_n, int expected_p) {
    check_int_values();
    return to_scalar_vector(expected_m, expected_n, expected_p);    
  }

  getfem::base_node
  mexarg_in::to_base_node(int expected_dim) {
    mlab_vect w = to_scalar_vector(expected_dim,1);
    getfem::base_node bn(w.size()); std::copy(w.begin(), w.end(), bn.begin());
    return bn;
  }

  void
  mexarg_in::to_sparse(gf_sparse_matlab_const_ref& M) {
    if (gfi_array_get_class(arg) != GFI_SPARSE) {
      THROW_BADARG("Argument " << argnum << " was expected to be a sparse matrix");
    }
    assert(gfi_array_get_ndim(arg)==2);
    M = gf_sparse_matlab_const_ref(gfi_sparse_get_pr(arg), gfi_sparse_get_ir(arg), gfi_sparse_get_jc(arg),
				   gfi_array_get_dim(arg)[0],gfi_array_get_dim(arg)[1]);
  }
  
  bool
  mlab_vect::in_range(double vmin, double vmax) {
    for (size_type i = 0; i < sz; i++) {
      if (data[i] < vmin || data[i] > vmax) {
	return false;
      }
    }
    return true;
  }
  

  /* converts the gfi_array into a bit vector , shift all its values by
     'shift' and checking that they are a subset of 'subsetof' ( if the pointer
     is non-nul ) */
  dal::bit_vector
  mexarg_in::to_bit_vector(const dal::bit_vector *subsetof, int shift) {
    dal::bit_vector bv;
    mlab_vect v = to_int_vector(-1);
    for (size_type i = 0; i < v.size(); i++) {
      int idx = (int)v[i] + shift;
      if (idx < 0 || idx > 1000000000) {
	THROW_BADARG("Argument " << argnum << 
		     " should only contain values greater or equal to "
		     << -shift << " ([found " << v[i] << ")");
      } else if (subsetof && !subsetof->is_in(idx)) {
	THROW_BADARG("Argument " << argnum << 
		     " is not a valid set");
      }
      bv.add(idx);
    }
    return bv;
  }

  gfi_array *
  create_object_id(int nid, id_type *ids, id_type cid) {    
    gfi_array *arg = gfi_array_create_1(nid, GFI_OBJID);
    for (size_type i=0; i < size_type(nid); ++i) {
      gfi_objid_get_data(arg)[i].id = ids[i];
      gfi_objid_get_data(arg)[i].cid = cid;
    }
    return arg;
  }

  void
  mexarg_out::from_object_id(id_type id, id_type cid)
  { arg = create_object_id(id, cid); }

  void
  mexarg_out::from_object_id(std::vector<id_type> ids, id_type cid)
  { arg = create_object_id(ids.size(), &ids[0], cid); }

  void
  mexarg_out::from_integer(int i)
  {
    from_scalar((double)i);
  }

  void
  mexarg_out::from_scalar(double v)
  {
    arg = gfi_array_create_1(1,GFI_DOUBLE);
    *gfi_double_get_data(arg) = v;
  }

  void 
  mexarg_out::from_string(const char *s) {
    arg = gfi_array_from_string(s);
  }

  void
  mexarg_out::from_bit_vector(const dal::bit_vector& bv, int shift)
  {
    mlab_vect w = create_vector(1, bv.card());
    size_type j = 0;
    for (dal::bit_vector::const_iterator it = bv.begin(); it != bv.end(); ++it) {
      if (*it) {
	w(0,j++) = double(it.index() + shift);
      }
    }
    if (j != bv.card()) THROW_INTERNAL_ERROR;
  }


  void
  mexarg_out::from_sparse(const gf_sparse_by_row& M, double /*threshold*/) {
    arg = convert_to_gfi_sparse(M);
  }

  void
  mexarg_out::from_sparse(const gf_sparse_by_col& M, double /*threshold*/) {
    arg = convert_to_gfi_sparse(M);
  }

  void
  mexarg_out::from_tensor(const getfem::base_tensor& t) {
    std::vector<int> tab(t.order());
    std::copy(t.sizes().begin(), t.sizes().end(), tab.begin());
    arg = gfi_array_create(t.order(), &(tab.begin()[0]), GFI_DOUBLE);
    double *q = (double *)(gfi_double_get_data(arg));
    std::copy(t.begin(), t.end(), q);
  }

  mlab_vect
  mexarg_out::create_vector(unsigned dim)
  {
    arg = gfi_array_create_2(1,dim, GFI_DOUBLE);
    return mlab_vect(arg);
  }

  mlab_vect
  mexarg_out::create_vector(unsigned n,unsigned m,unsigned p)
  {
    int sz[3];
    sz[0] = n; sz[1] = m; sz[2] = p;
    arg = gfi_array_create(3,sz,GFI_DOUBLE);
    return mlab_vect(arg);
  }

  /* creates a 'matrix' (from the matlab point of view.. for getfem 
     it is still a vector, stored in fortran style) */
  mlab_vect
  mexarg_out::create_vector(unsigned dim, unsigned nbdof)
  {
    arg = gfi_array_create_2(dim, nbdof, GFI_DOUBLE);
    return mlab_vect(arg);
  }


  /* very tolerant case-insensitive string comparison: 
     spaces are matched with underscores.*/
  bool cmd_strmatch(const std::string& a, const char *s) {
    if (a.length() != strlen(s)) return false;
    for (size_type i=0; s[i]; ++i) {
      if ((a[i] == ' ' || a[i] == '_') && (s[i] == ' ' || s[i] == '_')) continue;
      if (toupper(a[i]) != toupper(s[i])) return false;
    }
    return true;
  }

  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, 
		 int min_argin, int max_argin) {
    if (cmd_strmatch(cmdname,s)) {
      if ((int)in.remaining() < min_argin) {
	THROW_BADARG("Not enough input arguments for command '"<< 
		     cmdname << "' (got " << in.narg() << 
		     ", expected at least " << min_argin + in.narg()- in.remaining() << ")");
      }
      if ((int)in.remaining() > max_argin && max_argin != -1) {
	THROW_BADARG("Too much input arguments for command '"<< 
		     cmdname << "' (got " << in.narg() << 
		     ", expected at most " << max_argin + in.narg()- in.remaining()<< ")");
      }
      return true;
    }
    return false;
  }

  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_out& out,
		 int min_argout, int max_argout) {
    if (cmd_strmatch(cmdname,s)) {
      if (out.narg() < min_argout) {
	THROW_BADARG("Not enough output arguments for command '"<< 
		     cmdname << "' (got " << out.remaining() << 
		     ", expected at least " << min_argout << ")");
      }
      if (out.narg() > max_argout && max_argout != -1) {
	THROW_BADARG("Too much output arguments for command '"<< 
		     cmdname << "' (got " << out.narg() << 
		     ", expected at most " << max_argout << ")");
      }
      return true;
    }
    return false;
  }

  bool check_cmd(const std::string& cmdname, const char *s, 
		 const mexargs_in& in, const mexargs_out& out,
		 int min_argin, int max_argin,
		 int min_argout, int max_argout) {
    return 
      check_cmd(cmdname, s,  in, min_argin, max_argin) &&
      check_cmd(cmdname, s, out, min_argout, max_argout);
  }

  mexargs_in::mexargs_in(int n, const gfi_array *p[], bool _use_cell) {
    nb_arg = n; use_cell = _use_cell;
    if (!use_cell) {
      in = p; idx.add(0, n);
    } else {
      assert(n == 1);
      assert(p[0]!=0);
      assert(gfi_array_get_class(p[0])==GFI_CELL);
      nb_arg = gfi_array_nb_of_elements(p[0]);
      in = new const gfi_array*[nb_arg];
      for (int i = 0; i < nb_arg; i++) { in[i] = gfi_cell_get_data(p[0])[i]; idx.add(i); }
    }
  }
  
  mexargs_in::~mexargs_in() {
    if (in && use_cell) delete[] in;
  }

  mexargs_out::mexargs_out(int n, gfi_array *p[])
  {
    idx = 0;
    out = p; nb_arg = n; 
  }

  void 
  mexargs_out::return_packed_obj_ids(const std::vector<id_type>& id, id_type class_id) {
    std::vector<id_type> uid(id);
    std::sort(uid.begin(), uid.end());
    /* remove duplicates */
    uid.erase(std::unique(uid.begin(), uid.end()), uid.end());
    /* remove id_type(-1) */
    std::vector<id_type>::iterator it =
      std::find(uid.begin(), uid.end(), id_type(-1));
    if (it != uid.end())
      uid.erase(it);
    pop().from_object_id(uid, class_id);
    if (remaining()) {
      std::map<id_type,id_type> m;
      for (size_type i=0; i < uid.size(); ++i) m[uid[i]]=i+1;
      mlab_vect v = pop().create_vector(1,id.size());
      for (size_type i=0; i < id.size(); ++i) 
	v[i] = (id[i] != id_type(-1)) ? double(m[id[i]]) : id_type(-1);
    }
  }

  void check_cv_fem(const getfem::mesh_fem& mf, size_type cv) {
    if (!mf.convex_index()[cv]) THROW_ERROR( "convex " << cv+1 << " has no FEM");
  }

  double get_NaN() {
#ifdef NAN
    return NAN;
#else
    static double NaN = 0.;
    if (NaN == 0.) {
      double a=0.,b=0.;
      struct sigaction osa,nsa;
      sigaction(SIGFPE, 0, &nsa);
      nsa.sa_handler = SIG_IGN;
      sigaction(SIGFPE, &nsa, &osa);
      NaN = a / b;
      sigaction(SIGFPE,&osa, 0);      
      //assert(NaN != NaN);//great ... assertion fails on dec/alpha and origin2k ..
    }
    return NaN;
#endif
  }
} /* namespace matlabint */
