#include <map>
#include <matlabint_misc.h>
#include <matlabint_mesh.h>
#include <matlabint_mesh_slice.h>
#include <getfem_mesh_slice.h>

using namespace matlabint;

/*MLABCOM

  FUNCTION [...] = gf_slice_get(slice SL, [operation [, args]])

  * m = gf_slice_get(sl, 'linked mesh')
  Returns the mesh on which the slice was taken.

  * I = gf_slice_get(sl, 'dim')
  Returns the dimension of the slice (2 for a 2D mesh, etc..)

  * CVLST = gf_slice_get(sl, 'cvs')
  Returns the list of convexes contained in the slice.


  * N = gf_slice_get(sl, 'nbpts')
  * P = gf_slice_get(sl, 'pts')

  Returns the number of points, and the list of point coordinates in
  the slice.

  * NS = gf_slice_get(sl, 'nbsplxs' [,int DIM])

  Returns the number of simplexes in the slice. Since the slice may
  contain points (simplexes of dim 0), segments (simplexes of
  dimension 1), triangles etc, the result is a vector of size
  gf_slice_get(sl, 'dim')+1 , except if the optional argument DIM is
  used.

  * [S,CV2SPLX] = gf_slice_get(sl, 'splxs', int DIM)

  Returns the list of simplexes of dimension DIM. On output, S has
  DIM+1 rows, each column contains the point numbers of a simplex.
  The vector CV2SPLX can be used to find the list of simplexes for any
  convex stored in the slice. For example S(:,CV2SPLX(5)-CV2SPLX(4))
  give the list of simplexes for the fourth convex.

  * E = gf_slice_get(sl, 'edges')

  Returns the edges of the linked mesh contained in the slice (each
  column of E contains the vertice coordinates of an edge).

  * [mat P, ivec E1, ivec E2] = gf_slice_get(sl, 'edges')

  Returns also the edges of the linked mesh, but in a different style:
  P contains the list of all edge vertices, E1 contains the indices of
  each mesh edge in P, and E2 contains the indices of each "edges"
  which is on the border of the slice.

  * ms=gf_mesh_get(M, 'memsize')

  Returns the amount of memory (in bytes) used by the slice object.

  $Id: gf_slice_get.C,v 1.5 2003/07/23 18:31:14 pommier Exp $
MLABCOM*/


void gf_slice_get(matlabint::mexargs_in& in, matlabint::mexargs_out& out)
{
  if (in.narg() < 2) {
    THROW_BADARG( "Wrong number of input arguments");
  }
  matlabint_mesh_slice *mi_sl = in.pop().to_matlabint_mesh_slice();
  const getfem::mesh_slice *sl = &mi_sl->mesh_slice();
  std::string cmd                  = in.pop().to_string();
  if (check_cmd(cmd, "dim", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(sl->dim());
  } else if (check_cmd(cmd, "cvs", in, out, 0, 0, 0, 1)) {
    mlab_vect w = out.pop().create_vector(1, sl->nb_convex());
    for (size_type i=0; i < sl->nb_convex(); ++i) w[i] = sl->convex_num(i);
  } else if (check_cmd(cmd, "nbpts", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(sl->nb_points());
  } else if (check_cmd(cmd, "nbsplxs", in, out, 0, 1, 0, 1)) {
    std::vector<size_type> v; sl->nb_simplexes(v);
    if (in.remaining()) {
      size_type i= in.pop().to_integer(0,100);
      out.pop().from_integer(i < v.size() ? v[i] : 0);
    } else out.pop().from_vector(v);
  } else if (check_cmd(cmd, "pts", in, out, 0, 0, 0, 1)) {
    mlab_vect w = out.pop().create_vector(sl->dim(), sl->nb_points());
    for (size_type ic=0, cnt=0; ic < sl->nb_convex(); ++ic) {
      for (getfem::mesh_slice::cs_nodes_ct::const_iterator it=sl->nodes(ic).begin();
           it != sl->nodes(ic).end(); ++it) {
        for (size_type k=0; k < sl->dim(); ++k)
          w[cnt++] = (*it).pt[k];
      }
    }
  } else if (check_cmd(cmd, "splxs", in, out, 1, 1, 0, 1)) {
    size_type sdim = in.pop().to_integer(0,sl->dim());
    mlab_vect w = out.pop().create_vector(sdim+1, sl->nb_simplexes(sdim));
    size_type Scnt = 0;
    mlab_vect cv2splx;
    if (out.remaining()) {
      cv2splx = out.pop().create_vector(1, sl->nb_convex()+1); Scnt = 1;
    }
    for (size_type ic=0, cnt=0, pcnt=0; ic < sl->nb_convex(); ++ic) {
      size_type scnt = 0;
      for (getfem::mesh_slice::cs_simplexes_ct::const_iterator it=sl->simplexes(ic).begin();
           it != sl->simplexes(ic).end(); ++it) {
        if ((*it).dim() == sdim) {
          for (size_type k=0; k < sdim+1; ++k)
            w[cnt++] = (*it).inodes[k] + 1 + pcnt;
	  scnt++; 
	}
      }      
      pcnt += sl->nodes(ic).size();
      if (Scnt) {	
	cv2splx[ic] = Scnt; Scnt+=scnt;
      }
    }
    if (Scnt) cv2splx[sl->nb_convex()] = Scnt;
  } else if (check_cmd(cmd, "edges", in, out, 0, 0, 0, 3)) {
    getfem::getfem_mesh m; 
    dal::bit_vector slice_edges;
    sl->edges_mesh(m,slice_edges);
    
    if (out.narg() <= 1) {
      /* one arg : return a raw list of edges */
      mlab_vect w = out.pop().create_vector(m.dim()*2, m.nb_convex());
      mlab_vect::iterator out=w.begin();
      for (dal::bit_vector::const_iterator it=m.convex_index().begin(); 
	   it != m.convex_index().end(); ++it) {
	if (!(*it)) continue;
	size_type iA = m.ind_points_of_convex(it.index())[0],iB = m.ind_points_of_convex(it.index())[1];
	assert(iA != iB);
	out = std::copy(m.points()[iA].begin(), m.points()[iA].end(), out);
	out = std::copy(m.points()[iB].begin(), m.points()[iB].end(), out);
      }
    } else {
      /* return a point list, a connectivity array, and optionnaly a list of edges with are part of the slice */
      double nan = get_NaN();
      dal::bit_vector bv = m.points().index();
      mlab_vect P = out.pop().create_vector(m.dim(), bv.last_true()+1);
      mlab_vect T1 = out.pop().create_vector(2, m.nb_convex() - slice_edges.card());
      mlab_vect T2 = out.pop().create_vector(2, slice_edges.card());
      for (size_type j = 0; j < bv.last_true()+1; j++) {
	for (size_type i = 0; i < m.dim(); i++) {
	  P(i,j) = (bv.is_in(j)) ? (m.points()[j])[i] : nan;
	}
      }
      mlab_vect::iterator itt1=T1.begin(), itt2=T2.begin();
      for (dal::bit_vector::const_iterator it=m.convex_index().begin(); 
	   it != m.convex_index().end(); ++it) {
	if (!(*it)) continue;
	if (!slice_edges[it.index()]) {  
	  dal::copy_n(m.ind_points_of_convex(it.index()).begin(), 2, itt1); 
	  itt1[0]++; itt1[1]++; itt1 += 2;
	} else {
	  dal::copy_n(m.ind_points_of_convex(it.index()).begin(), 2, itt2); 
	  itt2[0]++; itt2[1]++; itt2 += 2;
	}
      }
    }
  } else if (check_cmd(cmd, "linked mesh", in, out, 0, 0, 0, 1)) {
    out.pop().from_object_id(mi_sl->linked_mesh_id(), MESH_CLASS_ID);
  } else if (check_cmd(cmd, "memsize", in, out, 0, 0, 0, 1)) {
    out.pop().from_integer(sl->memsize());
  } else bad_cmd(cmd);
}

